///////////////////////////////////////////////////////////////////////////////
//
//  File:    clockd.cpp
//
//  Class:   ClockDOS
//
//  Author:  Kevin Brisley
//
//
//      The ClockDOS class is a class encapsulating timing for the DOS
//      platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <time.h>

//  Application Headers.
#include "clockd.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ClockDOS
//
//  Description:
//
//      This is the main constructor for a DOS clock object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ClockDOS::ClockDOS(
    const KString&  iName
)
:
    Clock    ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ClockDOS
//
//  Description:
//
//      This is the destructor for a DOS clock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ClockDOS::~ClockDOS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the clock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ClockDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ClockDOS" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getTime
//
//  Description:
//
//      This member returns the current time in microseconds.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The current time.
//
///////////////////////////////////////////////////////////////////////////////
DWord
ClockDOS::getTime(
) const
{
    return( uclock( ) * 1000000 / UCLOCKS_PER_SEC );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: frameWait
//
//  Description:
//
//      This member is used to wait until the next frame is ready to be
//      drawn (as specified by specified values).
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
ClockDOS::frameWait(
)
{
    //  The difference between the current time and the last time.
    DWord dwDiff;

    //  The current time.
    DWord dwThisTime;


    //  Get the current time and calculate the difference between this and
    //  the last time.
    dwThisTime = uclock( ) * 1000000 / UCLOCKS_PER_SEC;
    dwDiff = dwThisTime - m_dwLastTime;

    //  If throttling is on then wait for the next frame, otherwise just note
    //  the current clock.
    if( m_bThrottle )
    {
        //  Loop until the interval has expired.
        while( dwDiff < m_dwInterval )
        {
            dwThisTime = uclock( ) * 1000000 / UCLOCKS_PER_SEC;
            dwDiff = dwThisTime - m_dwLastTime;
        }
    }

    //  Assign the measured FPS.  We only calculate based on a single 
    //  frame (the one just completed).  The .5 is used to ensure proper 
    //  rounding.
    m_dwMeasuredFPS = ( DWord )( ( double )1000000 / ( double )dwDiff + .5 );

    //  Record this time for next time.
    m_dwLastTime = dwThisTime;
}
