///////////////////////////////////////////////////////////////////////////////
//
//  File:    joyd.cpp
//
//  Class:   JoystickDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The JoystickDOS class encapsulates joystick access for the DOS
//      platform using the allegro joystick routines.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "joyd.h"

//  System Headers.
#include <allegro.h>


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The following indicates whether or not the joystick has been initialized.
Byte JoystickDOS::sm_bJoystickInitialized = FALSE;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a DOS Joystick object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
JoystickDOS*
JoystickDOS::s_build(
    const KString& iName
)
{
    //  Create the new object.
    JoystickDOS* pThis = new JoystickDOS( iName );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new joystick.
    return( pThis );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: JoystickDOS
//
//  Description:
//
//      This is the main constructor for the DOS Joystick object.  This is
//      a protected member.  Clients wishing to create a joystick object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
JoystickDOS::JoystickDOS(
    const KString&  iName
)
:
    Joystick  ( iName )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the joystick object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
JoystickDOS::init(
)
{
    //  Call the base class.
    Joystick::init( );

    //  If the joystick is disabled then just return.
    if( !m_bEnabled )
    {
        return;
    }

    //  Initialize the joystick if we haven't previously done so.
    if( !sm_bJoystickInitialized )
    {
        joy_type = JOY_TYPE_4BUTTON;
        if( initialise_joystick( ) )
        {
            m_bEnabled = FALSE;
        }
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~JoystickDOS
//
//  Description:
//
//      This is the destructor for a DOS joystick object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
JoystickDOS::~JoystickDOS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the joystick object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
JoystickDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "JoystickDOS" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member should be called periodically to allow the
//      joystick object to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if a joystick state has changed.
//      FALSE if no joystick state has changed.
//
///////////////////////////////////////////////////////////////////////////////
Byte
JoystickDOS::update(
)
{
    //  The status of the joystick controls.
    Byte abStatus[ JOY_COUNT ];


    //  If the joystick is disabled then simply return.
    if( !m_bEnabled )
    {
        return( FALSE );
    }

    //  Poll the joystick.
    poll_joystick( );


    //  Set the status.
    abStatus[ JOY1_LEFT ]  = joy_left;
    abStatus[ JOY1_RIGHT ] = joy_right;
    abStatus[ JOY1_UP ]    = joy_up;
    abStatus[ JOY1_DOWN ]  = joy_down;
    abStatus[ JOY1_B1 ]    = joy_b1;
    abStatus[ JOY1_B2 ]    = joy_b2; 
    abStatus[ JOY1_B3 ]    = joy_b3;
    abStatus[ JOY1_B4 ]    = joy_b4;
    abStatus[ JOY2_LEFT ]  = joy2_left;
    abStatus[ JOY2_RIGHT ] = joy2_right;
    abStatus[ JOY2_UP ]    = joy2_up;
    abStatus[ JOY2_DOWN ]  = joy2_down;
    abStatus[ JOY2_B1 ]    = joy2_b1;
    abStatus[ JOY2_B2 ]    = joy2_b2;
    abStatus[ JOY2_B3 ]    = FALSE;
    abStatus[ JOY2_B4 ]    = FALSE;
    
    //  Now compare the new states with the old states.
    for( DWord dwI = 1 ; dwI < JOY_COUNT ; dwI += 1 )
    {
        //  If the state has changed then set the new state.
        if( abStatus[ dwI ] != m_switchList[ dwI ]->onPhysically( ) )
        {
            m_switchList[ dwI ]->physicalOn( abStatus[ dwI ] );

            //  If the switch just turned off, then mark this switch as the 
            //  last one that was on.
            if( !abStatus[ dwI ] )
            {
                m_dwLastSwitchOn = dwI;
            }
        }
    }
        
    return( FALSE );
}
