///////////////////////////////////////////////////////////////////////////////
//
//  File:    nettcpd.cpp
//
//  Class:   NetworkTCPDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The NetworkTCPDOS class implements network code between two or more 
//      Replay processes.  The TCP/IP protocol is used for communication.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <unistd.h>

//  Application Headers.
#include "nettcpd.h"
#include "config.h"
#include "wsocket.h"
#include "replay.h"
#include "clock.h"



///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
int NetworkTCPDOS::sm_nDefaultPort = 6444;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetworkTCPDOS
//
//  Description:
//
//      This is the main constructor for a network TCP/IP object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkTCPDOS::NetworkTCPDOS(
    const KString&  iName
)
:
    Network               ( iName ),
    m_pSocket             ( NULL ),
    m_ppConnectSocketList ( NULL ),
    m_nPort               ( sm_nDefaultPort ),
    m_hostName            ( )
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    

    //  If the network is disabled then there's no point continuing.
    if( !m_bEnabled )
    {
        return;
    }

    //  Allocate space for the list of connected sockets.
    m_ppConnectSocketList = new WinSocket*[ getMaxConnections( ) ];
    for( DWord dwI = 0 ; dwI < getMaxConnections( ) ; dwI += 1 )
    {
        m_ppConnectSocketList[ dwI ] = NULL;
    }

    //  Now if there was a port number specified then use it, otherwise
    //  use the default port..
    if( !config.getParam( "-port", &m_nPort ) )
    {
        m_nPort = sm_nDefaultPort;
    }

    //  If a host name was specified then use it, otherwise use the local 
    //  host.
    if( !config.getParam( "-host", &m_hostName ) )
    {
        m_hostName = "127.0.0.1";
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetworkTCPDOS
//
//  Description:
//
//      This is the destructor for a network TCP/IP object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkTCPDOS::~NetworkTCPDOS(
)
{
    //  Make sure the network is closed.
    closeNetwork( );

    //  Get rid of the connection list.
    delete [] m_ppConnectSocketList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the network TCP/IP object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetworkTCPDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "NetworkTCPDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: openNetwork
//
//  Description:
//
//      This member is called to open the TCP/IP network connection.
//
//  Parameters:
//
//      eType (input)
//          Indicates whether or not the connection should be opened as
//          client or server.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkTCPDOS::openNetwork(
    ConnectType eType
)
{
    ASSERT( m_bEnabled );

    //  Make sure either client or server was specified.
    CONFIRM( eType != UNKNOWN, "Network must be opened as client or server" );

    //  Call the base class.
    Network::openNetwork( eType );


    //  If we're a server then we must create and bind a socket to the 
    //  server address and listen for connections.
    //  If we're a client then we create a socket and initiate the connect.
    if( eType == SERVER )
    {
        ASSERT( m_pSocket == NULL );

        //  Create the socket.
        m_pSocket = new WinSocket( "ServerSocket" );

        //  Bind to the server address.
        m_pSocket->bind( m_nPort );

        //  Listen on the socket.
        m_pSocket->listen( 5 );
    }
    else
    {
        //  Create the socket.
        m_pSocket = new WinSocket( "ClientSocket" );

        //  Attempt to connect.
        m_pSocket->connect( m_hostName.data( ), m_nPort );
    }


    //  The network was successfully opened.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: acceptClient
//
//  Description:
//
//      This member is called when the server is to accept client connections.
//      It performs a non-blocking accept.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE if a client connected, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkTCPDOS::acceptClient(
)
{
    //  A pointer to a new socket.
    WinSocket* pNewSocket;


    ASSERT( m_pSocket != NULL );

    //  Client connections can only be accepted if we are serving as the 
    //  server.
    CONFIRM( m_eConnectType == SERVER, "Must be server to accept clients" );

    //  We can't accept more connections than allowed.
    CONFIRM( 
        getNumConnections( ) < getMaxConnections( ), 
        "Cannot accept more than %d connections.",   
        getMaxConnections( )
    );

    //  Create a new socket.
    pNewSocket = new WinSocket( "ClientSocket", *m_pSocket );
   
    //  If the socket isn't valid then we have no connection.
    if( !pNewSocket->isValid( ) )
    {
        delete pNewSocket;
        return( FALSE );
    }

    //  Add the new client to the list of clients.
    m_ppConnectSocketList[ m_dwNumConnections ] = pNewSocket;
    m_dwNumConnections += 1; 

    //  A client has connected.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: connectToServer
//
//  Description:
//
//      This member is called when the client is to connect to the server.
//      It performs a non-blocking connect.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE if the client connected, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkTCPDOS::connectToServer(
)
{
    //  Indicates whether or not we have connected.
    Byte bConnected;


    ASSERT( m_pSocket != NULL );

    //  We can only connect to a server if we're a client.
    CONFIRM( m_eConnectType == CLIENT, "Must be client to connect to server." );

    //  We can only connect to a server if we haven't already connected.
    CONFIRM( m_dwNumConnections == 0, "Can't connect - already connected." );

    //  The connect isn't returned immediately, so we do a few loops and
    //  attempt to connect.
    bConnected = FALSE;
    for( DWord dwI = 0 ; dwI < 75 ; dwI += 1 )
    {
        if( m_pSocket->select( WinSocket::SELECT_CONNECT ) )
        {
            bConnected = TRUE;
            break;
        }
        Replay::s_instance( ).getClock( )->frameWait( );
    } 
    if( !bConnected )
    {
        return( FALSE );
    }

    //  OK, we've connected so place the socket at the head of the connection
    //  list.
    m_ppConnectSocketList[ 0 ] = m_pSocket;
    m_dwNumConnections = 1;

    //  A connection has been made.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeNetwork
//
//  Description:
//
//      This member is called to close the network down completely.  This
//      involves closing all client sockets (if we're a server) as well
//      as the original socket.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkTCPDOS::closeNetwork(
)
{
    //  If the network isn't opened then just return.
    if( !isOpen( ) )
    {
        return;
    }

    //  Close each connection. 
    while( m_dwNumConnections > 0 )
    {
        closeConnection( 0 );
    }

    //  If we are a server then we have to delete our original socket.
    if( m_eConnectType == SERVER )
    {
        //  Delete the original socket.
        delete m_pSocket;
    }

    //  Clean out the original socket.
    m_pSocket = NULL;

    //  Call the base class.
    Network::closeNetwork( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeConnection
//
//  Description:
//
//      This member is called to close a specific connection within the
//      network.  It will shutdown the socket and remove the connection
//      from the list.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to close down.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkTCPDOS::closeConnection(
    DWord dwConnection
)
{
    //  Make sure that the specified connection exists.
    CONFIRM( 
        dwConnection < m_dwNumConnections,
        "Connection %d cannot be closed as it does not exist.",
        dwConnection
    );

    //  Delete the socket.
    delete m_ppConnectSocketList[ dwConnection ];
    m_ppConnectSocketList[ dwConnection ] = NULL;

    //  We now have one less entry in the list.
    m_dwNumConnections -= 1;

    //  If this wasn't the last connection then move the connection at the
    //  end of the list into this position.
    if( dwConnection < m_dwNumConnections )
    {
        m_ppConnectSocketList[ dwConnection ] = 
            m_ppConnectSocketList[ m_dwNumConnections ];
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readMsg
//
//  Description:
//
//      This member is called to read a message from the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to read from.
//
//      pMsg (output)
//          The buffer to hold the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkTCPDOS::readMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );

    //  Has the connection closed?
    Byte bClosed = m_ppConnectSocketList[ dwConnection ]->select( 
        WinSocket::SELECT_CLOSE 
    );
    if( bClosed )
    {
        pMsg->eType = MSG_CLOSE;
        if( m_eConnectType == SERVER )
        {
            closeConnection( dwConnection );
        }
        else
        {
            closeNetwork( );
        }
        return;
    }

    //  Read data from the socket.
    int nRead = m_ppConnectSocketList[ dwConnection ]->recv( 
        ( void* )pMsg, sizeof( Msg )
    );
    if( nRead == 0 )
    {
        //  Indicate that there is no message.
        pMsg->eType = MSG_NONE;
    }
    else
    {
        ASSERT( nRead == sizeof( Msg ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeMsg
//
//  Description:
//
//      This member is called to write a message to the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to write to.
//
//      pMsg (output)
//          The buffer that contains the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkTCPDOS::writeMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );

    //  Write data to the socket.
    int nWrite = m_ppConnectSocketList[ dwConnection ]->send( 
        ( void* )pMsg, sizeof( Msg )
    );
    CONFIRM( nWrite == sizeof( Msg ), "Error writing to socket." );
}
