///////////////////////////////////////////////////////////////////////////////
//
//  File:    paged.cpp
//
//  Class:   PageDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window of the Replay application on
//      the DOS platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "paged.h"
#include "bitmap.h"
#include "config.h"

//  Allegro Headers.
#include <allegro.h>


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The list of possible graphics modes.
KPtrList<GraphicsMode> PageDOS::sm_graphicsModeList( 16 );

//  The following indicates whether the screen mode list has been 
//  initialized yet.
Byte PageDOS::sm_bGraphicsModeListInitialized = FALSE;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: PageDOS
//
//  Description:
//
//      This is the main constructor for the DOS Page object.  
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//      bTransposed (input)
//          Indicates that the page should be created  with 
//          the dimensions swapped.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
PageDOS::PageDOS(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bTransposed
)
:
    Page            ( iName, dwWidth, dwHeight, bTransposed ),
    m_pGraphicsMode ( NULL )
{
    //  If the screen mode list has not yet been initialized then do it now.
    if( !sm_bGraphicsModeListInitialized )
    {
        s_initializeGraphicsModes( );
        sm_bGraphicsModeListInitialized = TRUE;
    }

    //  Find a matching screen mode.
    findGraphicsMode( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~PageDOS
//
//  Description:
//
//      This is the main destructor for the DOS Page object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
PageDOS::~PageDOS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the DOS Page object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
PageDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "PageDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: transpose
//
//  Description:
//
//      This is called to transpose the page.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
PageDOS::transpose(
)
{
    //  Call the base class.
    Page::transpose( );

    //  If the page is non-square then we need to find a new mode that
    //  fits the new dimensions.
    if( m_dwWidth != m_dwHeight )
    {
        findGraphicsMode( );
        setGraphicsMode( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: findGraphicsMode
//
//  Description:
//
//      This is called to find a graphics mode matching the width and height
//      of the page.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
PageDOS::findGraphicsMode(
)
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    //  Indicates whether we are looking for scan lines and a tweaked mode.
    Byte bScanLines = FALSE;
    Byte bTweaked   = FALSE;

    //  If the requested page size is less than 640x480 then we allow
    //  searching for tweaked modes and scan lines.
    if( ( m_dwWidth < 640 ) && ( m_dwHeight < 480 ) )
    {
        bScanLines = config.getParam( "-lines" );
        bTweaked   = !config.getParam( "-vesa" );
    }
    
    //  Find a matching screen mode.
    for( DWord dwI = 0 ; dwI < sm_graphicsModeList.entries( ) ; dwI += 1 )
    {
        if( 
            sm_graphicsModeList[ dwI ]->match( 
                m_dwWidth, m_dwHeight, bScanLines, bTweaked
            ) 
        )
        {
            m_pGraphicsMode = sm_graphicsModeList[ dwI ];
            break;
        }
    }
    CONFIRM( 
        m_pGraphicsMode != NULL, 
        "Could not find a screen mode for %dx%d,%s,%s.", 
        m_dwWidth,
        m_dwHeight,
        bScanLines ? "scan lines" : "no scan lines",
        bTweaked   ? "no vesa" : "vesa"
    );

    //  The screen mode is a good fit if it matches the page width and
    //  height exactly.
    if( 
        ( m_pGraphicsMode->getWidth( ) == m_dwWidth ) &&
        ( m_pGraphicsMode->getHeight( ) == m_dwHeight )
    )
    {
        m_bGoodFit = TRUE;
    }
    else
    {
        m_bGoodFit = FALSE;
    }

    //  Calculate the border width/height.  This is the amount needed to
    //  center the page within the screen.
    m_dwBorderWidth  = ( m_pGraphicsMode->getWidth( )  - m_dwWidth  ) / 2;
    m_dwBorderHeight = ( m_pGraphicsMode->getHeight( ) - m_dwHeight ) / 2;
}
 

///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This is called to transpose the page.
//
//  Parameters:
//
//      bEasy (input)
//          Indicates that this is an "easy" draw.  i.e. the full bitmap
//          is being drawn to the origin of the page.
//
//      pBitmap (input)
//          The bitmap to draw.
//
//      nX (input)
//          The X position to draw the bitmap at.
//
//      nY (input)
//          The Y position to draw the bitmap at.
//
//      dwWidth (input)
//          The width to draw.
//
//      dwHeight (input)
//          The height to draw.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
PageDOS::draw(
    Byte    bEasy,
    Bitmap* pBitmap,
    int32   nX,
    int32   nY,
    DWord   dwWidth,
    DWord   dwHeight
)
{
    ASSERT( m_pGraphicsMode != NULL );

    //  We always check for a shortcut.  This is when the complete bitmap
    //  is to be drawn to the origin of the page and the screen mode
    //  is a good fit for the page.
    if( bEasy && m_bGoodFit )
    {
        //  If the mode is linear then we can copy the the complete bitmap 
        //  to the screen in double words for the quickest speed possible,
        //  otherwise, we copy each line.
        if( m_pGraphicsMode->isLinear( ) )
        {
            _dosmemputl( 
                pBitmap->getBuffer( ), 
                dwWidth * dwHeight / 4, 
                m_pGraphicsMode->getAddress( )
            );
        }
        else
        {
            //  The width of a line in double words.
            DWord dwWidthInDWords = dwWidth / 4;

            //  Cast the buffer to a double word pointer.
            DWord* pdwBuffer = ( DWord* )pBitmap->getBuffer( );

            //  Now copy each line.
            for( DWord dwLine = 0 ; dwLine < dwHeight ; dwLine += 1 )
            {
                _dosmemputl(
                    pdwBuffer,
                    dwWidthInDWords,
                    m_pGraphicsMode->getAddress( dwLine )
                );
                pdwBuffer += dwWidthInDWords;
            }
        }
    }
    else
    {
        //  Clip the bitmap to the page.
        if( nX + dwWidth > m_dwWidth )
        {
            dwWidth = m_dwWidth - nX;
        }
        if( nY + dwHeight > m_dwHeight )
        {
            dwHeight = m_dwHeight - nY;
        }

        //  We will copy in DWords so we set a convenient value of the 
        //  draw width divided by 4.
        DWord dwWidthInDWords = dwWidth / 4;

        //  The width of the bitmap in DWords.
        DWord dwBitmapWidthInDWords;
        if( m_bTransposed )
        {
            dwBitmapWidthInDWords = pBitmap->getHeight( ) / 4;
        }
        else
        {
            dwBitmapWidthInDWords = pBitmap->getWidth( ) / 4;
        }

        //  The width should always be divisible by 4.
        ASSERT( ( dwWidthInDWords * 4 ) == dwWidth );

        //  Cast the bitmap buffer to DWord.
        DWord* pdwBuffer = ( DWord* )( pBitmap->getBuffer( ) );

        //  Loop through each draw row.
        for( DWord dwI = 0 ; dwI < dwHeight ; dwI += 1 )
        {
            //  Write the line.
            _dosmemputl(
                pdwBuffer, 
                dwWidthInDWords, 
                m_pGraphicsMode->getAddress( 
                    m_dwBorderHeight + nY + dwI, m_dwBorderWidth + nX
                )
            );
            pdwBuffer += dwBitmapWidthInDWords;
        }
    }
}

 

///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_initializeGraphicsModes
//
//  Description:
//
//      This is called to set up the possible graphics modes.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
PageDOS::s_initializeGraphicsModes(
)
{
    //  The following array contains all of the graphics modes.  The values
    //  are stored as follows:
    //      - width
    //      - height
    //      - scanlines (T/F)
    //      - tweaked mode (T/F)
    //      - register list (Port/Index/Value), terminated by -1.
    //
    //  Important:  Graphics modes should be added in the order of increasing
    //              resolution so that the best match can be found by
    //              searching from the beginning.
    //
    static Word awModeDefs[ ] = 
    {
        //  224x288, no scan lines.
        224, 288,
        FALSE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x37,
        0x3d4, 0x02, 0x38,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x4a,
        0x3d4, 0x05, 0x9a,  0x3d4, 0x06, 0x55,  0x3d4, 0x07, 0xf0,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x61,  0x3d4, 0x10, 0x40,
        0x3d4, 0x11, 0xac,  0x3d4, 0x12, 0x3f,  0x3d4, 0x13, 0x1c,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0x40,  0x3d4, 0x16, 0x4a,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //  224x288, scan lines.
        224, 288,
        TRUE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x37,
        0x3d4, 0x02, 0x38,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x4a,
        0x3d4, 0x05, 0x9a,  0x3d4, 0x06, 0x43,  0x3d4, 0x07, 0x1f,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x60,  0x3d4, 0x10, 0x2a,
        0x3d4, 0x11, 0xac,  0x3d4, 0x12, 0x1f,  0x3d4, 0x13, 0x1c,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0x27,  0x3d4, 0x16, 0x3a,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00, 
        ( Word )-1,

        //  256x256, no scan lines.
        256, 256,
        FALSE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x3f,
        0x3d4, 0x02, 0x40,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x4a,
        0x3d4, 0x05, 0x9a,  0x3d4, 0x06, 0x23,  0x3d4, 0x07, 0xb2,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x61,  0x3d4, 0x10, 0x0a,
        0x3d4, 0x11, 0xac,  0x3d4, 0x12, 0xff,  0x3d4, 0x13, 0x20,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0x07,  0x3d4, 0x16, 0x1a,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //   256x256, scan lines.
        256, 256,
        TRUE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x3f,
        0x3d4, 0x02, 0x40,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x4a,
        0x3d4, 0x05, 0x9a,  0x3d4, 0x06, 0x23,  0x3d4, 0x07, 0x1d,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x60,  0x3d4, 0x10, 0x0a,
        0x3d4, 0x11, 0xac,  0x3d4, 0x12, 0xff,  0x3d4, 0x13, 0x20,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0x07,  0x3d4, 0x16, 0x1a,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //  288x224, no scan lines.
        288, 224,
        FALSE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x47,
        0x3d4, 0x02, 0x50,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x50,
        0x3d4, 0x05, 0x80,  0x3d4, 0x06, 0x0b,  0x3d4, 0x07, 0x3e,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x41,  0x3d4, 0x10, 0xda,
        0x3d4, 0x11, 0x9c,  0x3d4, 0x12, 0xbf,  0x3d4, 0x13, 0x24,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0xc7,  0x3d4, 0x16, 0x04,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //  288x224, scan lines.
        288, 224,
        TRUE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x47,
        0x3d4, 0x02, 0x47,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x50,
        0x3d4, 0x05, 0x9a,  0x3d4, 0x06, 0x0b,  0x3d4, 0x07, 0x19,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x40,  0x3d4, 0x10, 0xf5,
        0x3d4, 0x11, 0xac,  0x3d4, 0x12, 0xdf,  0x3d4, 0x13, 0x24,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0xc7,  0x3d4, 0x16, 0x04,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //  320x204, scan lines.
        320, 204,
        TRUE,
        TRUE,
        0x3c2, 0x00, 0xe3,  0x3d4, 0x00, 0x5f,  0x3d4, 0x01, 0x4f,
        0x3d4, 0x02, 0x50,  0x3d4, 0x03, 0x82,  0x3d4, 0x04, 0x54,
        0x3d4, 0x05, 0x80,  0x3d4, 0x06, 0xbf,  0x3d4, 0x07, 0x1f,
        0x3d4, 0x08, 0x00,  0x3d4, 0x09, 0x41,  0x3d4, 0x10, 0x9c,
        0x3d4, 0x11, 0x8e,  0x3d4, 0x12, 0x97,  0x3d4, 0x13, 0x28,
        0x3d4, 0x14, 0x40,  0x3d4, 0x15, 0x96,  0x3d4, 0x16, 0xb9,
        0x3d4, 0x17, 0xa3,  0x3c4, 0x01, 0x01,  0x3c4, 0x04, 0x0e,
        0x3ce, 0x05, 0x40,  0x3ce, 0x06, 0x05,  0x3c0, 0x10, 0x41,
        0x3c0, 0x13, 0x00,
        ( Word )-1,

        //  640x480, no scan lines.
        640, 480,
        FALSE,
        FALSE,
        ( Word )-1,
        
        //  800x600, no scan lines.
        800, 600,
        FALSE,
        FALSE,
        ( Word )-1,
        
        //  1024x768, no scan lines.
        1024, 768,
        FALSE,
        FALSE,
        ( Word )-1,
        
        //  End of modes.
        ( Word )-1
    };



    //  A convenient pointer to the current graphics mode of interest.
    GraphicsMode* pMode;


    //  Loop through each scan mode.
    DWord dwI = 0;
    while( awModeDefs[ dwI ] != ( Word )-1 )
    {
        //  Build the name of the scan mode.
        char strModeName[ 32 ];
        sprintf( 
            strModeName,
            "%dx%d-%slines", 
            awModeDefs[ dwI ], 
            awModeDefs[ dwI + 1 ],
            awModeDefs[ dwI + 2 ] ? "" : "no"
        );

        //  Add the mode.
        pMode = sm_graphicsModeList.add( 
            new GraphicsMode( 
                strModeName,
                awModeDefs[ dwI + 0 ],  //  Width.
                awModeDefs[ dwI + 1 ],  //  Height.
                awModeDefs[ dwI + 2 ],  //  ScanLines.
                awModeDefs[ dwI + 3 ]   //  Tweaked.
            )
        );
        dwI += 4;

        //  Add the registers for the mode. 
        while( awModeDefs[ dwI ] != ( Word )-1 )
        {
            pMode->addRegister( 
                awModeDefs[ dwI + 0 ],  //  Port.
                awModeDefs[ dwI + 1 ],  //  Index.
                awModeDefs[ dwI + 2 ]   //  Value.
            );
            dwI += 3;
        }
        dwI += 1;
    }
}
