///////////////////////////////////////////////////////////////////////////////
//
//  File:    repfactd.cpp
//
//  Class:   ReplayFactoryDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is the concrete factory for creating DOS based
//      objects used in Replay.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repfactd.h"
#include "config.h"
#include "canvasd.h"
#include "keybd.h"
#include "joyd.h"
#include "moused.h"
#include "soundd.h"
#include "diskd.h"
#include "netipxd.h"
#include "nettcpd.h"
#include "clockd.h"
#include "gamfact.h"
#ifdef RELEASE
#include "6502nba.h"
#include "6809pla.h"
#include "68000nca.h"
#include "z80nba.h"
#include "z80mdc.h"
#else
#include "6502mfc.h"
#include "6809kbc.h"
#include "68000bsc.h"
#include "z80mdc.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  File Static Variables.
///////////////////////////////////////////////////////////////////////////////

//  The singleton instance.
ReplayFactoryDOS ReplayFactoryDOS::sm_replayFactoryDOS( "ReplayFactory" );



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ReplayFactoryDOS
//
//  Description:
//
//      This is the main constructor for the DOS factory
//      object.  It is protected because it is a singleton and therefore
//      cannot be instantiated by anyone but itself or a derived class.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryDOS::ReplayFactoryDOS(
    const KString& iName
)
:
    ReplayFactory  ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ReplayDOS
//
//  Description:
//
//      This is the main destructor for the DOS Replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryDOS::~ReplayFactoryDOS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ReplayFactoryDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ReplayFactoryDOS" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCanvas
//
//  Description:
//
//      This creates the default canvas object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the canvas.
//
///////////////////////////////////////////////////////////////////////////////
Canvas*
ReplayFactoryDOS::createCanvas(
    const KString& iName
)
{
    return( CanvasDOS::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createKeyboard
//
//  Description:
//
//      This creates the default keyboard object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the keyboard.
//
///////////////////////////////////////////////////////////////////////////////
Keyboard*
ReplayFactoryDOS::createKeyboard(
    const KString& iName
)
{
    return( KeyboardDOS::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createJoystick
//
//  Description:
//
//      This creates the default joystick object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the joystick.
//
///////////////////////////////////////////////////////////////////////////////
Joystick*
ReplayFactoryDOS::createJoystick(
    const KString& iName
)
{
    return( JoystickDOS::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createMouse
//
//  Description:
//
//      This creates the default mouse object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the mouse.
//
///////////////////////////////////////////////////////////////////////////////
Mouse*
ReplayFactoryDOS::createMouse(
    const KString& iName
)
{
    return( MouseDOS::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createSound
//
//  Description:
//
//      This creates the default sound object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the sound.
//
///////////////////////////////////////////////////////////////////////////////
Sound*
ReplayFactoryDOS::createSound(
    const KString& iName
)
{
    return( new SoundDOS( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createDisk
//
//  Description:
//
//      This creates the default disk object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the disk.
//
///////////////////////////////////////////////////////////////////////////////
Disk*
ReplayFactoryDOS::createDisk(
    const KString& iName
)
{
    return( new DiskDOS( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createNetwork
//
//  Description:
//
//      This creates the default network object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the network.
//
///////////////////////////////////////////////////////////////////////////////
Network*
ReplayFactoryDOS::createNetwork(
    const KString& iName
)
{
    //  What type of network support are we handling?
    if( Configuration::s_instance( ).getParam( "-ipx" ) )
    {
        return( new NetworkIPXDOS( iName ) );
    }
    else
    {
        return( new NetworkTCPDOS( iName ) );
    }
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createClock
//
//  Description:
//
//      This creates the default clock object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the clock.
//
///////////////////////////////////////////////////////////////////////////////
Clock*
ReplayFactoryDOS::createClock(
    const KString& iName
)
{
    return( new ClockDOS( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createGameFactory
//
//  Description:
//
//      This creates the default game factory object for DOS.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the game factory.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory*
ReplayFactoryDOS::createGameFactory(
    const KString& iName
)
{
    return( GameFactory::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCPU
//
//  Description:
//
//      This creates CPUs for use with the Linux platform.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      nCPUType (input)
//          The type of CPU to create.
//
//      pGame (input)
//          The game that the CPU belongs to.
//
//      pSpace (input)
//          The address space to provide to the new CPU.
//
//      ePurpose (input)
//          The purpose of CPU to create.
//
//      bCONSERVATIVE (input)
//          A TEMPORARY FLAG USED TO ALLOW US TO PICK THE 'C' BASED Z80
//          CORE DUE TO THE INSTABILITY OF THE ASM.
//
//  Returns:
//
//      A pointer to the CPU.
//
///////////////////////////////////////////////////////////////////////////////
CPU*
ReplayFactoryDOS::createCPU(
    const KString&        iName,
    const CPU::CPUType    eCPUType,
    Game*                 pGame,
    AddrSpace*            pSpace,
    const CPU::CPUPurpose ePurpose      /* = CPU::GAME */,
    const Byte            bCONSERVATIVE /* = FALSE */
)
{
    ASSERT( eCPUType < CPU::NUM_CPU_TYPES );
 
    //  Create the appropriate CPU.  We can use the nice, speedy ASM versions
    //  of the core on DOS.
    switch( eCPUType )
    {
#ifdef RELEASE
        case CPU::CPU_6502:
        {
            return( CPU6502NBA::s_build( iName, pGame, pSpace, ePurpose ) );
        }
        
        case CPU::CPU_6809:
        {
            return( CPU6809PLA::s_build( iName, pGame, pSpace, ePurpose ) );
        } 

        case CPU::CPU_68000:
        {
            return( CPU68000NCA::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        case CPU::CPU_Z80:
        {
            if( bCONSERVATIVE )
            {
                return( CPUZ80MDC::s_build( iName, pGame, pSpace, ePurpose ) );
            }
            else
            {
                return( CPUZ80NBA::s_build( iName, pGame, pSpace, ePurpose ) );
            }
        }
#else
        case CPU::CPU_6502:
        {
            return( CPU6502MFC::s_build( iName, pGame, pSpace, ePurpose ) );
        }
        
        case CPU::CPU_6809:
        {
            return( CPU6809KBC::s_build( iName, pGame, pSpace, ePurpose ) );
        } 

        case CPU::CPU_68000:
        {
            return( CPU68000BSC::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        case CPU::CPU_Z80:
        {
            return( CPUZ80MDC::s_build( iName, pGame, pSpace, ePurpose ) );
        }
#endif

        default:
        {
            fatalError( 
                "CPUFactory::createCPU - unknown CPU Type (%d).", eCPUType
            );
        }
    }

    //  Should never arrive here.
    return( NULL );
}
