///////////////////////////////////////////////////////////////////////////////
//
//  File:    soundd.cpp
//
//  Class:   SoundDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The SoundDOS class is a class encapsulating sound for the DOS
//      platform.  This requires the SEAL audio library v1.03.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "soundd.h"
#include "sample.h"
#include "config.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Member Initialization.
///////////////////////////////////////////////////////////////////////////////
const int32 SoundDOS::sm_nNumVoices = 16;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SoundDOS
//
//  Description:
//
//      This is the main constructor for a DOS sound object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SoundDOS::SoundDOS(
    const KString&  iName
)
:
    Sound        ( iName ),
    m_pVoiceList ( NULL ),
    m_pWaveList  ( NULL )
{
    //  A return code and a buffer to hold the error message.
    int32 nErr;
    char  strErr[ 80 ];


    //  If sound is disabled then don't bother initializing.
    if( !m_bEnabled )
    {
        return;
    }


    //  Initialize with the audio library.
    nErr = AInitialize( );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "AInitialize - %d: %s.", nErr, strErr );
    }
    

    //  What device is installed?
    AUDIOINFO info;
    nErr = APingAudio( &( info.nDeviceId ) );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "APingAudio - %d: %s.", nErr, strErr );
    }
    
    //  Get the capabilities of the detected device.
    AUDIOCAPS caps;
    nErr = AGetAudioDevCaps( info.nDeviceId, &caps );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "AGetAudioDevCaps - %d: %s.", nErr, strErr );
    }

    //  If there is no soundcard then disable sound.
    if( caps.wProductId == AUDIO_PRODUCT_NONE )
    {
        m_bEnabled = FALSE;
        return;
    }

    //  There are problems with the AWE32 driver since it attempts to
    //  upload sound data to the card's memory which slows the emulation
    //  down considerably and also messes up the sound emulation.
    //  Therefore, if an AWE32 is detected and the user hasn't explicitly
    //  requested the AWE32 driver then see if we can't find just a regular
    //  old sound blaster driver.
    if( 
        ( caps.wProductId == AUDIO_PRODUCT_AWE32 ) &&
        ( !Configuration::s_instance( ).getParam( "-awe" ) )
    )
    {
        for( UINT nDevId = 0 ; nDevId < AGetAudioNumDevs( ) ; nDevId += 1 )
        {
            nErr = AGetAudioDevCaps( nDevId, &caps );
            if( nErr != AUDIO_ERROR_NONE )
            {
                AGetErrorText( nErr, strErr, sizeof( strErr ) );
                fatalError( "AGetAudioDevCaps - %d: %s.", nErr, strErr );
            }
      
            if( caps.wProductId == AUDIO_PRODUCT_SB )
            {
                info.nDeviceId = nDevId;
                break;
            }
        }
    }

    nErr = AGetAudioDevCaps( info.nDeviceId, &caps );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "AGetAudioDevCaps - %d: %s.", nErr, strErr );
    }

    //  Is OPL emulation supported?
    if( 
        ( caps.wProductId == AUDIO_PRODUCT_SB    ) ||
        ( caps.wProductId == AUDIO_PRODUCT_SB15  ) ||
        ( caps.wProductId == AUDIO_PRODUCT_SB20  ) ||
        ( caps.wProductId == AUDIO_PRODUCT_SBPRO ) ||
        ( caps.wProductId == AUDIO_PRODUCT_SB16  ) ||
        ( caps.wProductId == AUDIO_PRODUCT_AWE32 )
    )
    {
        m_bSupportsOPL = TRUE;
    }


    //  Open up an auto-detect 16-bit mono device at the specified sample rate.
    info.wFormat     = AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
    info.nSampleRate = m_dwSampleRate;
    nErr = AOpenAudio( &info );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "AOpenAudio - %d: %s.", nErr, strErr );
    }
    

    //  Open the voices.
    nErr = AOpenVoices( sm_nNumVoices );
    if( nErr != AUDIO_ERROR_NONE )
    {
        AGetErrorText( nErr, strErr, sizeof( strErr ) );
        fatalError( "AOpenVoices - %d: %s.", nErr, strErr );
    }
    

    //  Initialize the voices.
    m_pVoiceList = new HAC[ sm_nNumVoices ];
    m_pWaveList  = new LPAUDIOWAVE[ sm_nNumVoices ];
    for( int32 nI = 0 ; nI < sm_nNumVoices ; nI += 1 )
    {
        //  Create the voice.
        nErr = ACreateAudioVoice( &( m_pVoiceList[ nI ] ) ); 
        if( nErr != AUDIO_ERROR_NONE )
        {
            AGetErrorText( nErr, strErr, sizeof( strErr ) );
            fatalError( "ACreateAudioVoice - %d: %s.", nErr, strErr );
        }

        //  Set the volume.
        nErr = ASetVoiceVolume( m_pVoiceList[ nI ], 64 );
        if( nErr != AUDIO_ERROR_NONE )
        {
            AGetErrorText( nErr, strErr, sizeof( strErr ) );
            fatalError( "ASetVoiceVolume - %d: %s.", nErr, strErr );
        }
        
        //  Set the panning.
        nErr = ASetVoicePanning( m_pVoiceList[ nI ], 128 );
        if( nErr != AUDIO_ERROR_NONE )
        {
            AGetErrorText( nErr, strErr, sizeof( strErr ) );
            fatalError( "ASetVoicePanning - %d: %s.", nErr, strErr );
        }

        //  Initialize the waveform.
        m_pWaveList[ nI ] = NULL; 
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SoundDOS
//
//  Description:
//
//      This is the destructor for a DOS sound object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SoundDOS::~SoundDOS(
)
{
    //  If sound is disabled then nothing to clean up.
    if( !m_bEnabled )
    {
        return;
    }

    //  Tear down the voices.
    for( int32 nI = 0 ; nI < sm_nNumVoices ; nI += 1 )
    {
        AStopVoice( m_pVoiceList[ nI ] );
        ADestroyAudioVoice( m_pVoiceList[ nI ] );

        if( m_pWaveList[ nI ] )
        {
            ADestroyAudioData( m_pWaveList[ nI ] );
            delete m_pWaveList[ nI ];
            m_pWaveList[ nI ] = NULL;
        }
    }

    //  Shutdown the audio.
    ACloseVoices( );
    ACloseAudio ( );


    //  Delete the voices and the waveforms.
    delete [] m_pVoiceList;
    delete [] m_pWaveList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the sound object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SoundDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SoundDOS" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumChannels
//
//  Description:
//
//      This member is called to return the number of voice channels.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of voices.
//
///////////////////////////////////////////////////////////////////////////////
DWord
SoundDOS::getNumChannels(
) const
{
    return( sm_nNumVoices );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the sound system to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      FALSE.
//
///////////////////////////////////////////////////////////////////////////////
Byte
SoundDOS::update(
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return( FALSE );
    }

    //  Update the audio.
    AUpdateAudio( );


    return( FALSE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: play
//
//  Description:
//
//      This member is called to play the specified sample.
//
//  Parameters:
//
//      pSample (input)
//          The sample to play.
//
//      bLoop (input)
//          The sample is to loop.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundDOS::play(
    Sample*    pSample,
    const Byte bLoop
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  A convenience to the channel.
    int32 nChannel = pSample->getChannel( );
    ASSERT( nChannel < sm_nNumVoices );

    //  If there is a waveform for this voice that isn't of
    //  the proper size then tear it down.
    if( 
        m_pWaveList[ nChannel ] && 
        ( m_pWaveList[ nChannel ]->dwLength != pSample->getSize( ) )
    )
    {
        AStopVoice( m_pVoiceList[ nChannel ] );
        ADestroyAudioData( m_pWaveList[ nChannel ] );
        delete m_pWaveList[ nChannel ];
        m_pWaveList[ nChannel ] = NULL;
    }

    //  If there is no waveform than create one.
    if( m_pWaveList[ nChannel ] == NULL )
    {
        m_pWaveList[ nChannel ] = new AUDIOWAVE;
 
        //  Build the format of the data.
        m_pWaveList[ nChannel ]->wFormat = 
            AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
        if( bLoop )
        {
            m_pWaveList[ nChannel ]->wFormat |= AUDIO_FORMAT_LOOP;
        }

        //  Fill out the rest of the information.
        m_pWaveList[ nChannel ]->nSampleRate = m_dwSampleRate;
        m_pWaveList[ nChannel ]->dwLength    = pSample->getSize( );
        m_pWaveList[ nChannel ]->dwLoopStart = 0;
        m_pWaveList[ nChannel ]->dwLoopEnd   = pSample->getSize( );

        //  Create the audio data.
        if( ACreateAudioData( m_pWaveList[ nChannel ] ) != AUDIO_ERROR_NONE )
        {
            delete m_pWaveList[ nChannel ];
            m_pWaveList[ nChannel ] = NULL;
            return;
        }
    }
    else
    { 
        //  Build the format of the data.
        m_pWaveList[ nChannel ]->wFormat = 
            AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
        if( bLoop )
        {
            m_pWaveList[ nChannel ]->wFormat |= AUDIO_FORMAT_LOOP;
        }
    }

    //  Copy the sample data in.
    memcpy( 
        ( void* )( m_pWaveList[ nChannel ]->lpData ), 
        ( void* )( pSample->getBuffer( ) ),
        pSample->getSize( )
    );

    //  Upload the data and play the voice.
    AWriteAudioData( m_pWaveList[ nChannel ], 0, pSample->getSize( ) );
    APlayVoice( m_pVoiceList[ nChannel ], m_pWaveList[ nChannel ] );
    ASetVoiceFrequency( m_pVoiceList[ nChannel ], pSample->getFrequency( ) );
    ASetVoiceVolume( m_pVoiceList[ nChannel ], pSample->getVolume( ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: playStreamed
//
//  Description:
//
//      This member is called to stream the specified sample.
//
//  Parameters:
//
//      pSample (input)
//          The sample to play.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundDOS::playStreamed(
    Sample* pSample
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  A convenience to the channel.
    int32 nChannel = pSample->getChannel( );
    ASSERT( nChannel < sm_nNumVoices );

    //  If there is a waveform for this voice that isn't of
    //  the proper size then tear it down.
    if( 
        m_pWaveList[ nChannel ] && 
        ( m_pWaveList[ nChannel ]->dwLength != pSample->getSize( ) )
    )
    {
        AStopVoice( m_pVoiceList[ nChannel ] );
        ADestroyAudioData( m_pWaveList[ nChannel ] );
        delete m_pWaveList[ nChannel ];
        m_pWaveList[ nChannel ] = NULL;
    }

    //  If there is no waveform than create one.
    if( m_pWaveList[ nChannel ] == NULL )
    {
        m_pWaveList[ nChannel ] = new AUDIOWAVE;
 
        //  Build the format of the data.
        m_pWaveList[ nChannel ]->wFormat = 
            AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;

        //  Fill out the rest of the information.
        m_pWaveList[ nChannel ]->nSampleRate = m_dwSampleRate;
        m_pWaveList[ nChannel ]->dwLength    = pSample->getSize( );
        m_pWaveList[ nChannel ]->dwLoopStart = 0;
        m_pWaveList[ nChannel ]->dwLoopEnd   = pSample->getSize( );

        //  Create the audio data.
        if( ACreateAudioData( m_pWaveList[ nChannel ] ) != AUDIO_ERROR_NONE )
        {
            delete m_pWaveList[ nChannel ];
            m_pWaveList[ nChannel ] = NULL;
            return;
        }
    }
    else
    { 
        //  Build the format of the data.
        m_pWaveList[ nChannel ]->wFormat = 
            AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
    }

    //  Copy the sample data in.
    memcpy( 
        ( void* )( m_pWaveList[ nChannel ]->lpData ), 
        ( void* )( pSample->getBuffer( ) ),
        pSample->getSize( )
    );

    //  Upload the data and play the voice.
    AWriteAudioData( m_pWaveList[ nChannel ], 0, pSample->getSize( ) );
    APlayVoice( m_pVoiceList[ nChannel ], m_pWaveList[ nChannel ] );
    ASetVoiceFrequency( m_pVoiceList[ nChannel ], pSample->getFrequency( ) );
    ASetVoiceVolume( m_pVoiceList[ nChannel ], pSample->getVolume( ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: stop
//
//  Description:
//
//      This member is called to stop sample playback on the specified
//      channel.
//
//  Parameters:
//
//      dwChannel (input)
//          The channel to stop playback on.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundDOS::stop(
    const DWord dwChannel
)
{
    ASSERT( ( int32 )dwChannel < sm_nNumVoices );

    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  Stop the voice.
    AStopVoice( m_pVoiceList[ dwChannel ] );
}
