///////////////////////////////////////////////////////////////////////////////
//
//  File:    winsockd.cpp
//
//  Class:   WinSockDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to provide access to the Win32 Winsock capabilities
//      from DOS.  Ideally, use of Indrek Mandre's libsocket or Dan Hedlund's
//      winsock library should be used, however I've had slight problems with
//      both and therefore this will be used for the time being.
//
//      Even though Indrek's and Dan's libraries aren't used, the code found
//      here owes a lot to their work.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <dpmi.h>
#include <sys/farptr.h>

//  Application Headers.
#include "winsockd.h"

///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
Byte WinSockDOS::sm_bInitialized = FALSE;
int  WinSockDOS::sm_nSocketP     = 0x0000;
int  WinSockDOS::sm_nSocketD     = 0x0000;


///////////////////////////////////////////////////////////////////////////////
//  Global Data Initialization.
///////////////////////////////////////////////////////////////////////////////
int g_anVxdLoadEntry[ 2 ] = { 0, 0 };
int g_anWinSockEntry[ 2 ] = { 0, 0 };

__dpmi_meminfo g_socketP;
__dpmi_meminfo g_socketD;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: WinSockDOS
//
//  Description:
//
//      This is the main constructor for the WinSock object.  It
//      is protected because it is a singleton and therefore cannot be 
//      instantiated by anyone but itself.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WinSockDOS::WinSockDOS(
    const KString&  iName
)
 :
    RepBase          ( iName )
{
    //  Make sure we haven't already initialized.
    ASSERT( sm_bInitialized == FALSE );

    //  Load in the winsock devices.
    VxdLoadDevice( "WSOCK.VXD" );
    VxdLoadDevice( "WSOCK.386" );
    VxdGetEntry( g_anWinSockEntry, 0x003e );

    g_socketP.handle  = 0;
    g_socketP.size    = 0x1000;
    g_socketP.address = 0;
    __dpmi_allocate_memory( &g_socketP );

    g_socketD.handle  = 0;
    g_socketD.size    = 0x1000;
    g_socketD.address = 0;
    __dpmi_allocate_memory( &g_socketD );

    sm_nSocketP = __dpmi_allocate_ldt_descriptors( 1 );
    sm_nSocketD = __dpmi_allocate_ldt_descriptors( 1 );

    __dpmi_set_segment_base_address( sm_nSocketP, g_socketP.address );
    __dpmi_set_segment_base_address( sm_nSocketD, g_socketD.address );

    __dpmi_set_segment_limit( sm_nSocketP, g_socketP.size );
    __dpmi_set_segment_limit( sm_nSocketD, g_socketD.size );

    //  Initialization is complete.
    sm_bInitialized = TRUE;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~WinSockDOS
//
//  Description:
//
//      This is the main destructor for the WinSock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WinSockDOS::~WinSockDOS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the WinSock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
WinSockDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "WinSockDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_instance
//
//  Description:
//
//      This member returns the singleton WinSock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A reference to the singleton WinSock object.
//
///////////////////////////////////////////////////////////////////////////////
WinSockDOS&
WinSockDOS::s_instance(
)
{
    //  The instance.
    static WinSockDOS winSockDOS( "WinSockDOS" );

    return( winSockDOS );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: VxdLoadDevice
//
//  Description:
//
//      This member is called to load in devices drivers for WinSock.
//
//  Parameters:
//
//      pstrName (input)
//          The file name of the device to load.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSockDOS::VxdLoadDevice(
    char* pstrName 
)
{
    //  The first time through, we have to initialize the loader.
    static Byte bFirstTime = TRUE;

    //  Error and exit codes.
    int nError;
    int nExit;


    //  If this is the first time through this function then initialize the
    //  loader.
    if( bFirstTime )
    {
        VxdGetEntry( g_anVxdLoadEntry, 0x0027 );
        CONFIRM( g_anVxdLoadEntry[ 1 ] != 0, "Could not load device entry" );
        bFirstTime = FALSE;
    }

    //  The memory information for the device.
    __dpmi_meminfo device;
    int            nDevice;
    
    //  Fill out the device information.
    device.handle  = 0;
    device.size    = strlen( pstrName ) + 1;
    device.address = 0;


    //  Load the device.
    __dpmi_allocate_memory( &device );
    nDevice = __dpmi_allocate_ldt_descriptors( 1 );
    __dpmi_set_segment_base_address( nDevice, device.address );
    __dpmi_set_segment_limit( nDevice, device.size );

    _farpokex( nDevice, 0, pstrName, device.size );

    asm(
        "pushl    %%ds\n"
        "movw     %%cx, %%ds\n"
        "lcall    %%cs:_g_anVxdLoadEntry\n"
        "movzwl   %%ax, %%ebx\n"
        "setc     %%al\n"
        "movzbl   %%al, %%eax\n"
        "popl     %%ds"
        : "=a" ( nError ), "=b" ( nExit )
        : "a" ( 1 ), "c" ( nDevice ), "d" ( 0 )
    );

    __dpmi_free_ldt_descriptor( nDevice );
    __dpmi_free_memory( device.handle );

    CHECK1( !nError, "Could not load device %s.", pstrName );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: VxdGetEntry
//
//  Description:
//
//      This member is called to retrieve the specified entry.
//
//  Parameters:
//
//      pEntry (output)
//          A buffer to hold the entry.
//
//      nID (input)
//          The id of the entry to retrieve.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSockDOS::VxdGetEntry(
    int* pEntry,
    int  nID
)
{
    //  Get the device entry.
    asm(
        "pushl    %%es\n"
        "movw     %%di, %%es\n"
        "intb     $0x2f\n"
        "movl     $0, %%ecx\n"
        "movw     %%es, %%cx\n"
        "popl     %%es\n"
        : "=c" ( pEntry[ 1 ] ), "=D" ( pEntry[ 0 ] )
        : "a" ( 0x1684 ), "b" ( nID ), "D" ( 0 )
        : "%eax", "%edx"
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: callVxD
//
//  Description:
//
//      This member is used to call a function in the VxD.
//
//  Parameters:
//
//      nFunction (input)
//          Indicates which function to call.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSockDOS::callVxD(
    int nFunction
)
{
    //  An error code.
    int nError;


    asm volatile(
        "pushl    %%es\n"
        "pushl    %%ecx\n"
        "popl     %%es\n"
        "lcall    _g_anWinSockEntry\n"
        "andl     %%eax, 0x0000ffff\n"
        "popl     %%es"
        : "=a" ( nError )
        : "a" ( nFunction ), "b" ( 0 ), "c" ( sm_nSocketP )
    );

    CONFIRM( 
        !nError || ( nError == 0xffff ) || ( nError == 10035 ),
        "callVxD( ) failed for function 0x%04x.",
        nFunction
    );
}
