///////////////////////////////////////////////////////////////////////////////
//
//  File:    wsocket.cpp
//
//  Class:   WinSocket
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a WinSock style socket for use in DOS.
//      Much of this code was derived from Dan Hedlund's wsock library.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <sys/farptr.h>
#include <netinet/in.h>

//  Application Headers.
#include "wsocket.h"
#include "winsockd.h"

///////////////////////////////////////////////////////////////////////////////
//  Constants.
///////////////////////////////////////////////////////////////////////////////
const int cnAF_INET     = 2;
const int cnSOCK_STREAM = 1;
const int cnIPPROTO_TCP = 6;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: WinSocket
//
//  Description:
//
//      This is the main constructor for a WinSocket object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WinSocket::WinSocket(
    const KString&  iName
)
 :
    RepBase          ( iName ),
    m_nSocketP       ( WinSockDOS::s_instance( ).getSocketP( ) ),
    m_nSocketD       ( WinSockDOS::s_instance( ).getSocketD( ) ),
    m_nSocket        ( 0 ),
    m_nSocketHandle  ( 0 ),
    m_nAddressLength ( 0 )
{
    //  Create the socket.  The call has the following structure:
    //      o Address Family.
    //      o Socket Type.
    //      o Protocol.
    //      o New Socket.
    //      o New Socket Handle.
    _farpokel( m_nSocketP, 0 * 4, cnAF_INET );
    _farpokel( m_nSocketP, 1 * 4, cnSOCK_STREAM );
    _farpokel( m_nSocketP, 2 * 4, cnIPPROTO_TCP );
    _farpokel( m_nSocketP, 3 * 4, 0 );
    _farpokel( m_nSocketP, 4 * 4, 0 );

    WinSockDOS::s_instance( ).callVxD( 0x0110 );

    m_nSocket       = _farpeekl( m_nSocketP, 3 * 4 );
    m_nSocketHandle = _farpeekl( m_nSocketP, 4 * 4 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: WinSocket
//
//  Description:
//
//      This constructor is used to create a socket that is a newly accepted
//      socket from an existing socket.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      source (input)
//          The socket to accept on.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WinSocket::WinSocket(
    const KString&  iName,
    WinSocket&      source
)
 :
    RepBase          ( iName ),
    m_nSocketP       ( WinSockDOS::s_instance( ).getSocketP( ) ),
    m_nSocketD       ( WinSockDOS::s_instance( ).getSocketD( ) ),
    m_nSocket        ( 0 ),
    m_nSocketHandle  ( 0 ),
    m_nAddressLength ( 0 )
{
    //  An address for the new socket.
    InetAddress addr;


    //  Accept the socket.  The call has the following structure:
    //      o Address.
    //      o Listening Socket.
    //      o Connected Socket.
    //      o Address Length.
    //      o Connected Socket Handle.
    //      o APC Routing.
    //      o APC Context.
    _farpokel( m_nSocketP, 0 * 4, ( m_nSocketP << 16 ) + ( 7 *  4 ) );
    _farpokel( m_nSocketP, 1 * 4, source.getSocket( ) );
    _farpokel( m_nSocketP, 2 * 4, 0 );
    _farpokel( m_nSocketP, 3 * 4, sizeof( InetAddress ) );
    _farpokel( m_nSocketP, 4 * 4, 0 );
    _farpokel( m_nSocketP, 5 * 4, 0 );
    _farpokel( m_nSocketP, 6 * 4, 0 );
    _farpokex( m_nSocketP, 7 * 4, &addr, sizeof( InetAddress ) );

    WinSockDOS::s_instance( ).callVxD( 0x0100 );

    m_nAddressLength = _farpeekl( m_nSocketP, 3 * 4 );
    _farpeekx( m_nSocketP, 7 * 4, &m_acAddress, m_nAddressLength );

    m_nSocket       = _farpeekl( m_nSocketP, 2 * 4 );
    m_nSocketHandle = _farpeekl( m_nSocketP, 4 * 4 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~WinSocket
//
//  Description:
//
//      This is the main destructor for the WinSocket object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WinSocket::~WinSocket(
)
{
    //  Clean up the socket.  The call has the following structure:
    //      o Socket.
    if( m_nSocket )
    {
        _farpokel( m_nSocketP, 0 * 4, m_nSocket );

        WinSockDOS::s_instance( ).callVxD( 0x0102 );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the WinSocket object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
WinSocket::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "WinSocket" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: bind
//
//  Description:
//
//      This member is called to bind a socket to a port.
//
//  Parameters:
//
//      nPort (input)
//          The port number to bind to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSocket::bind(
    const int nPort
)
{
    //  The address to bind to.
    InetAddress addr;


    //  Fill in the address.
    addr.m_nFamily  = cnAF_INET;
    addr.m_nPort    = htons( nPort );
    addr.m_nAddress = 0;


    //  Bind the socket.  The call has the following structure:
    //      o Address.
    //      o Socket.
    //      o Address Length.
    //      o APC Routine.
    //      o APC Context.
    //      o Address Structure.
    _farpokel( m_nSocketP, 0 * 4, ( m_nSocketP << 16 ) + ( 5 * 4 ) );
    _farpokel( m_nSocketP, 1 * 4, m_nSocket );
    _farpokel( m_nSocketP, 2 * 4, sizeof( InetAddress ) );
    _farpokel( m_nSocketP, 3 * 4, 0 );
    _farpokel( m_nSocketP, 4 * 4, 0 );
    _farpokex( m_nSocketP, 5 * 4, &addr, sizeof( InetAddress ) );

    WinSockDOS::s_instance( ).callVxD( 0x0101 );

    _farpeekx( m_nSocketP, 5 * 4, &addr, sizeof( InetAddress ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: listen
//
//  Description:
//
//      This member is called to listen on socket.
//
//  Parameters:
//
//      nMax (input)
//          The maximum back log.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSocket::listen(
    const int nMax
)
{
    //  Listen on the socket.  The call has the following structure:
    //      o Socket.
    //      o Max Backlog.
    _farpokel( m_nSocketP, 0 * 4, m_nSocket );
    _farpokel( m_nSocketP, 1 * 4, nMax );

    WinSockDOS::s_instance( ).callVxD( 0x0108 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: connect
//
//  Description:
//
//      This member is called to connect to a socket.
//
//  Parameters:
//
//      pstrHost (input)
//          The host to connect to.
//
//      nPort (input)
//          The port to connect to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WinSocket::connect(
    const char* pstrHost,
    const int   nPort 
)
{
    //  Parse the host name.
    int n[ 4 ];
    CONFIRM( 
        sscanf( pstrHost, "%d.%d.%d.%d", &n[0], &n[1], &n[2], &n[3] ) == 4,
        "Host must be specified in dot-notation (%s).",
        pstrHost
    );
    
    //  Create the address.
    InetAddress addr;
    addr.m_nFamily  = cnAF_INET;
    addr.m_nPort    = htons( nPort );
    addr.m_nAddress = 
        n[ 0 ] + ( n[ 1 ] << 8 ) + ( n[ 2 ] << 16 ) + ( n[ 3 ] << 24 );
    
    //  Connect to the socket.  The call has the following structure:
    //      o Address.
    //      o Socket.
    //      o Address Length.
    //      o APC Routine.
    //      o APC Context.
    //      o Address.
    _farpokel( m_nSocketP, 0 * 4, ( m_nSocketP << 16 ) + ( 5 * 4 ) );
    _farpokel( m_nSocketP, 1 * 4, m_nSocket );
    _farpokel( m_nSocketP, 2 * 4, sizeof( InetAddress ) );
    _farpokel( m_nSocketP, 3 * 4, 0 );
    _farpokel( m_nSocketP, 4 * 4, 0 );
    _farpokex( m_nSocketP, 5 * 4, &addr, sizeof( InetAddress ) );

    WinSockDOS::s_instance( ).callVxD( 0x0103 );

    m_nAddressLength = _farpeekl( m_nSocketP, 2 * 4 );
    _farpeekx( m_nSocketP, 7 * 4, &m_acAddress, m_nAddressLength );
    _farpeekx( m_nSocketP, 5 * 4, m_acAddress, sizeof( InetAddress ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: select
//
//  Description:
//
//      This member is called to select on a socket.
//
//  Parameters:
//
//      nType (input)
//          The type of select to perform.
//
//  Returns:
//
//      TRUE  if there is data waiting.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
WinSocket::select(
    const int nType
)
{
    //  Select on the socket.  The call has the following structure:
    //      o ReadList.
    //      o WriteList.
    //      o ExceptList.
    //      o ReadCount.
    //      o WriteCount.
    //      o ExceptCount.
    //      o Socket.
    //      o Type.
    _farpokel( m_nSocketP, 0 * 4, ( m_nSocketP << 16 ) + ( 6 * 4 ) );
    _farpokel( m_nSocketP, 1 * 4, 0 );
    _farpokel( m_nSocketP, 2 * 4, 0 );
    _farpokel( m_nSocketP, 3 * 4, 1 );
    _farpokel( m_nSocketP, 4 * 4, 0 );
    _farpokel( m_nSocketP, 5 * 4, 0 );
    _farpokel( m_nSocketP, 6 * 4, m_nSocket );
    _farpokel( m_nSocketP, 7 * 4, nType );
    _farpokel( m_nSocketP, 8 * 4, 0 );
    
    WinSockDOS::s_instance( ).callVxD( 0x010b );

    return( _farpeekl( m_nSocketP, 6 * 4 ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: recv
//
//  Description:
//
//      This member is called to receive a message.
//
//  Parameters:
//
//      pBuffer (output)
//          A buffer to store the message in.
//
//      nLength (input)
//          The length of the buffer.
//
//  Returns:
//
//      The number of bytes read.
//
///////////////////////////////////////////////////////////////////////////////
int
WinSocket::recv(
    void* pBuffer,
    int   nLength
)
{
    //  Read from the socket.  The call has the following structure:
    //      o Buffer Pointer.
    //      o Address.
    //      o Socket.
    //      o Buffer Length.
    //      o Flags.
    //      o Address Length.
    //      o Bytes Received.
    //      o APC Routine.
    //      o APC Context.
    //      o Timeout.
    _farpokel( m_nSocketP,  0 * 4, m_nSocketD << 16 );
    _farpokel( m_nSocketP,  1 * 4, ( m_nSocketP << 16 ) + ( 10 * 4 ) );
    _farpokel( m_nSocketP,  2 * 4, m_nSocket );
    _farpokel( m_nSocketP,  3 * 4, nLength );
    _farpokel( m_nSocketP,  4 * 4, 0 );
    _farpokel( m_nSocketP,  5 * 4, m_nAddressLength );
    _farpokel( m_nSocketP,  6 * 4, 0 );
    _farpokel( m_nSocketP,  7 * 4, 0 );
    _farpokel( m_nSocketP,  8 * 4, 0 );
    _farpokel( m_nSocketP,  9 * 4, 0 );
    _farpokex( m_nSocketP, 10 * 4, m_acAddress, m_nAddressLength );

    WinSockDOS::s_instance( ).callVxD( 0x0109 );

    _farpeekx( m_nSocketD, 0, pBuffer, _farpeekl( m_nSocketP, 6 * 4 ) );

    return( _farpeekl( m_nSocketP, 6 * 4 ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: send
//
//  Description:
//
//      This member is called to send a message.
//
//  Parameters:
//
//      pBuffer (input)
//          A buffer containing the message.
//
//      nLength (input)
//          The length of the buffer.
//
//  Returns:
//
//      The number of bytes sent.
//
///////////////////////////////////////////////////////////////////////////////
int
WinSocket::send(
    void* pBuffer,
    int   nLength
)
{
    //  Write to the socket.  The call has the following structure:
    //      o Buffer Pointer.
    //      o Address.
    //      o Socket.
    //      o Buffer Length.
    //      o Flags.
    //      o Address Length.
    //      o Bytes Sent.
    //      o APC Routine.
    //      o APC Context.
    //      o Timeout.
    _farpokel( m_nSocketP,  0 * 4, m_nSocketD << 16 );
    _farpokel( m_nSocketP,  1 * 4, ( m_nSocketP << 16 ) + ( 10 * 4 ) );
    _farpokel( m_nSocketP,  2 * 4, m_nSocket );
    _farpokel( m_nSocketP,  3 * 4, nLength );
    _farpokel( m_nSocketP,  4 * 4, 0 );
    _farpokel( m_nSocketP,  5 * 4, m_nAddressLength );
    _farpokel( m_nSocketP,  6 * 4, 0 );
    _farpokel( m_nSocketP,  7 * 4, 0 );
    _farpokel( m_nSocketP,  8 * 4, 0 );
    _farpokel( m_nSocketP,  9 * 4, 0 );
    _farpokex( m_nSocketP, 10 * 4, m_acAddress, m_nAddressLength );
    _farpokex( m_nSocketD,  0 * 4, pBuffer, nLength );

    WinSockDOS::s_instance( ).callVxD( 0x010d );

    return( _farpeekl( m_nSocketP, 6 * 4 ) );
}
