///////////////////////////////////////////////////////////////////////////////
//
//  File:    appfile.cpp
//
//  Class:   AppFile
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The AppFile class encapsulates the concept of an application file.
//      An application file is one that is used by the Replay application.
//      It is a flexible file that can be opened, read, written, have
//      characters inserted, deleted, etc.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <unistd.h>

//  Application Headers.
#include "appfile.h"
#include "replay.h"
#include "disk.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Initializations.
///////////////////////////////////////////////////////////////////////////////
//  The size transfer buffer and its size.
Byte  AppFile::sm_abTransfer[ 1000 ];
DWord AppFile::sm_dwTransferSize = sizeof( AppFile::sm_abTransfer );



///////////////////////////////////////////////////////////////////////////////
//
//  Function: AppFile
//
//  Description:
//
//      This is the main constructor for an application file object.
//
//  Parameters:
//
//      fileName (input)
//          The name of the file the object should open.
//
//      bCreate (input)
//          Indicates whether or not the file should be created if it does
//          not exist.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AppFile::AppFile(
    const KString& fileName,
    const Byte     bCreate   /* = TRUE */
)
:
    RepBase           ( fileName ),
    m_fileName        ( ),
    m_fpFile          ( NULL ),
    m_dwCount         ( 0 ),
    m_dwTotal         ( 0 )
{
    //  Open the file.
    open( fileName, bCreate );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~AppFile
//
//  Description:
//
//      This is the destructor for an application file object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AppFile::~AppFile(
)
{
    //  Close the file if it's open.
    if( m_fpFile != NULL )
    {
        close( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the AppFile object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
AppFile::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "AppFile" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: open
//
//  Description:
//
//      This member opens the specified file for read/write.
//
//  Parameters:
//
//      fileName (input)
//          The name of the file to open.
//
//      bCreate (input)
//          Indicates whether or not the file should be created if it does
//          not exist.
//
//  Returns:
//
//      TRUE  if the file was opened successfully. 
//      FALSE if the file could not be opened.
//
///////////////////////////////////////////////////////////////////////////////
Byte
AppFile::open(
    const KString& fileName,
    const Byte     bCreate    /* = TRUE */
)
{
    //  Make sure a file is not already open.
    CONFIRM( 
        m_fpFile == NULL,
        "Attempting to open %s when %s is already open\n",
        fileName.data( ),
        m_fileName.data( )
    );

    //  Attempt to open the file.  If the file can't be opened (i.e. it 
    //  doesn't exist) then create it if we were instructed to, otherwise
    //  just return).
    m_fpFile = fopen( fileName, "rb+" );
    if( m_fpFile == NULL )
    {
        //  If the caller wants to create the file then do it, otherwise
        //  return FALSE.
        if( bCreate )
        {
            //  First, make sure all directory components exist.
            Replay::s_instance( ).getDisk( )->makeDirectoryComponents(
                fileName 
            );

            //  Now create the file.
            m_fpFile = fopen( fileName, "wb+" );
            if( m_fpFile == NULL )
            {
                return( FALSE );
            }
        }
        else
        {
            return( FALSE );
        }
    }

    //  Hold on to the file name.
    m_fileName = fileName;

    //  Return whether or not the file has been opened successfully.
    return( m_fpFile != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: close
//
//  Description:
//
//      This member closes the currently open file.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AppFile::close(
)
{
    //  Make sure a file is open.
    CONFIRM( m_fpFile != NULL, "Attempting to close when no file is open" );

    //  Close the file.
    fclose( m_fpFile );

    //  Mark the file closed.
    m_fileName = "";
    m_fpFile   = NULL;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: insertBlock
//
//  Description:
//
//      This member is used to insert a blank area into the file.  It will
//      shift all of the existing data towards the end of the file to make
//      room (extending the file end of course).
//
//  Parameters:
//
//      dwOffset (input)
//          Insert before this position.
//
//      dwSize (input)
//          Insert this many bytes.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AppFile::insertBlock(
    DWord dwOffset,
    DWord dwSize
)
{
    ASSERT( m_fpFile != NULL );
    ASSERT( dwSize   > 0 );

    //  The start and end of the current block to move.
    int32 nBlockStart;
    int32 nBlockEnd;

    //  The number of bytes to move and the number of blocks processed.
    DWord dwBytesToMove;
    DWord dwBlocksProcessed;


    //  If we are inserting after the end of the file then don't bother.
    if( dwOffset >= length( ) )
    {
        return;
    }

    //  Loop until the end of the current block is less than the offset.
    for( 
        nBlockEnd  = length( ) - 1 ;
        nBlockEnd >= ( int32 )dwOffset ;
        nBlockEnd -= sm_dwTransferSize
    )
    {
        //  Calculate the block start.
        nBlockStart = nBlockEnd - sm_dwTransferSize + 1;
 
        //  If the start of the block is less than the offset then make it
        //  the offset.
        if( nBlockStart < ( int32 )dwOffset )
        {
            nBlockStart = dwOffset;
        }

        //  The number of bytes to move.
        dwBytesToMove = nBlockEnd - nBlockStart + 1;

        //  Read the block.  We use the direct read and write instead of
        //  our virtualized read and write so we can read/write blocks instead
        //  of single characters.  Also, we don't want to affect the state
        //  of the count data member.
        dwBlocksProcessed = fread( 
            ( void* )&( sm_abTransfer[ 0 ] ), dwBytesToMove, 1, m_fpFile 
        );
        CONFIRM( dwBlocksProcessed == 1, "Failed reading during blank insert" );

        //  Move to the new location for the block.
        fseek( m_fpFile, nBlockStart + dwSize, BEG );
        
        //  Write the block.
        dwBlocksProcessed = fwrite( 
            ( void* )&( sm_abTransfer[ 0 ] ), dwBytesToMove, 1, m_fpFile 
        );
        CONFIRM( dwBlocksProcessed == 1, "Failed writing during blank insert" );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteBlock
//
//  Description:
//
//      This member is used to delete an area from the file.  It will
//      shift all of the existing data towards the start of the file to fill
//      in.
//
//  Parameters:
//
//      dwOffset (input)
//          Delete starting at this position.
//
//      dwSize (input)
//          Delete this many bytes.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AppFile::deleteBlock(
    DWord dwOffset,
    DWord dwSize
)
{
    ASSERT( m_fpFile != NULL );
    ASSERT( dwSize   > 0 );

    //  The start and end of the current block to move.
    int32 nBlockStart;
    int32 nBlockEnd;

    //  The number of bytes to move and the number of blocks processed.
    DWord dwBytesToMove;
    DWord dwBlocksProcessed;


    //  Don't allow deletion past the end of the file.
    if( dwOffset + dwSize > length( ) )
    {
        return;
    }

    //  Loop until the start of the current block is greater than the eof.
    for( 
        nBlockStart  = dwOffset + dwSize ;
        nBlockStart  < ( int32 )length( ) ;
        nBlockStart += sm_dwTransferSize
    )
    {
        //  Calculate the block end.
        nBlockEnd = nBlockStart + sm_dwTransferSize - 1;
 
        //  If the end of the block extends past the eof then limit it.
        if( nBlockEnd >= ( int32 )length( ) )
        {
            nBlockEnd = length( ) - 1;
        }

        //  The number of bytes to move.
        dwBytesToMove = nBlockEnd - nBlockStart + 1;

        //  Seek to the start of the block to read.
        fseek( m_fpFile, nBlockStart, BEG );

        //  Read the block.  We use the direct read and write instead of
        //  our virtualized read and write so we can read/write blocks instead
        //  of single characters.  Also, we don't want to affect the state
        //  of the count data member.
        dwBlocksProcessed = fread( 
            ( void* )&( sm_abTransfer[ 0 ] ), dwBytesToMove, 1, m_fpFile 
        );
        CONFIRM( dwBlocksProcessed == 1, "Failed reading during block delete" );

        //  Move to the new location for the block.
        fseek( m_fpFile, nBlockStart - dwSize, BEG );
        
        //  Write the block.
        dwBlocksProcessed = fwrite( 
            ( void* )&( sm_abTransfer[ 0 ] ), dwBytesToMove, 1, m_fpFile 
        );
        CONFIRM( dwBlocksProcessed == 1, "Failed writing during blank insert" );
    }

    //  The block has been deleted.  Time to truncate the file.
#ifdef DOS
    fwrite( ( void* )&( sm_abTransfer[ 0 ] ), 0, 0, m_fpFile );
#else
    truncate( m_fileName.data( ), length( ) - dwSize );
#endif
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: length
//
//  Description:
//
//      This member returns the current length of the file.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The length of the file.
//
///////////////////////////////////////////////////////////////////////////////
DWord
AppFile::length(
)
{
    //  Make sure a file is open.
    ASSERT( m_fpFile != NULL );

    //  The length of the file.
    DWord dwLength;

    //  The current position in the file.
    DWord dwCurPos = ftell( m_fpFile );

    //  Calculate away the length of the file.
    fseek( m_fpFile, 0, END );
    dwLength = ftell( m_fpFile );
    fseek( m_fpFile, 0, BEG );
    dwLength -= ftell( m_fpFile );

    //  Now go back to where we were.
    fseek( m_fpFile, dwCurPos, BEG );

    //  Return the length.
    return( dwLength );
}
