///////////////////////////////////////////////////////////////////////////////
//
//  File:       appfile.h
//
//  Class:      AppFile - Concrete
//  
//  Hierarchy:  AppFile - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The AppFile class encapsulates the concept of an application file.
//      An application file is one that is used by the Replay application.
//      It is a flexible file that can be opened, read, written, have
//      characters inserted, deleted, etc.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _appfile_h_
#define _appfile_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>

//  Application Headers.
#include "repbase.h"
#include "kstring.h"


///////////////////////////////////////////////////////////////////////////////
//  AppFile Class Definition
///////////////////////////////////////////////////////////////////////////////
class AppFile : public RepBase
{
    public:
        
        //  Possible seek directives.
        typedef enum
        {
            BEG = SEEK_SET,
            CUR = SEEK_CUR,
            END = SEEK_END
        }
        AppFileSeek;

        //  Creation/Deletion.
                 AppFile ( const KString& fileName, const Byte bCreate = TRUE );
        virtual ~AppFile ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        Byte  open    ( const KString& fileName, const Byte bCreate = TRUE );
        Byte  isOpen  ( );
        void  close   ( );
        void  seek    ( 
            const DWord dwOffset, const AppFileSeek eSeekDir = BEG
        );
        DWord tell       ( );
        void  read       ( char*                pstrDest,   DWord dwCount );
        void  read       ( unsigned char*       pstrDest,   DWord dwCount );
        void  write      ( const char*          pstrSource, DWord dwCount );
        void  write      ( const unsigned char* pstrSource, DWord dwCount );
        DWord length     ( );
        DWord count      ( ) const;
        DWord total      ( ) const;
        void  clearTotal ( );

        //  Special
        void  insertBlock ( DWord dwOffset, DWord dwSize );
        void  deleteBlock ( DWord dwOffset, DWord dwSize );

    protected:

        //  Member Data.
        KString  m_fileName;
        FILE*    m_fpFile;
        DWord    m_dwCount;
        DWord    m_dwTotal;

    private:

        //  Static Members.
        static Byte  sm_abTransfer[ 1000 ];
        static DWord sm_dwTransferSize;
};

///////////////////////////////////////////////////////////////////////////////
//  AppFile Inline Functions
///////////////////////////////////////////////////////////////////////////////
inline
Byte
AppFile::isOpen(
)
{
    return( m_fpFile != NULL );
}

inline
void
AppFile::seek(
    const DWord         dwOffset, 
    const AppFileSeek   eSeekDir  /* = BEG */
)
{
    ASSERT( m_fpFile );
    fseek( m_fpFile, dwOffset, eSeekDir );
}

inline
DWord
AppFile::tell(
)
{
    ASSERT( m_fpFile );
    return( ftell( m_fpFile ) );
}
 
inline
void
AppFile::read(
    char* pstrDest,
    DWord dwCount
)
{
    ASSERT( m_fpFile );
    m_dwCount = fread( ( void* ) pstrDest, 1, dwCount, m_fpFile );
    m_dwTotal += m_dwCount;
}
 
inline
void
AppFile::read(
    unsigned char* pstrDest,
    DWord          dwCount
)
{
    ASSERT( m_fpFile );
    m_dwCount = fread( ( void* )pstrDest, 1, dwCount, m_fpFile );
    m_dwTotal += m_dwCount;
}
 
inline
void
AppFile::write(
    const char* pstrSource,
    DWord       dwCount
)
{
    ASSERT( m_fpFile );
    m_dwCount = fwrite( ( void* )pstrSource, 1, dwCount, m_fpFile );
    m_dwTotal += m_dwCount;
    fflush( m_fpFile );
}
 
inline
void
AppFile::write(
    const unsigned char* pstrSource,
    DWord                dwCount
)
{
    ASSERT( m_fpFile );
    m_dwCount = fwrite( ( void* )pstrSource, 1, dwCount, m_fpFile );
    m_dwTotal += m_dwCount;
    fflush( m_fpFile );
}
 
inline
DWord
AppFile::count(
) const
{
    return( m_dwCount );
}
 
inline
DWord
AppFile::total(
) const
{
    return( m_dwTotal );
}
 
inline
void
AppFile::clearTotal(
)
{
    m_dwTotal = 0;
}
 
 
#endif
