///////////////////////////////////////////////////////////////////////////////
//
//  File:       gfile.h
//
//  Class:      GameFile - Abstract
//  
//  Hierarchy:  GameFile - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The GameFile class serves a base class for a group of classes that
//      are responsible for manipulating files used exclusively for the 
//      emulated games (i.e. ROMs, Samples, etc.).  
//
//      The reason why this is a separate hierarchy from the AppFile is that
//      an application file (e.g. Registry) is always a plain disk file found
//      in a specific location that can be both read and written to.  Game
//      files, however, can be found in one of numerous places depending
//      on the preference of the user.  Maybe the user likes to keep their
//      ROMs uncompressed in directories named for the games, maybe they
//      like to keep them in a zipfile, maybe they like to have them gziped
//      or unix compressed.  
//
//      GameFile provides an abstract interface to retrieve these files
//      and the subclasses are used to implement a certain method.
//
//      An additional note is that Game Files are always read from and never
//      written to which simplifies the interface.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _gfile_h_
#define _gfile_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>

//  Application Headers.
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"


///////////////////////////////////////////////////////////////////////////////
//  GameFile Class Definition
///////////////////////////////////////////////////////////////////////////////
class GameFile : public RepBase
{
    public:
        
        //  Creation/Deletion.
                 GameFile ( const KString& gameId, const KString& fileName );
        virtual ~GameFile ( );

        //  Interface.
        Byte  isOpen  ( );
        DWord length  ( );
        
        //  Retrieval.
        const DWord read     ( 
            Byte*       pbDest, 
            const DWord dwMaxLength, 
            const DWord dwOffset = 0x0000
        );
        const DWord readSkip ( 
            Byte*       pbDest, 
            const DWord dwMaxLength, 
            const DWord dwOffset = 0x0000
        );

    protected:

        //  Member Data.
        KString  m_gameId;
        KString  m_fileName;
        Byte*    m_pbBuffer;
        DWord    m_dwLength;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  GameFile Inline Functions
///////////////////////////////////////////////////////////////////////////////
inline
Byte
GameFile::isOpen(
)
{
    return( m_pbBuffer != NULL );
}

inline
DWord
GameFile::length(
)
{
    return( m_dwLength );
}

 
#endif
