///////////////////////////////////////////////////////////////////////////////
//
//  File:    regular.cpp
//
//  Class:   RegularFile
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The RegularFile class is a GameFile class that implements reading
//      of a plain old vanilla disk file.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>

//  Application Headers.
#include "regular.h"
#include "replay.h"
#include "disk.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: RegularFile
//
//  Description:
//
//      This is the main constructor for a regular file object.
//
//  Parameters:
//
//      gameId (input)
//          The id of the game to which the file belongs.
//
//      fileName (input)
//          The name of the file.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
RegularFile::RegularFile(
    const KString& gameId,
    const KString& fileName
)
:
    GameFile          ( gameId, fileName )
{
    //  Attempt to open the file.
    open( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~RegularFile
//
//  Description:
//
//      This is the destructor for a regular file object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
RegularFile::~RegularFile(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the RegularFile object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
RegularFile::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "RegularFile" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: open
//
//  Description:
//
//      This is called to open the specified file and read it into memory.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
RegularFile::open(
)
{
    //  The full path to the desired file.
    KString fullFileName;

    //  A file pointer to the file we are going to open.
    FILE* fpFile;

    //  Check if the file exists.
    DWord dwSize = s_fileExists( m_gameId, m_fileName, &fullFileName );

    //  If the file doesn't exist then return.
    if( dwSize == 0 )
    {
        return;
    }

    //  Open the file.
    fpFile = fopen( fullFileName.data( ), "rb" );
    if( fpFile == NULL )
    {
        return;
    }

    //  Assign the length of the file.
    m_dwLength = dwSize;

    //  Allocate space to hold the file.
    m_pbBuffer = new Byte[ m_dwLength ];

    //  Read the file in.
    if( fread( ( void* )m_pbBuffer, m_dwLength, 1, fpFile ) != 1 )
    {
        CONFIRM( FALSE, "Problem reading %s.", fullFileName.data( ) );
    }

    //  Close up the file.
    fclose( fpFile );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_fileExists
//
//  Description:
//
//      This is called to check if the specified file exists as a regular
//      file or not.
//
//  Parameters:
//
//      gameId (input)
//          The id of the game to which the file belongs.
//
//      fileName (input)
//          The name of the file.
//
//      pFullFileName (output)
//          A pointer to the full output file name.
//
//  Returns:
//
//      The size of the file or 0 if not found.
//
///////////////////////////////////////////////////////////////////////////////
DWord
RegularFile::s_fileExists(
    const KString& gameId,
    const KString& fileName,
    KString*       pFullFileName /* = NULL */
)
{
    //  Indices used for iterations.
    DWord dwI;

    //  The size of the file.
    DWord dwSize( 0 );

    //  The full path to the desired file.
    KString fullFileName;

    //  Get the handle to the disk object.
    Disk* pDisk = Replay::s_instance( ).getDisk( );
    ASSERT( pDisk != NULL );

    //  How many paths are there to search?
    DWord dwNumPaths = pDisk->getNumPaths( );
 

    //  Check each path specified.
    for( dwI = 0 ; dwI < dwNumPaths ; dwI += 1 )
    {
        //  The first place to look is in <path>/<gameId>/<file>.
        fullFileName  = pDisk->getPath( dwI );
        fullFileName += pDisk->getDirectorySeparator( );
        fullFileName += gameId;
        dwSize = pDisk->checkForFile( fullFileName, fileName );
        if( dwSize )
        {
            break;
        }

        //  Next check <path>/roms/<gameId>/<file>.
        fullFileName  = pDisk->getPath( dwI );
        fullFileName += pDisk->getDirectorySeparator( );
        fullFileName += "roms";
        fullFileName += pDisk->getDirectorySeparator( );
        fullFileName += gameId;
        dwSize = pDisk->checkForFile( fullFileName, fileName );
        if( dwSize )
        {
            break;
        }

        //  Next check <path>/roms/<file>.
        fullFileName  = pDisk->getPath( dwI );
        fullFileName += pDisk->getDirectorySeparator( );
        fullFileName += "roms";
        dwSize = pDisk->checkForFile( fullFileName, fileName );
        if( dwSize )
        {
            break;
        }
    }

    //  If the file was found and the client wants the full name then assign 
    //  it.
    if( dwSize && pFullFileName )
    {
        *pFullFileName = fullFileName;
    }
    return( dwSize );
}
