///////////////////////////////////////////////////////////////////////////////
//
//  File:    buffer.cpp
//
//  Class:   Buffer
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class can be used to hold a buffer of bytes.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <string.h>

//  Application Headers.
#include "reptypes.h"
#include "replay.h"
#include "buffer.h"
#include "disk.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Buffer
//
//  Description:
//
//      This is the main constructor for a buffer object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSize (input)
//          The size of the address space. 
//
//      bSet (input)
//          The value to set the buffer to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Buffer::Buffer(
    const KString&  iName,
    DWord           dwSize,
    Byte            bSet    /* = 0x00 */
)
:
    RepBase           ( iName ),
    m_dwSize          ( dwSize ),
    m_pbBuffer        ( NULL )
{
    //  Check the parameters.
    ASSERT( dwSize > 0 );

    //  Allocate the buffer.
    m_pbBuffer = new Byte[ m_dwSize ];

    //  Zero the buffer.
    memset( ( void* )m_pbBuffer, ( int )bSet, dwSize );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Buffer
//
//  Description:
//
//      This is a secondary constructor for a buffer object that takes a
//      pointer to the data that is to be initially contained in the buffer.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSize (input)
//          The size of the address space. 
//
//      pbSource (input)
//          The source buffer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Buffer::Buffer(
    const KString&  iName,
    DWord           dwSize,
    const Byte*     pbSource
)
:
    RepBase           ( iName ),
    m_dwSize          ( dwSize ),
    m_pbBuffer        ( NULL )
{
    //  Check the parameters.
    ASSERT( dwSize    > 0 );
    ASSERT( pbSource != NULL );

    //  Allocate the buffer.
    m_pbBuffer = new Byte[ m_dwSize ];

    //  Copy from the source buffer.
    memcpy( ( void* )m_pbBuffer, ( void* )pbSource, dwSize );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Buffer
//
//  Description:
//
//      This is the destructor for the buffer object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Buffer::~Buffer(
)
{
    ASSERT( m_pbBuffer != NULL );

    //  Free up the buffer.
    delete [] m_pbBuffer;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the buffer object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Buffer::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Buffer" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the buffer to a file for retrieval at a later 
//      date.
//
//  Parameters:
//
//      pSaveFile (input)
//          The file to save the buffer to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Buffer::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  Save the buffer contents to the file.
    pSaveFile->write( m_pbBuffer, m_dwSize );
 
    //  Was it saved successfully?
    return( pSaveFile->count( ) != m_dwSize );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the buffer from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile (input)
//          The file to load the buffer from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Buffer::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  Load the buffer contents from the file.
    pLoadFile->read( m_pbBuffer, m_dwSize );

    //  Was it loaded successfully?
    return( pLoadFile->count( ) != m_dwSize );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadFile
//
//  Description:
//
//      This member is called to load a file (or a piece of) into the 
//      buffer.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to store the file at.
//
//      gameId (input)
//          The name of the game the file is for.
//
//      fileName (input)
//          The name of the file.
//
//      dwOffset (input)
//          The offset into the file to start loading from.
//
//      dwLength (input)
//          The length to load in.
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
Buffer::loadFile(
    const DWord     dwAddress,
    const KString&  gameId,
    const KString&  fileName,
    const DWord     dwOffset /* =0x0000 */,
    const DWord     dwLength /* =0x0000 */
)
{
    //  The number of bytes to load into the ROM.
    DWord dwBytesToLoad;


    //  If the length specified is 0 or if the length would exceed the
    //  end of the buffer then we calculate the number of bytes to load
    //  as the length between the specified address and the end of the
    //  address space, otherwise we just use the length.
    if( ( dwLength == 0x0000 ) || ( ( dwAddress + dwLength ) > m_dwSize ) )
    {
        dwBytesToLoad = m_dwSize - dwAddress;
    }
    else
    {
        dwBytesToLoad = dwLength;
    }


    //  Load the game file.
    return(
        Replay::s_instance( ).getDisk( )->gameFileLoad(
            gameId,
            fileName,
            m_pbBuffer + dwAddress,
            dwBytesToLoad,
            dwOffset
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadFileSkip
//
//  Description:
//
//      This member is called to load a file (or a piece of) into the 
//      buffer at every other byte.  This is typically
//      used for games that store the even bytes in one ROM and the odd
//      bytes in another ROM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to store the file at.
//
//      gameId (input)
//          The name of the game the file is for.
//
//      fileName (input)
//          The name of the file.
//
//      dwOffset (input)
//          The offset into the file to start loading from.
//
//      dwLength (input)
//          The length to load in.
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
Buffer::loadFileSkip(
    const DWord     dwAddress,
    const KString&  gameId,
    const KString&  fileName,
    const DWord     dwOffset /* =0x0000 */,
    const DWord     dwLength /* =0x0000 */
)
{
    //  The number of bytes to load into the ROM.
    DWord dwBytesToLoad;


    //  If the length specified is 0 or if the length would exceed the
    //  end of the buffer then we calculate the number of bytes to load
    //  as the length between the specified address and the end of the
    //  address space, otherwise we just use the length.
    if( 
        ( dwLength == 0x0000 ) || 
        ( ( dwAddress + 2 * ( dwLength - 1 ) ) >= m_dwSize ) 
    )
    {
        dwBytesToLoad = ( m_dwSize - dwAddress + 1 ) / 2;
    }
    else
    {
        dwBytesToLoad = dwLength;
    }


    //  Load the game file.
    return(
        Replay::s_instance( ).getDisk( )->gameFileLoad(
            gameId,
            fileName,
            m_pbBuffer + dwAddress,
            dwBytesToLoad,
            dwOffset,
            TRUE
        )
    );
}
