///////////////////////////////////////////////////////////////////////////////
//
//  File:    cqueue.cpp
//
//  Class:   CommandQueue
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a queue of commands.  It is used as a buffer
//      between CPUs to allow one CPU to pass 1 byte commands to another.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "cqueue.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CommandQueue
//
//  Description:
//
//      This is the main constructor for a command queue object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSize (input)
//          The size of the buffer.
// 
//      bSustain (input)
//          If sustain is specified, then when the queue is empty, the
//          last command removed will continuously be returned.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CommandQueue::CommandQueue(
    const KString& iName, 
    const DWord    dwSize   /* = 0x08 */, 
    const Byte     bSustain /* = FALSE */
)
:
    Buffer            ( iName, dwSize ),
    m_bSustain        ( bSustain ),
    m_dwIndex         ( 0 )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CommandQueue
//
//  Description:
//
//      This is the destructor for the command queue object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CommandQueue::~CommandQueue(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the command queue object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CommandQueue::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CommandQueue" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member is called to save the current state of the command queue
//      to a save game file.  This can be retrieved with the load member.
//
//  Parameters:
//
//      pSaveFile (input)
//          A pointer to the file to save to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CommandQueue::save( 
    AppFile* pSaveFile
)
{
    //  The number of errors encountered during the save.
    DWord dwNumErrors = 0;

    //  We always allow the base class to save itself first.
    dwNumErrors += Buffer::save( pSaveFile );

    //  Now save our attributes.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bSustain, sizeof( m_bSustain ) );
    pSaveFile->write( ( Byte* )&m_dwIndex, sizeof( m_dwIndex ) );
    if( pSaveFile->total( ) != ( sizeof( m_bSustain ) + sizeof( m_dwIndex ) ) )
    {
        dwNumErrors += 1;
    }

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member is called to load the current state of the command queue
//      from a save game file that was created in the save member.
//
//  Parameters:
//
//      pLoadFile (input)
//          A pointer to the file to load from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CommandQueue::load( 
    AppFile* pLoadFile
)
{
    //  The number of errors encountered during the load.
    DWord dwNumErrors = 0;


    //  We always allow the base class to load itself first.
    dwNumErrors += Buffer::load( pLoadFile );

    //  Now load our attributes.
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bSustain, sizeof( m_bSustain ) );
    pLoadFile->read( ( Byte* )&m_dwIndex, sizeof( m_dwIndex ) );
    if( pLoadFile->total( ) != ( sizeof( m_bSustain ) + sizeof( m_dwIndex ) ) )
    {
        dwNumErrors += 1;
    }

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}





///////////////////////////////////////////////////////////////////////////////
//
//  Function: addCommand
//
//  Description:
//
//      This member adds a command to the queue.
//
//  Parameters:
//
//      bCommand (input)
//          The command to add.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CommandQueue::addCommand(
    const Byte bCommand
)
{
    //  Add the command to the buffer.
    if( m_dwIndex < m_dwSize )
    {
        m_pbBuffer[ m_dwIndex ] = bCommand;
        m_dwIndex += 1;
    }
    else
    {
        CHECK1( FALSE, "Command queue overflow at %d bytes.", m_dwSize );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getCommand
//
//  Description:
//
//      This member is called to retrieve the next command from the queue.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The next command.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CommandQueue::getCommand(
)
{
    //  The command to return.
    Byte bCommand = m_pbBuffer[ 0 ];


    //  Are there any more commands?
    if( m_dwIndex > 0 )
    {
        //  We have one less command.
        m_dwIndex -= 1;

        //  Move all of the commands up a spot.
        for( DWord dwI = 0 ; dwI < m_dwIndex ; dwI += 1 )
        {
            m_pbBuffer[ dwI ] = m_pbBuffer[ dwI + 1 ];
        }
    }
    else
    if( !m_bSustain )
    {
        //  Since there are no commands and we're not sustaining the last
        //  command, just return 0.
        bCommand = 0x00;
    }
        
    return( bCommand );
}
