///////////////////////////////////////////////////////////////////////////////
//
//  File:    ctable.cpp
//
//  Class:   ColourTable
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class contains the definition of a colour table.  A colour table
//      is used to map game colours to screen colours.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "replay.h"
#include "appfile.h"
#include "ctable.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ColourTable
//
//  Description:
//
//      This is the main constructor for a Colour Table object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwNumEntries (input)
//          The number of entries in the table.
//
//      dwValuesPerEntry (input)
//          The number of values for each entry in the table.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ColourTable::ColourTable(
    const KString&  iName,
    const DWord     dwNumEntries,
    const DWord     dwValuesPerEntry /* = 1 */
)
:
    RepBase             ( iName ),
    m_dwTotal           ( dwNumEntries * dwValuesPerEntry ),
    m_dwNumEntries      ( dwNumEntries ),
    m_dwValuesPerEntry  ( dwValuesPerEntry ),
    m_pdwGameValues     ( NULL ),
    m_pbDirty           ( NULL )
{
    //  Check the parameters.
    ASSERT( m_dwNumEntries     > 0 );
    ASSERT( m_dwValuesPerEntry > 0 );

    //  Allocate space for the game values.
    m_pdwGameValues = new DWord[ m_dwTotal ];

    //  Allocate space for the dirty buffer.
    m_pbDirty = new Byte[ m_dwNumEntries ];
    clearDirty( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ColourTable
//
//  Description:
//
//      This is the destructor for a Colour Table object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ColourTable::~ColourTable(
)
{
    //  Free up the game values.
    delete [] m_pdwGameValues;

    //  Free up the dirty buffer.
    delete [] m_pbDirty; 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Colour Table object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ColourTable::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ColourTable" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the colour table to a file for retrieval at a 
//      later date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the colour table to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
ColourTable::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->clearTotal( );
    pSaveFile->write( ( Byte* )m_pdwGameValues, m_dwTotal * sizeof( DWord ) );
    pSaveFile->write( m_pbDirty, m_dwNumEntries * sizeof( Byte ) );

    //  Was it saved successfully?
    return( 
        pSaveFile->total( ) != 
        ( m_dwTotal * sizeof( DWord ) + m_dwNumEntries * sizeof( Byte ) )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the colour table from a file that it was 
//      previously saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the colour table from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
ColourTable::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->read( ( Byte* )m_pdwGameValues, m_dwTotal * sizeof( DWord ) );
    pLoadFile->read( m_pbDirty, m_dwNumEntries * sizeof( Byte ) );

    //  Was it loaded successfully?
    return(
        pLoadFile->total( ) != 
        ( m_dwTotal * sizeof( DWord ) + m_dwNumEntries * sizeof( Byte ) )
    );
}

