///////////////////////////////////////////////////////////////////////////////
//
//  File:       ctable.h
//
//  Class:      ColourTable - Concrete
//  
//  Hierarchy:  ColourTable - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class contains the definition of a colour table.  A colour table
//      is used to map game colours to screen colours.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _ctable_h_
#define _ctable_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"


///////////////////////////////////////////////////////////////////////////////
//  ColourTable Class Definition
///////////////////////////////////////////////////////////////////////////////
class ColourTable : public RepBase
{
    public:

        //  Creation/Deletion.
        ColourTable ( 
            const KString& iName, 
            const DWord dwNumEntries,
            const DWord dwValuesPerEntry = 1
        );
        virtual ~ColourTable ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  Interface.
        DWord  getNumEntries     ( ) const;
        DWord  getValuesPerEntry ( ) const;
        operator DWord*          ( ) const { return( m_pdwGameValues ); }
        DWord* getEntry          ( ) const;
        DWord* getEntry          ( const DWord dwEntry ) const;
        void   setValue          ( const DWord dwIndex, const DWord dwValue );
        void   setValue          ( 
            const DWord dwEntry, const DWord dwEntryValue, const DWord dwValue
        );
        DWord  getValue          ( const DWord dwIndex ) const;
        DWord  getValue          ( 
            const DWord dwEntry, const DWord dwEntryValue
        ) const;
        Byte   getDirty          ( const DWord dwEntry ) const;
        void   setDirty          ( const DWord dwEntry );
        void   clearDirty        ( );

    protected:

        //  Member data.
        DWord   m_dwTotal;
        DWord   m_dwNumEntries;
        DWord   m_dwValuesPerEntry;
        DWord*  m_pdwGameValues;
        Byte*   m_pbDirty;
 
    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Input Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
DWord
ColourTable::getNumEntries(
) const
{
    return( m_dwNumEntries );
}

inline
DWord
ColourTable::getValuesPerEntry(
) const
{
    return( m_dwValuesPerEntry );
}

inline
DWord*
ColourTable::getEntry(
) const
{
    return( m_pdwGameValues );
}

inline
DWord*
ColourTable::getEntry(
    const DWord dwEntry
) const
{
    ASSERT( dwEntry < m_dwNumEntries );
    return( m_pdwGameValues + dwEntry * m_dwValuesPerEntry );
}

inline
void
ColourTable::setValue(
    const DWord dwIndex,
    const DWord dwValue
)
{
    ASSERT( dwIndex < m_dwTotal );
    m_pdwGameValues[ dwIndex ] = dwValue;
    m_pbDirty[ dwIndex / m_dwValuesPerEntry ] = TRUE;
}

inline
void
ColourTable::setValue(
    const DWord dwEntry,
    const DWord dwEntryValue,
    const DWord dwValue
)
{
    ASSERT( dwEntry      < m_dwNumEntries );
    ASSERT( dwEntryValue < m_dwValuesPerEntry );
    m_pdwGameValues[ dwEntry * m_dwValuesPerEntry + dwEntryValue ] = dwValue;
    m_pbDirty[ dwEntry ] = TRUE;
}

inline
DWord
ColourTable::getValue(
    const DWord dwIndex
) const
{
    ASSERT( dwIndex < m_dwTotal );
    return( m_pdwGameValues[ dwIndex ] );
}

inline
DWord
ColourTable::getValue(
    const DWord dwEntry,
    const DWord dwEntryValue
) const
{
    ASSERT( dwEntry      < m_dwNumEntries );
    ASSERT( dwEntryValue < m_dwValuesPerEntry );
    return( m_pdwGameValues[ dwEntry * m_dwValuesPerEntry + dwEntryValue ] );
}

inline
Byte
ColourTable::getDirty(
    const DWord dwEntry
) const
{
    ASSERT( dwEntry < m_dwNumEntries );
    return( m_pbDirty[ dwEntry ] );
}

inline
void
ColourTable::setDirty(
    const DWord dwEntry
)
{
    ASSERT( dwEntry < m_dwNumEntries );
    m_pbDirty[ dwEntry ] = TRUE;
}

inline
void
ColourTable::clearDirty(
)
{
    for( DWord dwI = 0 ; dwI < m_dwNumEntries ; dwI += 1 )
    {
        m_pbDirty[ dwI ] = FALSE;
    }
}
    
#endif
