///////////////////////////////////////////////////////////////////////////////
//
//  File:    dip.cpp
//
//  Class:   DipSwitch
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to keep track of a dip switch for the game.
//      A dip switch is a group of 8 bits.  Each bit (switch) contributes to a 
//      setting for the game.  The switches can be set by the user.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <stdarg.h>

//  Application Headers.
#include "reptypes.h"
#include "dip.h"
#include "setting.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: DipSwitch
//
//  Description:
//
//      This is the main constructor for a dip switch object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pbLocation (input)
//          The pointer to the location the dip switch affects.
//
//      bReverse (input)   
//          Indicates whether the location bits are reversed.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DipSwitch::DipSwitch(
    const KString&        iName,
    Byte*                 pbLocation,
    Byte                  bReverse
)
:
    RepBase           ( iName ),
    m_pbLocation      ( pbLocation ),
    m_bReverse        ( bReverse ),
    m_settingList     ( 8 )
{
    //  Check the parameters.
    ASSERT( m_pbLocation != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DipSwitch
//
//  Description:
//
//      This is the destructor for a dip switch object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DipSwitch::~DipSwitch(
)
{
    //  Clean up the settings.
    for( DWord dwI = 0 ; dwI < m_settingList.entries( ) ; dwI += 1 )
    {
        delete m_settingList[ dwI ];
    }
    m_settingList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the dip switch object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DipSwitch::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DipSwitch" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addSetting
//
//  Description:
//
//      This member is used to add a setting to the dip switch.  A setting
//      indicates what a certain pattern of bits means to the game.  For
//      example, a certain bit may indicate that the player gets 3 lives if
//      it is off and 5 lives if it is on. 
//
//  Parameters:
//
//      settingName (input)
//          The name of the setting.
//
//      bMask (input)
//          The mask of bits involved in the setting.
//
//      ... (input)
//          The text descriptions of each possible setting.  This list must
//          must be at least 2^^#_bits_set_in_bMask.  Any descriptions after
//          this number will be ignored.  If the list contains less than
//          this amount problems will be encountered.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DipSwitch::addSetting( 
    const KString& settingName, 
    const Byte     bMask, 
    ...
)
{
    //  A pointer to the current setting.
    Setting* pSetting;

    //  The variable argument list.
    va_list ap;

    //  The number of descriptions expected.
    DWord dwNumDescriptions = 1;


    //  Count the number of descriptions expected.
    for( DWord dwI = 0 ; dwI < 8 ; dwI += 1 )
    {
        //  If the bit is set then increase the number of descriptions.
        if( bMask & (1 << dwI ) )
        {
            dwNumDescriptions <<= 1;
        } 
    }

    //  Make sure at least one bit was specified (resulting in 
    //  dwNumDescriptions being at least 2.
    ASSERT( dwNumDescriptions >= 2 );

    //  The variable argument list begins after the mask.
    va_start( ap, bMask );

    //  Add the setting to the list.
    pSetting = m_settingList.add( new Setting( settingName, this, bMask ) );

    //  Now add the descriptions for the settings.
    for( DWord dwI = 0 ; dwI < dwNumDescriptions ; dwI += 1 )
    {
        pSetting->addDescription( va_arg( ap, char* ) );
    }

    //  All done with the list.
    va_end( ap );
}
