///////////////////////////////////////////////////////////////////////////////
//
//  File:       dip.h
//
//  Class:      DipSwitch - Concrete
//  
//  Hierarchy:  DipSwitch - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class is used to keep track of a dip switch for the game.
//      A dip switch is a group of 8 bits.  Each bit (switch) contributes to a 
//      setting for the game.  The switches can be set by the user.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _dip_h_
#define _dip_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"

//  Forward Class Declarations.
class Setting;


///////////////////////////////////////////////////////////////////////////////
//  DipSwitch Class Definition
///////////////////////////////////////////////////////////////////////////////
class DipSwitch : public RepBase
{
    public:

        //  Creation/Deletion.
        DipSwitch ( const KString& iName, Byte* pbLocation, Byte bReverse );
        virtual ~DipSwitch ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        Byte                   getValue       ( ) const;
        void                   setValue       ( const Byte bValue );
        Byte*                  getLocation    ( ) const;
        DWord                  getNumSettings ( ) const;
        Setting*               getSetting     ( DWord dwIndex ) const;
        void                   addSetting     (
            const KString& settingName, const Byte bMask, ...
        );
        Byte                   isReversed     ( ) const;

    protected:

        //  Member Data.
        Byte*                  m_pbLocation;
        Byte                   m_bReverse;
        KPtrList<Setting>      m_settingList;

    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Input Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte
DipSwitch::getValue(
) const
{
    return( *m_pbLocation );
}

inline
void
DipSwitch::setValue(
    const Byte bValue
)
{
    *m_pbLocation = bValue;
}

inline
Byte*
DipSwitch::getLocation(
) const
{
    return( m_pbLocation );
}

inline
DWord
DipSwitch::getNumSettings(
) const
{
    return( m_settingList.entries( ) );
}

inline
Setting*
DipSwitch::getSetting(
    DWord dwIndex
) const
{
    ASSERT( dwIndex < m_settingList.entries( ) );
    return( m_settingList[ dwIndex ] );
}

inline
Byte
DipSwitch::isReversed(
) const
{
    return( m_bReverse );
}

#endif
