///////////////////////////////////////////////////////////////////////////////
//
//  File:       game.h
//
//  Class:      Game - Abstract
//  
//  Hierarchy:  Game - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      All games derive from this abstract base class.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _game_h_
#define _game_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <string.h>

//  Application Headers.
#include "repbase.h"
#include "kplist.h"
#include "kpslist.h"
#include "replay.h"
#include "canvas.h"
#include "config.h"
#include "cpu.h"


//  Forward class declarations.
class GameInfo;
class Buffer;
class AddrSpace;
class ReadHandler;
class WriteHandler;
class Input;
class CtrlMap;
class DipSwitch;
class Bitmap;
class GfxSet;
class ColourTable;
class Clipping;
class Sample;
class SoundDevice;
class Scheduler;
class Screen;
class HiScore;
class AppFile;
class Registry;

///////////////////////////////////////////////////////////////////////////////
//  Game Class Definition
///////////////////////////////////////////////////////////////////////////////
class Game : public RepBase
{
    public:

        //  Creation/Deletion.
                 Game ( const KString& iName );
        virtual ~Game ( );

        //  Persistence.
        virtual Byte save ( );
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( );
        virtual Byte load ( AppFile* pLoadFile );
            
        //  Interface.
        virtual Byte  startUp                ( ); 
        virtual void  shutDown               ( );
        virtual void  reset                  ( );
        virtual void  run                    ( );
        virtual void  updateDisplay          ( 
            Bitmap* pScreen, Bitmap* pTempScreen 
        );
        virtual void  updateSound            ( );
        virtual void  hiScorePostLoad        ( const Byte bExisting );
        virtual void  hiScorePreSave         ( );
        
        //  Access.
        void                       setGameInfo         ( GameInfo* pGameInfo);
        GameInfo*                  getGameInfo         ( ) const;
        Buffer*                    getColourPROM       ( ) const;
        Buffer*                    getSoundPROM        ( ) const;
        Screen*                    getScreen           ( ) const;
        HiScore*                   getHiScore          ( ) const;
        Scheduler*                 getScheduler        ( ) const;
        const KPtrList<CPU>&       getCPUList          ( ) const;
        const KPtrList<CtrlMap>&   getCtrlMapList      ( ) const;
        const KPtrList<DipSwitch>& getDipSwitchList    ( ) const;
        const DWord                getMaxPlayers       ( ) const;
        Registry*                  getSettingsRegistry ( ) const;
        Registry*                  getHiScoreRegistry  ( ) const;
        void                       resetControls       ( );

        //  Static Handler Utility.
        static Byte       s_isRAMHandler           ( void* pfnHandler );
        static Byte       s_isROMHandler           ( void* pfnHandler );

        //  Static member data used for easy/fast access.
        static Game* sm_pGame;

    protected:

        //  Specific start-up functions.
        virtual void startUpStart    ( );
        virtual void startUpSpace    ( );
        virtual void startUpROMs     ( );
        virtual void startUpCPU      ( );
        virtual void startUpInput    ( );
        virtual void startUpGraphics ( );
        virtual void startUpColour   ( );
        virtual void startUpSound    ( );
        virtual void startUpMemMap   ( );
        virtual void startUpOther    ( );
        virtual void startUpEnd      ( );

        //  Utility members for games.
        void setMaxPlayers      ( const DWord dwMaxPlayers );
        CPU* createCPU          ( 
            const KString&        iName, 
            const CPU::CPUType    eCPUType, 
            AddrSpace*            pSpace,
            const CPU::CPUPurpose ePurpose      = CPU::GAME,
            const Byte            bCONSERVATIVE = FALSE
        );
        Bitmap* createBitmap    (
            const KString& iName,
            const DWord    dwWidth,
            const DWord    dwHeight,
            const Byte     bForScreen = FALSE
        );

        //  Registry access.
        const DWord settingsGetValueSize ( const KString& name ); 
        const DWord settingsGetValue     (
            const KString& name,
            Byte*          pbResult,
            DWord          dwMaxLength = 1
        );
        const DWord settingsGetValue     ( 
            const KString& name, 
            int32*         pnResult 
        );
        const DWord settingsGetValue     ( 
            const KString& name, 
            KString*       pResult 
        );
        void settingsSetValue            (
            const KString& name,
            Byte*          pbValue,
            DWord          dwLength = 1
        );
        void settingsSetValue            (
            const KString& name,
            int32          nValue
        ); 
        void settingsSetValue            (
            const KString& name,
            KString&       value
        ); 


        //  Default memory read handlers.
        static DWord s_readNo            ( 
            DWord dwAddress, ReadHandler* pHandler 
        );
        static DWord s_readRAM           ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readSharedRAM     ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readBuffer        ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readROM           ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readInput         ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readInputs        ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readCommand       ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_read8910          ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_read3812Read      ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_read3812Status    ( 
            DWord dwAddress, ReadHandler* pHandler
        );
        static DWord s_readWatchDog      ( 
            DWord dwAddress, ReadHandler* pHandler
        );

        //  Default memory write handlers.
        static void s_writeNo             ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeRAM            ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeSharedRAM      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeBuffer         ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeROM            ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeDirtyByte      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeDirtyWord      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeHiScore        ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeScreenRotate   ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeScreenFlipX    ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeScreenFlipY    ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeCommand        ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeCPUEnable      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeIntEnable      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_write8910Write      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_write8910Control    ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_write3812Write      ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_write3812Control    ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );
        static void s_writeWatchDog       ( 
            DWord dwAddress, Byte bValue, WriteHandler* pHandler
        );

        //  The default interrupt functions.
        static int32 s_irqInterrupt     ( CPU* pCPU, Game* pGame );
        static int32 s_nmiInterrupt     ( CPU* pCPU, Game* pGame );
        static int32 s_noInterrupt      ( CPU* pCPU, Game* pGame );
        static int32 s_vectorInterrupt  ( CPU* pCPU, Game* pGame );

        //  Member data.
        GameInfo*             m_pGameInfo;
        Replay*               m_pReplay;
        int32                 m_nCPUInterleave;
        Byte                  m_bScreenFlippedX;
        Byte                  m_bScreenFlippedY;
        DWord                 m_dwMaxPlayers;
        Registry*             m_pSettingsRegistry;
        Registry*             m_pHiScoreRegistry;
        KPtrList<Bitmap>      m_bitmapList;
        KPtrList<Input>       m_inputList;
        KPtrList<CtrlMap>     m_ctrlMapList;
        KPtrList<DipSwitch>   m_dipSwitchList;
        KPtrList<AddrSpace>   m_spaceList;
        KPtrList<CPU>         m_cpuList;
        KPtrList<GfxSet>      m_gfxSetList;
        KPtrList<Buffer>      m_bufferList;
        KPtrList<Buffer>      m_tempBufferList;
        KPtrList<Buffer>      m_dirtyList;
        KPtrList<ColourTable> m_colourTableList;
        KPtrList<Clipping>    m_clippingList;
        KPtrList<Sample>      m_sampleList;
        KPtrList<SoundDevice> m_soundDeviceList;
        Scheduler*            m_pScheduler;
        Screen*               m_pScreen;
        Buffer*               m_pColourPROM;
        Buffer*               m_pSoundPROM;
        HiScore*              m_pHiScore;
        Byte                  m_bSoundDisabled;
        DWord                 m_dwStartStep;
        int32                 m_nWatchDog;

    private:

        //  Utility.
        void setControls         ( );
        void setDipSwitches      ( );
};


///////////////////////////////////////////////////////////////////////////////
//  Game Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
Game::setGameInfo(
    GameInfo* pGameInfo
)
{
    ASSERT( pGameInfo != NULL );
    
    m_pGameInfo = pGameInfo;
}

inline
GameInfo*
Game::getGameInfo(
) const
{
    return( m_pGameInfo );
}

inline
void
Game::setMaxPlayers(
    const DWord dwMaxPlayers
)
{
    ASSERT( dwMaxPlayers >= 1 );
    
    m_dwMaxPlayers = dwMaxPlayers;
}


inline
Bitmap*
Game::createBitmap(
    const KString& iName,
    DWord          dwWidth,
    DWord          dwHeight,
    Byte           bForScreen = FALSE
) 
{
    return( 
        Replay::s_instance( ).getCanvas( )->createBitmap( 
            iName, dwWidth, dwHeight, bForScreen
        )
    );
}

inline
Buffer*
Game::getColourPROM(
) const
{
    return( m_pColourPROM );
}

inline
Buffer*
Game::getSoundPROM(
) const
{
    return( m_pSoundPROM );
}

inline
Screen*
Game::getScreen(
) const
{
    return( m_pScreen );
}

inline
HiScore*
Game::getHiScore(
) const
{
    return( m_pHiScore );
}

inline
Scheduler*
Game::getScheduler(
) const
{
    return( m_pScheduler );
}

inline
const
KPtrList<CPU>&
Game::getCPUList(
) const
{
    return( m_cpuList );
}

inline
const
KPtrList<CtrlMap>&
Game::getCtrlMapList(
) const
{
    return( m_ctrlMapList );
}

inline
const
KPtrList<DipSwitch>&
Game::getDipSwitchList(
) const
{
    return( m_dipSwitchList );
}

inline
const
DWord
Game::getMaxPlayers(
) const
{
    return( m_dwMaxPlayers );
}

inline
Registry*
Game::getSettingsRegistry(
) const
{
    return( m_pSettingsRegistry );
}

inline
Registry*
Game::getHiScoreRegistry(
) const
{
    return( m_pHiScoreRegistry );
}


///////////////////////////////////////////////////////////////////////////////
//  Game Macros.
///////////////////////////////////////////////////////////////////////////////
//  The following macros are used as a convenience in the driver files to
//  provide information about the game included in the driver.
#define INFO_ID              "i"
#define INFO_NAME            "n"
#define INFO_VERSION         "v"
#define INFO_CONTRIBUTORS    "c"
#define INFO_REQD_FILES      "r"
#define INFO_END             "e"

#define GAME_INFO                                                             \
    static const char* ppGameInfo[ ] =                                        \
    {
#define ASSIGN_GAME_ID                     NULL,INFO_ID
#define ASSIGN_GAME_NAME                   NULL,INFO_NAME
#define ASSIGN_GAME_VERSION                NULL,INFO_VERSION
#define ASSIGN_GAME_CONTRIBUTORS           NULL,INFO_CONTRIBUTORS
#define ASSIGN_GAME_REQD_FILES             NULL,INFO_REQD_FILES
#define FOR(ID,CLASS)                      NULL,INFO_END                      \
    };                                                                        \
    extern "C" const char** ID ## _info( )   { return( ppGameInfo ); }        \
    extern "C" Game*        ID ## _create( ) { return( new CLASS( #ID ) ); }
    
//  The following provide quick access to the game information.
#define GAME_ID()    m_pGameInfo->getGameId( )
#define GAME_NAME()  m_pGameInfo->getGameName( )

#endif
