///////////////////////////////////////////////////////////////////////////////
//
//  File:    gfxset.cpp
//
//  Class:   GfxSet
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class contains a set of graphics (sprites/chars/etc. for a game).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <stdarg.h>

//  Application Headers.
#include "reptypes.h"
#include "gfxset.h"
#include "replay.h"
#include "bitmap.h"



//  Static data.
const Byte  GfxSet::MAX_BPP        = 8;
const DWord GfxSet::MAX_GFX_WIDTH  = 64;
const DWord GfxSet::MAX_GFX_HEIGHT = 64;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: GfxSet
//
//  Description:
//
//      This is the main constructor for a Graphics Set object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GfxSet::GfxSet(
    const KString&  iName
)
:
    RepBase           ( iName ),
    m_dwNumber        ( 256 ),
    m_dwWidth         ( 8 ),
    m_dwHeight        ( 8 ),
    m_bBPP            ( 2 ),
    m_pdwBPStart      ( NULL ),
    m_pdwXStart       ( NULL ),
    m_pdwYStart       ( NULL ),
    m_dwIncrement     ( 64 ),
    m_bitmapList      ( 64 )
{
    //  Initialize the default starting locations for the bit planes.
    m_pdwBPStart = new DWord[ MAX_BPP ];
    for( int32 nI = 0 ; nI < m_bBPP ; nI += 1 )
    {
        m_pdwBPStart[ nI ] = nI * 16384;
    }

    //  Initialize the starting X & Y values for the pixels.
    m_pdwXStart = new DWord[ MAX_GFX_WIDTH ];
    for( DWord dwX = 0 ; dwX < m_dwWidth ; dwX += 1 )
    {
        m_pdwXStart[ dwX ] = dwX;
    }
    m_pdwYStart = new DWord[ MAX_GFX_HEIGHT ];
    for( DWord dwY = 0 ; dwY < m_dwHeight ; dwY += 1 )
    {
        m_pdwYStart[ dwY ] = dwY * m_dwWidth;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GfxSet
//
//  Description:
//
//      This is the destructor for a Graphics Set object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GfxSet::~GfxSet(
)
{
    //  Free allocated space.
    delete [] m_pdwBPStart;
    delete [] m_pdwXStart;
    delete [] m_pdwYStart;

    //  Clean up the bitmaps.
    for( uint32 nIdx = 0 ; nIdx < m_bitmapList.entries( ) ; nIdx += 1 )
    {
        delete m_bitmapList[ nIdx ];
    }
    m_bitmapList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Graphics Set object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GfxSet::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GfxSet" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setBitPlanes
//
//  Description:
//
//      This member is used to set the starting locations (in bits) of the
//      bitplanes for the graphics set.
//
//  Parameters:
//
//      dwPlane0 (input)
//          The start of the first bit plane.
//
//      ... (input)
//          The remainder of the starting values.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GfxSet::setBitPlanes(
    DWord dwPlane0,
    ...
)
{
    //  The variable argument list.
    va_list ap;


    //  Assign the value of the first plane.
    m_pdwBPStart[ 0 ] = dwPlane0; 

    //  The variable argument list begins after the first plane.
    va_start( ap, dwPlane0 );

    //  Now extract each of the remaining start values.
    for( Byte bI = 1 ; bI < m_bBPP ; bI += 1 )
    {
        m_pdwBPStart[ bI ] = va_arg( ap, DWord );
    }

    //  All done with the list.
    va_end( ap );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setBitXBits
//
//  Description:
//
//      This member is used to set the starting locations (in bits) of the
//      pixels in the X direction.
//
//  Parameters:
//
//      dwXStart0 (input)
//          The start of the first pixel in the X direction.
//
//      ... (input)
//          The remainder of the starting values.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GfxSet::setXBits(
    DWord dwXStart0,
    ...
)
{
    //  The variable argument list.
    va_list ap;


    //  Assign the value of the first pixel.
    m_pdwXStart[ 0 ] = dwXStart0; 

    //  The variable argument list begins after the first pixel start.
    va_start( ap, dwXStart0 );

    //  Now extract each of the remaining start values.
    for( DWord dwI = 1 ; dwI < m_dwWidth ; dwI += 1 )
    {
        m_pdwXStart[ dwI ] = va_arg( ap, DWord );
    }

    //  All done with the list.
    va_end( ap );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setBitYBits
//
//  Description:
//
//      This member is used to set the starting locations (in bits) of the
//      pixels in the Y direction.
//
//  Parameters:
//
//      dwYStart0 (input)
//          The start of the first pixel in the Y direction.
//
//      ... (input)
//          The remainder of the starting values.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GfxSet::setYBits(
    DWord dwYStart0,
    ...
)
{
    //  The variable argument list.
    va_list ap;


    //  Assign the value of the first pixel.
    m_pdwYStart[ 0 ] = dwYStart0; 

    //  The variable argument list begins after the first pixel start.
    va_start( ap, dwYStart0 );

    //  Now extract each of the remaining start values.
    for( DWord dwI = 1 ; dwI < m_dwHeight ; dwI += 1 )
    {
        m_pdwYStart[ dwI ] = va_arg( ap, DWord );
    }

    //  All done with the list.
    va_end( ap );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: decode
//
//  Description:
//
//      This member is used to create the graphics set from the bit buffer
//      specified.
//
//  Parameters:
//
//      pbBits (input)
//          A pointer to the buffer of bits.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GfxSet::decode(
    Byte* pbBits
)
{
    //  Adjust the list size to match the specified size.
    for( DWord dwI = m_bitmapList.entries( ) ; dwI > m_dwNumber ; dwI -= 1 )
    {
        delete m_bitmapList[ dwI - 1 ];
    }
    m_bitmapList.shrink( m_dwNumber );

    //  Create new bitmaps as required.
    for( DWord dwI = m_bitmapList.entries( ) ; dwI <= m_dwNumber ; dwI += 1 )
    {
        m_bitmapList.add( 
            Replay::s_instance( ).getCanvas( )->createBitmap( 
                "x", m_dwWidth, m_dwHeight 
            ) 
        );
    }

    //  If the bit buffer is empty then just return at this point.
    if( pbBits == NULL )
    {
        return;
    }

    //  Decode each graphic.
    for( DWord dwGraphic = 0 ; dwGraphic < m_dwNumber ; ++dwGraphic )
    {
        decode( pbBits, dwGraphic );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: decode
//
//  Description:
//
//      This member is used to create a single graphic within the graphics
//      set.
//
//  Parameters:
//
//      pbBits (input)
//          A pointer to the buffer of bits.
//
//      dwGraphic (input)
//          The character to decode.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GfxSet::decode(
    Byte* pbBits,
    DWord dwGraphic
)
{
    ASSERT( pbBits != NULL );
    ASSERT( dwGraphic < m_dwNumber );

    //  The current bit plane.
    DWord bCurPlane;

    //  The current pixel row and column in the destination buffer.
    DWord dwCurRow;
    DWord dwCurCol;

    //  The source bit of interest.
    DWord dwSourceBit;

    //  The value of the current pixel in the bitmap.
    DWord dwValue;

    //  A pointer to the bitmap we are decoding.
    Bitmap& rBitmap = *( m_bitmapList[ dwGraphic ] );

    //  The offset.
    DWord dwOffset = dwGraphic * m_dwIncrement;


    //  Loop through each row and column of the graphic.
    for( dwCurRow = 0 ; dwCurRow < m_dwHeight ; ++dwCurRow )
    {
        for( dwCurCol = 0 ; dwCurCol < m_dwWidth ; ++dwCurCol )
        {
            //  Initialize the value.
            dwValue = 0;

            //  Loop through each bit plane for the character.
            for( bCurPlane = 0 ; bCurPlane < m_bBPP ; ++bCurPlane )
            {
                //  Shift the value to make room for the new plane.
                dwValue <<= 1;

                //  Calculate the source bit.
                dwSourceBit = 
                    dwOffset + 
                    m_pdwBPStart[ bCurPlane ] +
                    m_pdwYStart[ dwCurRow ] + 
                    m_pdwXStart[ dwCurCol ];

                //  Add the bit from the current bit plane to the 
                //  destination byte.
                dwValue |= 
                    ( 
                        pbBits[ dwSourceBit >> 3 ] >> 
                        ( 7 - ( dwSourceBit & 7 ) ) 
                    ) & 0x01;

                //  Set the pixel value in the bitmap.
                rBitmap.setPixel( dwCurCol, dwCurRow, dwValue );
            }
        }
    }
}
