///////////////////////////////////////////////////////////////////////////////
//
//  File:    input.cpp
//
//  Class:   Input
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a hardware input for a game.  It holds a single
//      byte value that can be read during the course of execution.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "input.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Input
//
//  Description:
//
//      This is the main constructor for a hardware input object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      bInitial (input)
//          The initial value of the input.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Input::Input(
    const KString&  iName,
    const Byte      bInitial /* =0x00 */
)
:
    RepBase           ( iName ),
    m_bInitial        ( bInitial ),
    m_bValue          ( bInitial )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Input
//
//  Description:
//
//      This is the destructor for a hardware input object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Input::~Input(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the hardware input object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Input::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Input" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the input to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the input to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Input::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );
 
    //  Save the member data that keeps the state.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bInitial, sizeof( m_bInitial ) );
    pSaveFile->write( &m_bValue,   sizeof( m_bValue ) );

    //  Was it saved successfully?
    return( 
        pSaveFile->total( ) != ( sizeof( m_bInitial ) + sizeof( m_bValue ) ) 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the input from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the input from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Input::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );
 
    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bInitial, sizeof( m_bInitial ) );
    pLoadFile->read( &m_bValue,   sizeof( m_bValue ) );

    //  Was it loaded successfully?
    return(
        pLoadFile->total( ) != ( sizeof( m_bInitial ) + sizeof( m_bValue ) )
    );
}
