///////////////////////////////////////////////////////////////////////////////
//
//  File:    schedule.cpp
//
//  Class:   Scheduler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The scheduler is a class that is used for controlling the execution
//      of the CPUs.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "schedule.h"
#include "screen.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: Scheduler
//
//  Description:
//
//      This is the main constructor for a scheduler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Scheduler::Scheduler(
    const KString& iName
)
:
    RepBase             ( iName ),
    m_sliceList         ( Scheduler::s_compareSlices, 32 ),
    m_pSlice            ( NULL ),
    m_pScreen           ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Scheduler
//
//  Description:
//
//      This is the destructor for the Scheduler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Scheduler::~Scheduler(
)
{
    //  Clean up the CPU slice list.
    for( DWord dwI = 0 ; dwI < m_sliceList.entries( ) ; dwI += 1 )
    {
        delete m_sliceList[ dwI ];
    }
    m_sliceList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the scheduler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Scheduler::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Scheduler" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateCPUs
//
//  Description:
//
//      This member is called when a game is to update it's CPU state (i.e.
//      run through one frame's worth of instructions).
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Scheduler::updateCPUs(
)
{
    //  Loop through the schedule.
    for( DWord dwI = 0 ; dwI < m_sliceList.entries( ) ; dwI += 1 )
    {
        m_pSlice = m_sliceList[ dwI ];

        //  Set the VBlank status.
        ASSERT( m_pScreen );
        m_pScreen->switchVBlank( m_pSlice->m_bInVBlank );

        //  Only run the CPU if it is enabled.
        if( m_pSlice->m_pCPU->isEnabled( ) )
        {
            //  Switch to the new CPU.
            m_pSlice->m_pCPU->activate( );
 
            //  Run the time slice on the CPU.
            m_pSlice->m_pCPU->run( m_pSlice->m_nCycles );

            //  If the CPU is to be interrupted then do it.
            if( m_pSlice->m_bInterrupt )
            {
                m_pSlice->m_pCPU->interrupt( );
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: scheduleCPUs
//
//  Description:
//
//      For each video frame, the CPUs involved in emulating the game
//      will be executed for varying amounts of time returning to the
//      game loop for reasons such as CPU swapping, time for an interrupt,
//      and end of frame.
//
//      To simplify calculations during the loop, a schedule for a video
//      frame is built and the game loop can quickly execute the frame
//      by stepping through the schedule.
//
//  Parameters:
//
//      rCPUList (input)
//          The list of CPUs to schedule.
//
//      nCPUInterleave (input)
//          The number of times per frame that we swap CPU execution.
//
//      pScreen (input)
//          A pointer to the screen object which will contain information
//          used to synchronize the CPUs with the video retrace.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Scheduler::scheduleCPUs( 
    KPtrList<CPU>& rCPUList,
    const int      nCPUInterleave,
    Screen*        pScreen
)
{
    //  Currently we only allow CPUs to be scheduled once.
    ASSERT( m_sliceList.entries( ) == 0 );

    //  Verify the information passed in.
    ASSERT( nCPUInterleave > 0 );
    ASSERT( pScreen );
    ASSERT( pScreen->getFPS( ) > 0 );

    //  Cache the screen since we'll use it during CPU updating.
    m_pScreen = pScreen;

    //  We need the VBlank duration in seconds.
    double fVBlankInSeconds = ( double )pScreen->getVBlankLength( ) / 1000000;

    //  We need a running cycle count total for each CPU.
    int32 anRunningTotal[ 8 ];
    ASSERT( rCPUList.entries( ) <= 8 );
    for( DWord dwI = 0 ; dwI < rCPUList.entries( ) ; dwI += 1 )
    {
        anRunningTotal[ dwI ] = 0;
    }


    //  Create a list of chunks.  This will be populated with the complete
    //  list of frame divisions required to service interrupts and interleave.
    DWord dwNumChunks = nCPUInterleave;
    for( DWord dwCPU = 0 ; dwCPU < rCPUList.entries( ) ; dwCPU += 1 )
    {
        dwNumChunks += rCPUList[ dwCPU ]->getIntsPerFrame( );
    }
    double* pfChunkList = new double[ dwNumChunks ];


    //  Add the chunks for the CPU interleave.
    DWord dwChunkIndex = 0;
    for( int32 nI = 1 ; nI <= nCPUInterleave ; nI += 1 )
    {
        s_addChunkToList( 
            pfChunkList, 
            dwChunkIndex, 
            ( double )nI / nCPUInterleave
        );
    }

    //  Add the chunks for the CPU interrupts.
    for( DWord dwCPU = 0 ; dwCPU < rCPUList.entries( ) ; dwCPU += 1 )
    {
        CPU* pCPU = rCPUList[ dwCPU ];
        for( int32 nI = 1 ; nI <= pCPU->getIntsPerFrame( ) ; nI += 1 )
        {
            s_addChunkToList( 
                pfChunkList, 
                dwChunkIndex, 
                ( double )nI / pCPU->getIntsPerFrame( )
            );
        }
    }

    //  Resize the list to the size without duplicates.
    dwNumChunks = dwChunkIndex;

    //  Sort the chunk list.
    qsort( 
        ( void* )pfChunkList, dwNumChunks, sizeof( double ), s_compareChunks 
    );


    //  Now we run through the list of chunks scheduling slices for each CPU.
    for( DWord dwChunk = 0 ; dwChunk < dwNumChunks ; dwChunk += 1 )
    {
        double fChunk = pfChunkList[ dwChunk ];

        for( DWord dwCPU = 0 ; dwCPU < rCPUList.entries( ) ; dwCPU += 1 )
        {
            CPU* pCPU = rCPUList[ dwCPU ];

            //  The length of the full frame in cycles is the clock speed
            //  (i.e. # cycles per second) divided by the number of frames
            //  per second.
            DWord dwCyclesPerFrame = 
                pCPU->getClockSpeed( ) / pScreen->getFPS( );

            //  The length of the vblank in cycles is equal to the clock
            //  speed (i.e. # cycles per second) multiplied by the 
            //  duration of the vblank (in usec) divided by the number
            //  of usec in a second.
            DWord dwCyclesPerVBlank = 
                ( DWord )( pCPU->getClockSpeed( ) * fVBlankInSeconds );

            //  The start of the slice.
            int32 nStart = anRunningTotal[ dwCPU ] - dwCyclesPerVBlank;
            
            //  The end of the slice.
            int32 nEnd = 
                ( int32 )( dwCyclesPerFrame * fChunk ) - 1 - dwCyclesPerVBlank;

            //  Does the chunk end in an interrupt for this CPU?  This is
            //  a horrible brute force method but it's a one shot deal and
            //  the loop counter is relatively small so what the hell.
            Byte bInterrupt = FALSE;
            for( int32 nI = 1 ; nI <= pCPU->getIntsPerFrame( ) ; nI += 1 )
            {
                double fPortion = ( double )nI / pCPU->getIntsPerFrame( );
                if( CLOSE_ENOUGH( fPortion, fChunk ) )
                {
                    bInterrupt = TRUE;
                    break;
                }
            }

            //  Update the running total.
            anRunningTotal[ dwCPU ] += ( nEnd - nStart + 1 );

            //  To accurately emulate the vertical blanking of the original
            //  game we want the update of the screen to occur a period of 
            //  time after the vblank interrupt of the game.  Since the 
            //  screen update occurs immediately after one run of the schedule
            //  and since we are calculating even divisions, the vblank
            //  interrupt will fall exactly on the end of the schedule.
            //
            //  Therefore, to introduce the appropriate delay after the
            //  vblank interrupt, we subtract the vblank duration from the
            //  start/end we would normally calculate.  If the start falls
            //  off the start of the schedule (i.e. less than 0) then we
            //  tack that portion on the end of the schedule.
            //
            //  In effect, we are rotating the schedule in a circular fashion.
            //  e.g.
            //     Iabcdef----------------VBI update
            //  would become
            //     Ief---------------VBIabcdI update
            //  where the a,b,c,d slices have been rotated off the left and
            //  tacked onto the right.
            //
            //  There are three cases for the slice:
            //      1) Shifting it doesn't cause it to fall off the left side.
            //      2) Shifting causes it to partially fall off the left side.
            //      3) Shifting causes it to fully fall off the left side.
            // 
            //  1) Not off.
            if( nStart >= 0 )
            {
                //  Create a single slice to add to the list of slices.
                m_sliceList.add( 
                    new Slice( 
                        dwCPU,
                        rCPUList[ dwCPU ], 
                        nStart,
                        nEnd,
                        dwCyclesPerFrame,
                        bInterrupt, 
                        FALSE
                    ) 
                );
            }
            else
            //  2) Partially off.
            if( nEnd >= 0 )
            {
                //  Add the portion that hasn't fallen off.
                m_sliceList.add( 
                    new Slice( 
                        dwCPU,
                        rCPUList[ dwCPU ], 
                        0,
                        nEnd,
                        dwCyclesPerFrame,
                        bInterrupt, 
                        FALSE
                    ) 
                );

                //  Adjust the start/end.
                nStart += dwCyclesPerFrame;

                //  Add the portion that fell off to the other end.
                m_sliceList.add( 
                    new Slice( 
                        dwCPU,
                        rCPUList[ dwCPU ], 
                        nStart,
                        dwCyclesPerFrame - 1,
                        dwCyclesPerFrame,
                        FALSE,
                        TRUE
                    ) 
                );
            }
            //  3) All off.
            else
            {
                //  Adjust the start and end.
                nStart += dwCyclesPerFrame;
                nEnd   += dwCyclesPerFrame;

                //  Add the slice to the other end.
                m_sliceList.add( 
                    new Slice( 
                        dwCPU,
                        rCPUList[ dwCPU ], 
                        nStart,
                        nEnd,
                        dwCyclesPerFrame,
                        bInterrupt,
                        TRUE
                    ) 
                );
            }
        }
    }

    
    //  At this point the schedule has completed.  Following are some
    //  sanity checks that can be taken out once the algorithm has proven
    //  itself in battle for a while.
    //
    //  First check that all expected interrupts exist.
    for( DWord dwI = 0 ; dwI < rCPUList.entries( ) ; dwI += 1 )
    {
        DWord dwNumInts = 0;
        for( DWord dwJ = 0 ; dwJ < m_sliceList.entries( ) ; dwJ += 1 )
        {
            if( 
                ( m_sliceList[ dwJ ]->m_nCPU == ( int32 )dwI ) &&
                ( m_sliceList[ dwJ ]->m_bInterrupt )
            )
            {
                dwNumInts += 1;
            }
        }
        ASSERT( dwNumInts = rCPUList[ dwI ]->getIntsPerFrame( ) );
    }
    //  Now check that the total cycles is correct.
    for( DWord dwI = 0 ; dwI < rCPUList.entries( ) ; dwI += 1 )
    {
        DWord dwCycles = 0;
        for( DWord dwJ = 0 ; dwJ < m_sliceList.entries( ) ; dwJ += 1 )
        {
            if( m_sliceList[ dwJ ]->m_nCPU == ( int32 )dwI )
            {
                dwCycles += m_sliceList[ dwJ ]->m_nCycles;
            }
        }
        ASSERT( 
            dwCycles == rCPUList[ dwI ]->getClockSpeed( ) / pScreen->getFPS( ) 
        );
    }
    //  Now check that the vblank period is correct.
    for( DWord dwI = 0 ; dwI < rCPUList.entries( ) ; dwI += 1 )
    {
        DWord dwVBlankCycles = 0;
        for( DWord dwJ = 0 ; dwJ < m_sliceList.entries( ) ; dwJ += 1 )
        {
            if( 
                ( m_sliceList[ dwJ ]->m_nCPU == ( int32 )dwI ) &&
                ( m_sliceList[ dwJ ]->m_bInVBlank )
            )
            {
                dwVBlankCycles += m_sliceList[ dwJ ]->m_nCycles;
            }
        }
        ASSERT( 
            dwVBlankCycles == 
            ( DWord )( rCPUList[ dwI ]->getClockSpeed( ) * fVBlankInSeconds )
        );
    }
    //  Now check that the CPUs alternate.
    int32 nCPU = 0;
    for( DWord dwI = 0 ; dwI < m_sliceList.entries( ) ; dwI += 1 )
    {
        ASSERT( nCPU == m_sliceList[ dwI ]->m_nCPU );
        nCPU += 1;
        nCPU %= rCPUList.entries( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_addChunkToList
//
//  Description:
//
//      This is a simple utility function that will add an item to
//      the specified list of chunks if it doesn't already exist.
//
//  Parameters:
//
//      pfChunkList (input/output)
//          The existing list of chunks.
//
//      rdwChunkSize (input/output)
//          The size of the chunk list.
//
//      fChunk (input)
//          The chunk (i.e. percentage of frame) to add.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
Scheduler::s_addChunkToList( 
    double*      pfChunkList, 
    DWord&       rdwChunkSize, 
    const double fChunk
)
{
    //  Does the current chunk exist?
    for( DWord dwChunk = 0 ; dwChunk < rdwChunkSize ; dwChunk += 1 )
    {
        if( pfChunkList[ dwChunk ] == fChunk )
        {
            return;
        }
    }

    //  The chunk wasn't found so add a new one.
    pfChunkList[ rdwChunkSize ] = fChunk;
    rdwChunkSize += 1;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_compareChunks
//
//  Description:
//
//      This member function is used to compare two chunks of a frame.
//
//  Parameters:
//
//      pChunk1 (input)
//          A pointer to the first Chunk object.
//
//      pChunk2 (input)
//          A pointer to the second Chunk object.
//
//  Returns:
//
//      -1 if the first chunk is less than the second.
//       0 if the first chunk is equal to the second.
//      +1 if the first chunk is greater than the second.
//
///////////////////////////////////////////////////////////////////////////////
int
Scheduler::s_compareChunks(
    const void* pChunk1,
    const void* pChunk2
)
{
    //  Cast the items to the appropriate type.
    double fChunk1 = *( ( double* )pChunk1 );
    double fChunk2 = *( ( double* )pChunk2 );

    //  Compare the chunks.
    if( fChunk1 < fChunk2 )
    {
        return( -1 );
    }
    else
    if( fChunk1 > fChunk2 )
    {
        return( 1 ); }
    else
    {
        return( 0 );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_compareSlices
//
//  Description:
//
//      This member function is used to compare two time slices.
//
//  Parameters:
//
//      pSlice1 (input)
//          A pointer to the first slice object.
//
//      pSlice2 (input)
//          A pointer to the second slice object.
//
//  Returns:
//
//      -1 if the first slice is less than the second.
//       0 if the first slice is equal to the second.
//      +1 if the first slice is greater than the second.
//
///////////////////////////////////////////////////////////////////////////////
int
Scheduler::s_compareSlices(
    const void* pSlice1,
    const void* pSlice2
)
{
    //  Cast the items to the appropriate type.
    Slice* pSliceA = *( ( Slice** )pSlice1 );
    Slice* pSliceB = *( ( Slice** )pSlice2 );

    //  We can't compare the end positions since the end position value
    //  is measured in cycles which differ from CPU to CPU.  Therefore, we
    //  compare the relative positions of the end positions within the frame.
    double fTermA = ( double )pSliceA->m_nEnd / pSliceA->m_nTotal;
    double fTermB = ( double )pSliceB->m_nEnd / pSliceB->m_nTotal;

    //  Perform the comparison based on the frame position.  To avoid
    //  round-off error, if the difference is less than .001 then we 
    //  consider the terms to be equal.
    if( ( fTermA < fTermB ) && ( ( fTermB - fTermA ) > .001 ) )
    {
        return( -1 );
    }
    else
    if( ( fTermA > fTermB ) && ( ( fTermA - fTermB ) > .001 ) )
    {
        return( 1 );
    }
    else
    {
        //  They're at the same position so check the cpu index.
        if( pSliceA->m_nCPU < pSliceB->m_nCPU )
        {
            return( -1 );
        }
        else
        if( pSliceA->m_nCPU > pSliceB->m_nCPU )
        {
            return( 1 );
        }
    }
        
    //  If we get this far, the slices must be equivalent.
    //  But we should never have equivalent slices!!!
    ASSERT( FALSE );
    return( 0 );
}
