///////////////////////////////////////////////////////////////////////////////
//
//  File:    screen.cpp
//
//  Class:   Screen
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents the screen of the emulated game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "screen.h"
#include "replay.h"
#include "canvas.h"
#include "appfile.h"
#include "bitmap.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Screen
//
//  Description:
//
//      This is the main constructor for a screen object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Screen::Screen(
    const KString&  iName
)
:
    RepBase           ( iName ),
    m_pScreen         ( NULL ),
    m_pTempScreen     ( NULL ),
    m_dwFPS           ( 60 ),
    m_dwVBlankLength  ( 0 ),
    m_pbVBlankByte    ( NULL ),
    m_bVBlankMask     ( 0x00 )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Screen
//
//  Description:
//
//      This is the destructor for a screen object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Screen::~Screen(
)
{
    //  Delete the bitmaps.
    delete m_pScreen;
    delete m_pTempScreen;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the screen object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Screen::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Screen" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the screen to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the screen to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Screen::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );

    //  Allow the screen bitmaps to save themselves.
    DWord dwNumErrors( 0 );
    dwNumErrors += m_pScreen->save( pSaveFile );
    if( m_pTempScreen )
    {
        dwNumErrors += m_pTempScreen->save( pSaveFile );
    }

    //  Was it saved successfully?
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the screen from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the vblank from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Screen::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );

    //  Load the member data that keeps the state.
    DWord dwNumErrors( 0 );
    dwNumErrors += m_pScreen->save( pLoadFile );
    if( m_pTempScreen )
    {
        dwNumErrors += m_pTempScreen->save( pLoadFile );
    }

    //  Was it loaded successfully?
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setScreenSize
//
//  Description:
//
//      This member is called to create the screen for the game.
//
//  Parameters:
//
//      dwWidth (input)
//          The width of the screen.
//
//      dwHeight (input)
//          The height of the screen.
//
//      dwWidthScale (input)
//          The width (in screen widths) of the double buffered screen.
//
//      dwHeightScale (input)
//          The height (in screen heights) of the double buffered screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Screen::setScreenSize( 
    const DWord dwWidth,
    const DWord dwHeight,
    const DWord dwWidthScale  /* = 0 */,
    const DWord dwHeightScale /* = 0 */
)
{
    //  Currently, we don't allow a resize of the screen.
    CONFIRM( m_pScreen == NULL, "Cannot change screen size" );

    //  Make sure the screen has some size.
    ASSERT( dwWidth  > 0 );
    ASSERT( dwHeight > 0 );

    //  Make a bitmap for the screen and clear it.
    m_pScreen = Replay::s_instance( ).getCanvas( )->createBitmap( 
        "Game Screen", dwWidth, dwHeight, TRUE
    );
    m_pScreen->clear( 0 );

    //  If there is to be a double buffered screen bitmap then create it and
    //  clear it.
    if( ( dwWidthScale != 0 ) && ( dwHeightScale != 0 ) )
    {
        m_pTempScreen = Replay::s_instance( ).getCanvas( )->createBitmap( 
            "Temp Screen", 
            dwWidth * dwWidthScale, 
            dwHeight * dwHeightScale,
            FALSE
        );
        m_pTempScreen->clear( 0 );
    }
}


    
///////////////////////////////////////////////////////////////////////////////
//
//  Function: suggestScreenSize
//
//  Description:
//
//      This member is called to suggest a screen size for the game.  Calling
//      this function instead of setScreenSize indicates that the game is
//      willing to have a user-specified screen size override the suggestion. 
//
//  Parameters:
//
//      dwWidth (input)
//          The width of the screen.
//
//      dwHeight (input)
//          The height of the screen.
//
//      dwWidthScale (input)
//          The width (in screen widths) of the double buffered screen.
//
//      dwHeightScale (input)
//          The height (in screen heights) of the double buffered screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Screen::suggestScreenSize( 
    const DWord dwWidth,
    const DWord dwHeight,
    const DWord dwWidthScale  /* = 0 */,
    const DWord dwHeightScale /* = 0 */
)
{
    //  Currently, we don't allow a resize of the screen.
    CONFIRM( m_pScreen == NULL, "Cannot change screen size" );

    //  Here we would check if the user has specified a screen size and use
    //  that.
    setScreenSize( dwWidth, dwHeight, dwWidthScale, dwHeightScale );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: switchVBlank
//
//  Description:
//
//      This member is called to set or unset the status of the VBlank (i.e.
//      whether or not we are currently in the VBlank).
//
//  Parameters:
//
//      bOn (input)
//          TRUE if we are in VBlank, FALSE otherwise.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Screen::switchVBlank(
    const Byte bOn
)
{
    //  If no VBlank has been specified then just return.
    if( m_pbVBlankByte == NULL )
    {
        return;
    }

    //  If the VBlank is on then set the bit, otherwise unset it.
    if( bOn )
    {
        *m_pbVBlankByte |= m_bVBlankMask;
    }
    else
    {
        *m_pbVBlankByte &= ~m_bVBlankMask;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clear
//
//  Description:
//
//      This member is called to clear the screen bitmaps.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Screen::clear(
)
{
    //  Shouldn't be called unless a screen has been created.
    ASSERT( m_pScreen != NULL );
    m_pScreen->clear( );
    if( m_pTempScreen )
    {
        m_pTempScreen->clear( );
    }
}
