///////////////////////////////////////////////////////////////////////////////
//
//  File:       screen.h
//
//  Class:      Screen - Concrete
//  
//  Hierarchy:  Screen - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents the screen of the emulated game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _screen_h_
#define _screen_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "clock.h"

//  Forward class declarations.
class Bitmap;

///////////////////////////////////////////////////////////////////////////////
//  Screen Class Definition
///////////////////////////////////////////////////////////////////////////////
class Screen : public RepBase
{
    public:

        //  Creation/Deletion.
        Screen ( const KString& iName );
        virtual ~Screen ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  Interface.
        void    setScreenSize     (
            const DWord dwWidth,
            const DWord dwHeight,
            const DWord dwWidthScale  = 0,
            const DWord dwHeightScale = 0
        );
        void    suggestScreenSize (
            const DWord dwWidth,
            const DWord dwHeight,
            const DWord dwWidthScale  = 0,
            const DWord dwHeightScale = 0
        );
        void    setFPS            ( const DWord dwFPS );
        void    setVBlank         ( 
            const DWord dwLength, 
            Byte*       pbByte = NULL,
            Byte        bMask  = 0x00 
        );
        void    switchVBlank      ( const Byte bOn );
        void    clear             ( );

        //  Access.
        Bitmap* getBitmap         ( ) const;
        Bitmap* getTempBitmap     ( ) const;
        DWord   getFPS            ( ) const;
        DWord   getVBlankLength   ( ) const;

    protected:

        //  Member data.
        Bitmap* m_pScreen;
        Bitmap* m_pTempScreen;
        DWord   m_dwFPS;
        DWord   m_dwVBlankLength;
        Byte*   m_pbVBlankByte;
        Byte    m_bVBlankMask;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  Screen Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
Screen::setFPS(
    const DWord dwFPS
)
{
    m_dwFPS = dwFPS;
}

inline
void
Screen::setVBlank(
    const DWord dwLength, 
    Byte*       pbByte /* = NULL */,
    Byte        bMask  /* = 0x00 */
)
{
    m_dwVBlankLength = dwLength;
    m_pbVBlankByte   = pbByte;
    m_bVBlankMask    = bMask;
}

inline
Bitmap*
Screen::getBitmap(
) const
{
    return( m_pScreen );
}

inline
Bitmap*
Screen::getTempBitmap(
) const
{
    return( m_pTempScreen );
}

inline
DWord
Screen::getFPS(
) const
{
    return( m_dwFPS );
}

inline
DWord
Screen::getVBlankLength(
) const
{
    return( m_dwVBlankLength );
}


#endif
