///////////////////////////////////////////////////////////////////////////////
//
//  File:    dip.cpp
//
//  Class:   DipSwitch
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to keep track of a dip switch setting.  A setting
//      consists of a bit mask indicating which bits of the dip switch are
//      involved in the setting and a list of text descriptions of what
//      the various values of the mask mean in regards to the game operation.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "setting.h"
#include "dip.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The default string for an unknown description.
const KString Setting::sm_unknownDescription( "Unknown" );



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Setting
//
//  Description:
//
//      This is the main constructor for a dip switch setting object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pDipSwitch (input)
//          The dip switch that the setting is associated with.
//
//      bMask (input)      
//          The bit mask indicating which bits the setting corresponds to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Setting::Setting(
    const KString&        iName,
    DipSwitch*            pDipSwitch,
    Byte                  bMask
)
:
    RepBase           ( iName ),
    m_pDipSwitch      ( pDipSwitch ),
    m_bMask           ( bMask ),
    m_bBitsInMask     ( 0 ),
    m_descriptionList ( 8 )
{
    //  Check the parameters.
    ASSERT( m_pDipSwitch != NULL );
    ASSERT( bMask != 0x00 );

    //  Calculate the number of bits in the mask.
    for( DWord dwI = 0 ; dwI < 8 ; dwI += 1 )
    {
        if( bMask & (1 << dwI ) )
        {
            m_bBitsInMask += 1;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Setting
//
//  Description:
//
//      This is the destructor for a dip switch setting object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Setting::~Setting(
)
{
    //  Clean up the descriptions.
    for( DWord dwI = 0 ; dwI < m_descriptionList.entries( ) ; dwI += 1 )
    {
        delete m_descriptionList[ dwI ];
    }
    m_descriptionList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the dip switch setting object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Setting::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Setting" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addDescription
//
//  Description:
//
//      This member is used to add the next description to the list of
//      descriptions.  There should be one description for each combination
//      of bits in the bitmask.
//
//  Parameters:
//
//      pstrDescription (input)
//          The description of the setting.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Setting::addDescription(
    const char* pstrDescription
)
{
    //  Check the parameters.
    ASSERT( pstrDescription != NULL );

    //  Add it to the description list.
    m_descriptionList.add( new KString( pstrDescription ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: squashValue
//
//  Description:
//
//      This member is used to move the bits that correspond to the setting
//      mask to the right side of a byte and return the corresponding value.
//      It also takes into account whether the dip switch for the setting
//      is reversed or not.
//
//  Parameters:
//
//      bUnsquashed (input)
//          The unsquashed value.
//
//  Returns:
//
//      The squashed value.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Setting::squashValue(
    const Byte bUnsquashed
) const
{
    //  A temporary byte and the squashed value.
    Byte bTemp;
    Byte bSquashed = 0x00;

    //  Initialize the temp to the unsquashed value.
    bTemp = bUnsquashed;

    //  If the dip switch is reversed then flip the bits.
    if( m_pDipSwitch->isReversed( ) )
    {
        bTemp ^= m_bMask;
    }

    //  Mask out the bits we don't care about.
    bTemp &= m_bMask;

    //  Now squash the bits to the right hand side.
    for( Byte bSrc = 0, bDst = 0 ; bSrc < 8 ; bSrc += 1 )
    {
        //  If the bit is set in the mask then place the corresponding bit
        //  in the destination.
        if( m_bMask & ( 1 << bSrc ) )
        {
            bSquashed |= ( bTemp & ( 1 << bSrc ) ) >> ( bSrc - bDst );
            bDst += 1;
        }
    }

    //  Return the squashed value.
    return( bSquashed );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: unsquashValue
//
//  Description:
//
//      This member is used to move the bits to the right of the specified 
//      byte out to the bits specified in the setting mask.  It also
//      takes into account whether the dip switch for the setting
//      is reversed or not.
//
//  Parameters:
//
//      bSquashed (input)
//          The squashed value.
//
//  Returns:
//
//      The Unsquashed value.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Setting::unsquashValue(
    const Byte bSquashed
) const
{
    //  A temporary byte and the unsquashed value.
    Byte bTemp;
    Byte bUnsquashed = 0x00;

    //  Initialize the temp to the squashed value.
    bTemp = bSquashed;

    //  Now unsquash the bits from the right hand side.
    for( Byte bSrc = 0, bDst = 0 ; bDst < 8 ; bDst += 1 )
    {
        //  If the bit is set in the mask then place the corresponding bit
        //  in the destination.
        if( m_bMask & ( 1 << bDst ) )
        {
            bUnsquashed |= ( bTemp & ( 1 << bSrc ) ) << ( bDst - bSrc );
            bSrc += 1;
        }
    }

    //  If the dip switch is reversed then flip the bits.
    if( m_pDipSwitch->isReversed( ) )
    {
        bUnsquashed ^= m_bMask;
    }

    //  Return the unsquashed value.
    return( bUnsquashed );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getCurDescription
//
//  Description:
//
//      This member is used to return the description corresponding to
//      the current state of bitmask bits as read from the associated dip
//      switch.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The description requested.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Setting::getCurDescription(
) const
{
    //  The index of the description.
    Byte bIndex = squashValue( m_pDipSwitch->getValue( ) );

    //  OK, if there is a corresponding description then return it, otherwise
    //  return the "Unknown" string.
    if( bIndex < m_descriptionList.entries( ) )
    {
        return( *( m_descriptionList[ bIndex ] ) );
    }
    else
    {
        return( sm_unknownDescription );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: next
//
//  Description:
//
//      This member is used to modify the dip switch byte so that it 
//      represents the next description of the setting.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Setting::next(
)
{
    //  The original value of the dip switch.
    Byte bOriginal = m_pDipSwitch->getValue( );

    //  Get the squashed value of the setting bits.
    Byte bIndex = squashValue( bOriginal );

    //  Move on to the next setting.
    bIndex += 1;

    //  If we have exceeded the maximum number of settings then wrap around.
    if( bIndex >= ( 1 << m_bBitsInMask ) )
    {
        bIndex = 0x00;
    }

    //  Place the modified value back into the dip switch.
    m_pDipSwitch->setValue( bOriginal & ~m_bMask | unsquashValue( bIndex ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: prev
//
//  Description:
//
//      This member is used to modify the dip switch byte so that it 
//      represents the previous description of the setting.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Setting::prev(
)
{
    //  The original value of the dip switch.
    Byte bOriginal = m_pDipSwitch->getValue( );

    //  Get the squashed value of the setting bits.
    Byte bIndex = squashValue( bOriginal );

    //  Move on to the previous setting with wrapping.
    if( bIndex == 0 )
    {
        bIndex = ( 1 << m_bBitsInMask ) - 1;
    }
    else
    {
        bIndex -= 1;
    }

    //  Place the modified value back into the dip switch.
    m_pDipSwitch->setValue( bOriginal & ~m_bMask | unsquashValue( bIndex ) );
}
