///////////////////////////////////////////////////////////////////////////////
//
//  File:    _amihw.cpp
//
//  Class:   GameAmidarHW
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for a series of
//      games from Stern that is based on what we'll call the "Amidar" 
//      hardware.
//
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////
//  System Headers.

//  Application Headers.
#include "_amihw.h"
#include "space.h"
#include "cqueue.h"
#include "input.h"
#include "ctrlmap.h"
#include "clip.h"
#include "gfxset.h"
#include "ctable.h"
#include "ay8910.h"
#include "hiscore.h"
#include "bitmap.h"
#include "screen.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameAmidarHW
//
//  Description:
//
//      This is the main constructor for a game object based on Amidar
//      hardware.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameAmidarHW::GameAmidarHW(
    const KString& iName
)
:
    Game              ( iName ),
    m_pInputIN0       ( NULL ),
    m_pInputIN1       ( NULL ),
    m_pInputIN2       ( NULL ),
    m_pInputDSW       ( NULL ),
    m_pSpaceGame      ( NULL ),
    m_pSpaceSound     ( NULL ),
    m_pBufferGfx      ( NULL ),
    m_pSoundCommand   ( NULL ),
    m_pCPUGame        ( NULL ),
    m_pCPUSound       ( NULL ),
    m_pDirty          ( NULL ),
    m_pbVideoRAM      ( NULL ),
    m_pbAttributeRAM  ( NULL ),
    m_pbSpriteRAM     ( NULL ),
    m_pGfxSetChar     ( NULL ),
    m_pGfxSetSprite   ( NULL ),
    m_pColTable       ( NULL ),
    m_pAY8910One      ( NULL ),
    m_pAY8910Two      ( NULL ),
    m_pClipping       ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameAmidarHW
//
//  Description:
//
//      This is the destructor for a game object based on Amidar hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameAmidarHW::~GameAmidarHW(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of a game based on Amidar hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameAmidarHW::startUpSpace(
)
{
    //  Create the address spaces for the CPUs.
    m_pSpaceGame  = m_spaceList.add( new AddrSpace( "Game" ) );
    m_pSpaceSound = m_spaceList.add( new AddrSpace( "Sound" ) );

    //  Create temporary buffer for decoding the graphics.
    m_pBufferGfx = m_tempBufferList.add( new Buffer( "FG Chars", 0x1000 ) );

    //  Create a command queue for communication between the CPUs.
    m_pSoundCommand = m_bufferList.add( new CommandQueue( "Cmd", 1, TRUE ) );

    //  Some convenient pointers.
    m_pbVideoRAM     = m_pSpaceGame->getBuffer( ) + 0x9000;
    m_pbAttributeRAM = m_pSpaceGame->getBuffer( ) + 0x9800;
    m_pbSpriteRAM    = m_pSpaceGame->getBuffer( ) + 0x9840;
}

void
GameAmidarHW::startUpCPU(
)
{
    //  We interleave the CPUs at the rate of 10 times per frame to allow
    //  proper communication.
    m_nCPUInterleave = 10;

    //  Amidar hardware uses two Z80's, one for the game and one for the Sound.
    m_pCPUGame = m_cpuList.add( 
        createCPU( "Game",  CPU::CPU_Z80, m_pSpaceGame )
    );
    m_pCPUSound = m_cpuList.add( 
        createCPU( "Sound", CPU::CPU_Z80, m_pSpaceSound, CPU::SOUND )
    );
    
    //  The game CPU runs at 3.072Mhz.  The sound CPU runs at 1.78975Mhz.
    m_pCPUGame->setClockSpeed( 3072000 );
    m_pCPUSound->setClockSpeed( 1789750 );
     
    //  The first CPU uses NMI interrupts, the second has interrupts triggered
    //  from the first CPU.
    m_pCPUGame->interruptHandler( s_nmiInterrupt );
    m_pCPUSound->interruptHandler( s_noInterrupt );
}

void
GameAmidarHW::startUpInput(
)
{
    //  Create the inputs needed.
    m_pInputIN0 = m_inputList.add( new Input( "IN0", 0xff ) );
    m_pInputIN1 = m_inputList.add( new Input( "IN1", 0xff ) );
    m_pInputIN2 = m_inputList.add( new Input( "IN2", 0xf1 ) );
    m_pInputDSW = m_inputList.add( new Input( "DSW", 0xff ) );

    //  Setup the controls.
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Up", 
            CtrlMap::COCKTAIL2, 
            CtrlMap::P2_UP, 
            m_pInputIN0, 
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 3", 
            CtrlMap::PLAYER1, 
            CtrlMap::COIN3, 
            m_pInputIN0, 
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Action",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION1, 
            m_pInputIN0,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Right",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_RIGHT, 
            m_pInputIN0,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Left",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_LEFT, 
            m_pInputIN0,
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 2", 
            CtrlMap::PLAYER1, 
            CtrlMap::COIN2, 
            m_pInputIN0, 
            0x40
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 1", 
            CtrlMap::PLAYER1, 
            CtrlMap::COIN1, 
            m_pInputIN0, 
            0x80
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Action",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION1, 
            m_pInputIN1,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Right",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_RIGHT, 
            m_pInputIN1,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Left",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_LEFT, 
            m_pInputIN1,
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_pInputIN1,
            0x40
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START, 
            m_pInputIN1,
            0x80
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Down",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_DOWN, 
            m_pInputIN2,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Up",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_UP, 
            m_pInputIN2,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Down",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_DOWN, 
            m_pInputIN2,
            0x40
        )
    );
}

void
GameAmidarHW::startUpGraphics(
)
{
    //  Amidar HW games must have a 256x256 screen.  We also request a double
    //  buffer of the same size for storing characters.
    //  The VBlank duration is set to 2500usec although this is unverified.
    m_pScreen->setScreenSize( 256, 256, 1, 1 );
    m_pScreen->setVBlank( 2500 );

    //  The main clipping area chops 2 columns on the left and right.
    m_pClipping = m_clippingList.add( new Clipping(  16, 239, 0, 255 ) );

    //  Map the static character positions on the temporary bitmap.
    m_pScreen->getTempBitmap( )->setMapSize( 0x400 );
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        m_pScreen->getTempBitmap( )->setMapping( 
            dwI, ( 31 - dwI / 32 ) * 8, ( dwI % 32 ) * 8, m_pClipping
        ); 
    }

    //  Allocate a buffer that will be used to indicate which
    //  characters need to be redrawn.
    m_pDirty = m_dirtyList.add( new Buffer( "Dirty", 0x0400, TRUE ) );

    //  Create the gfxset to hold the characters and set the information.
    m_pGfxSetChar = m_gfxSetList.add( new GfxSet( "Characters" ) );
    m_pGfxSetChar->setNumber( 256 );
    m_pGfxSetChar->setDimensions( 8, 8 );
    m_pGfxSetChar->setBPP( 2 );
    m_pGfxSetChar->setBitPlanes( 0, 16384 );
    m_pGfxSetChar->setXBits(  56, 48, 40, 32, 24, 16,  8,  0 );
    m_pGfxSetChar->setYBits(   0,  1,  2,  3,  4,  5,  6,  7 );
    m_pGfxSetChar->setIncrement( 64 );

    //  Decode the foreground characters.
    m_pGfxSetChar->decode( m_pBufferGfx->getBuffer( ) );
    

    //  Create the gfxset to hold the sprites and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprites" ) );
    m_pGfxSetSprite->setNumber( 64 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 2 );
    m_pGfxSetSprite->setBitPlanes( 0, 16384 );
    m_pGfxSetSprite->setXBits(  
        184, 176, 168, 160, 152, 144, 136, 128, 
         56,  48,  40,  32,  24,  16,   8,   0 
    );
    m_pGfxSetSprite->setYBits(  
          0,   1,   2,   3,   4,   5,   6,   7, 
         64,  65,  66,  67,  68,  69,  70,  71
    );
    m_pGfxSetSprite->setIncrement( 256 );

    //  Decode the foreground characters.
    m_pGfxSetSprite->decode( m_pBufferGfx->getBuffer( ) );
}

void
GameAmidarHW::startUpColour(
)
{
    ASSERT( getColourPROM( ) != NULL );
    Byte* pbColourPROM = getColourPROM( )->getBuffer( );

    //  Amidar hardware games use a PROM to indicate the static palette.
    //
    //  The PROM is hooked up to RGB output in the following manner:
    //
    //  Bit       Resistor               Colour     State
    //   0  /\/\/   1 kohm  \/\/\/\/\/\/\  Red      normal
    //   1  /\/\/ 470  ohm  \/\/\/\/\/\/\  Red      normal
    //   2  /\/\/ 220  ohm  \/\/\/\/\/\/\  Red      normal
    //   3  /\/\/   1 kohm  \/\/\/\/\/\/\  Green    normal
    //   4  /\/\/ 470  ohm  \/\/\/\/\/\/\  Green    normal
    //   5  /\/\/ 220  ohm  \/\/\/\/\/\/\  Green    normal
    //   6  /\/\/ 470  ohm  \/\/\/\/\/\/\  Blue     normal
    //   7  /\/\/ 220  ohm  \/\/\/\/\/\/\  Blue     normal
    //
    for( DWord dwI = 0 ; dwI < getColourPROM( )->getSize( ) ; dwI += 1 )
    {
        //  Used to calculate the palette entries.
        Byte bBits[ 3 ];

        //  The RGB components.
        DWord dwRed; 
        DWord dwGreen; 
        DWord dwBlue; 

        //  Calculate the colour components.
        bBits[ 0 ] = ( *pbColourPROM >> 0 ) & 0x01;
        bBits[ 1 ] = ( *pbColourPROM >> 1 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 2 ) & 0x01;
        dwRed      = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

        bBits[ 0 ] = ( *pbColourPROM >> 3 ) & 0x01;
        bBits[ 1 ] = ( *pbColourPROM >> 4 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 5 ) & 0x01;
        dwGreen    = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

        bBits[ 0 ] = 0;
        bBits[ 1 ] = ( *pbColourPROM >> 6 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 7 ) & 0x01;
        dwBlue     = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];
 
        pbColourPROM += 1;

        //  Set the colour.
        m_pReplay->getCanvas( )->setColour( dwI, dwRed, dwGreen, dwBlue );
    }

    //  The characters and sprites use the same colour table which is 32
    //  entries long at 2 bits per entry (8 entries * 4 values/entry).
    m_pColTable = m_colourTableList.add( new ColourTable( "ColTable", 8, 4 ) );

    //  Assign the values for the colour tables.
    for( DWord dwI = 0 ; dwI < 8 * 4 ; dwI += 1 )
    {
        ( *m_pColTable )[ dwI ] = ( dwI % 4 ) ? dwI : 0;
    }
}

void
GameAmidarHW::startUpSound(
)
{
    //  Games based on the Amidar hardware use two AY-3-8910s hooked up to the 
    //  sound CPU.
    m_pAY8910One = m_soundDeviceList.add( new AY8910( "1", 1789750, 0x60ff ) );
    m_pAY8910Two = m_soundDeviceList.add( new AY8910( "2", 1789750, 0x60ff ) );

    //  The first 8910 gets input for port A from the sound command, and 
    //  the input for port B from the timer clock that feeds the CPU.
    ( ( AY8910* )m_pAY8910One )->setPortAReadHandler( 
        s_readCommand, m_pSoundCommand 
    );
    ( ( AY8910* )m_pAY8910One )->setPortBReadHandler( s_readTimer, NULL );
}

void
GameAmidarHW::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "ROM",            0x0000, 0x4fff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x8000, 0x87ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x9000, 0x93ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x9800, 0x985f, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Watchdog Reset", 0xa800, 0xa800, s_readWatchDog,     NULL
    );
    m_pCPUGame->readMemHandler( 
        "IN0",            0xb000, 0xb000, s_readInput,        m_pInputIN0
    );
    m_pCPUGame->readMemHandler( 
        "IN1",            0xb010, 0xb010, s_readInput,        m_pInputIN1
    );
    m_pCPUGame->readMemHandler( 
        "IN2",            0xb020, 0xb020, s_readInput,        m_pInputIN2
    );
    m_pCPUGame->readMemHandler( 
        "IN3",            0xb820, 0xb820, s_readInput,        m_pInputDSW
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Scratch RAM",    0x8000, 0x87ff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Video RAM",      0x9000, 0x93ff, s_writeDirtyByte,    
        m_pDirty->getBuffer ( )
    );
    m_pCPUGame->writeMemHandler(
        "Attributes RAM", 0x9800, 0x983f, s_writeAttributes,   this
    );
    m_pCPUGame->writeMemHandler(
        "Sprite RAM",     0x9840, 0x985f, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler( 
        "Protected",      0x9860, 0x987f, s_writeNo,           NULL
    );
    m_pCPUGame->writeMemHandler(
        "IEnable",        0xa008, 0xa008, s_writeIntEnable,    m_pCPUGame
    );
    m_pCPUGame->writeMemHandler(
        "Screen FlipX",   0xa010, 0xa010, s_writeScreenFlipX,  this
    );
    m_pCPUGame->writeMemHandler(
        "Screen FlipY",   0xa018, 0xa018, s_writeScreenFlipY,  this
    );
    m_pCPUGame->writeMemHandler(
        "Protected",      0xa030, 0xa030, s_writeNo,           NULL
    );
    m_pCPUGame->writeMemHandler(
        "Protected",      0xa038, 0xa038, s_writeNo,           NULL
    );
    m_pCPUGame->writeMemHandler(
        "Sound Command",  0xb800, 0xb800, s_writeCommand,      m_pSoundCommand
    );
    m_pCPUGame->writeMemHandler(
        "Sound IRQ",      0xb810, 0xb810, s_writeSoundIRQ,     m_pCPUSound
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0x0000, 0x4fff, s_writeROM,          NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "ROM",            0x0000, 0x1fff, s_readROM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "RAM",            0x8000, 0x83ff, s_readRAM,          NULL
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "RAM",            0x8000, 0x83ff, s_writeRAM,         NULL
    );
    m_pCPUSound->writeMemHandler(
        "Protected",      0x9000, 0x9000, s_writeNo,          NULL
    );
    m_pCPUSound->writeMemHandler(
        "Protected",      0x9080, 0x9080, s_writeNo,          NULL
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0x0000, 0x1fff, s_writeROM,         NULL
    );

    //  Register the Port read handlers for the Sound CPU.
    m_pCPUSound->readPortHandler(
        "8910#1 Read",    0x0080, 0x0080, s_read8910,         m_pAY8910One
    );
    m_pCPUSound->readPortHandler(
        "8910#2 Read",    0x0020, 0x0020, s_read8910,         m_pAY8910Two
    );

    //  Register the Port write handlers for the Sound CPU.
    m_pCPUSound->writePortHandler(
        "8910#1 Control", 0x0040, 0x0040, s_write8910Control, m_pAY8910One
    );
    m_pCPUSound->writePortHandler(
        "8910#1 Write",   0x0080, 0x0080, s_write8910Write,   m_pAY8910One
    );
    m_pCPUSound->writePortHandler(
        "8910#2 Control", 0x0010, 0x0010, s_write8910Control, m_pAY8910Two
    );
    m_pCPUSound->writePortHandler(
        "8910#2 Write",   0x0020, 0x0020, s_write8910Write,   m_pAY8910Two
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameAmidarHW::updateDisplay(
    Bitmap* pScreen,
    Bitmap* pTempScreen
)
{
    //  A pointer to the dirty buffer.
    Byte* pbDirty = m_pDirty->getBuffer( );


    //  Draw any dirty characters.
    //  There are 32x32=1024 characters that make up the background.
    for( DWord dwI = 0 ; dwI < 0x0400 ; dwI += 1 )
    {
        //  Draw the characters onto the temporary bitmap if it is dirty.
        if( pbDirty[ dwI ] )
        {
            //  Reset the dirty flag.
            pbDirty[ dwI ] = FALSE;

            //  Draw the character into the double buffer.
            pTempScreen->blit(
                ( *m_pGfxSetChar )[ m_pbVideoRAM[ dwI ] ],
                dwI,
                m_pColTable->getEntry( 
                    m_pbAttributeRAM[ 2 * ( dwI % 32 ) + 1 ] & 0x07
                )
            );
        }
    }

    //  Copy the temporary bitmap onto the screen.
    pScreen->copy( pTempScreen );

    //  Draw the sprites.  There are 8 sprites each using 4 bytes.
    for( int32 nI = 7 * 4 ; nI >= 0 ; nI -= 4 )
    {
        //  Draw the sprite.
        pScreen->blit(
            ( *m_pGfxSetSprite )[ m_pbSpriteRAM[ nI + 1 ] & 0x3f ],
            m_pbSpriteRAM[ nI + 0 ],
            m_pbSpriteRAM[ nI + 3 ],
            m_pColTable->getEntry( m_pbSpriteRAM[ nI + 2 ] & 0x07 ),
            m_pbSpriteRAM[ nI + 1 ] & 0x80,
            m_pbSpriteRAM[ nI + 1 ] & 0x40,
            m_pClipping,
            Bitmap::TRANSPARENCY_RAW
        );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readTimer
//
//  Description:
//
//      This member is called by the 8910.  It simulates a clock input that
//      drives the PSG.
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the timer.
//
///////////////////////////////////////////////////////////////////////////////
DWord
GameAmidarHW::s_readTimer(
    DWord        /* dwAddress is not used */,
    ReadHandler* /* pHandler is not used  */
)
{
    //  From the MAME Scramble driver:
    // 
    //  The timer clock which feeds the upper 4 bits of
    //  AY-3-8910 port B is based on the same clock
    //  feeding the sound CPU Z80.|  It is a divide by
    //  5120, formed by a standard divide by 512, followed
    //  by a divide by 10 using a 4 bit bi-quinary count
    //  sequence. (See LS90 data sheet for an example)
    //  The upper three bits come directly from the
    //  upper three bits of the bi-quinary counter.
    //  Bit 4 comes from the output of the divide by 512.
    static int anTimer[ 20 ] = 
    {
        0x00, 0x10, 0x00, 0x10, 0x20, 0x30, 0x20, 0x30, 0x40, 0x50,
        0x80, 0x90, 0x80, 0x90, 0xa0, 0xb0, 0xa0, 0xb0, 0xc0, 0xd0
    };

    //  This is used to protect from cycle overflow.
    static int nLastTotalCycles = 0;

    //  Number of Z80 clock cycles to count.
    static int nClock;

    //  The current total cycles.
    int nCurrentTotalCycles = CPU::sm_pCPU->getCyclesTotal( );

    //  Calculate the clock.
    nClock = ( nClock + ( nCurrentTotalCycles - nLastTotalCycles ) ) % 5120;

    nLastTotalCycles = nCurrentTotalCycles;

    return( anTimer[ nClock / 256 ] );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeAttributes
//
//  Description:
//
//      This function is called when the attributes RAM is written to.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameAmidarHW::s_writeAttributes(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  Get the dirty buffer.
    Byte* pbDirty = 
        ( ( GameAmidarHW* )( pHandler->getData( ) ) )->m_pDirty->getBuffer( );

    //  The attributes are stored in every other address on the odd boundaries.
    if( 
        ( dwAddress & 0x01 ) && 
        ( ( CPU::sm_pSpace->getBuffer( ) )[ dwAddress ] != bValue ) 
    )
    {
        //  Mark the affected items as dirty.
        for( DWord dwI = ( dwAddress & 0x3f ) / 2 ; dwI < 0x400 ; dwI += 32 )
        {
            pbDirty[ dwI ] = TRUE;
        }
    }

    ( CPU::sm_pSpace->getBuffer( ) )[ dwAddress ] = bValue;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSoundIRQ
//
//  Description:
//
//      This function is called when the main CPU requests that the sound
//      CPU be interrupted.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameAmidarHW::s_writeSoundIRQ(
    DWord         /* dwAddress is unused */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The last value.
    static Byte bLast = 0x00;

    //  Get the CPU being interrupted.
    CPU* pCPU = ( CPU* )( pHandler->getData( ) );


    //  Perform the interrupt.
    if( ( bLast == 0 ) && ( bValue & 0x08 ) )
    {
        pCPU->interrupt( 0xff );

        //  Cease operation of the current CPU so the interrupt can take
        //  effect.
        CPU::sm_pCPU->setCyclesLeft( 0 );
    }

    bLast = bValue & 0x08;
}
