///////////////////////////////////////////////////////////////////////////////
//
//  File:    _bm6809.cpp
//
//  Class:   GameBallyMidway6809
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for a series of
//      games from Bally Midway that run on hardware with two Motorola
//      6809 processors.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "_bm6809.h"
#include "space.h"
#include "input.h"
#include "dip.h"
#include "ctrlmap.h"
#include "gfxset.h"
#include "ctable.h"
#include "bitmap.h"
#include "waveform.h"
#include "screen.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameBallyMidway6809
//
//  Description:
//
//      This is the main constructor for a game object based on dual 6809 
//      Bally Midway hardware.
//      
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBallyMidway6809::GameBallyMidway6809(
    const KString& iName
)
:
    Game              ( iName ),
    m_pInputDSW0      ( NULL ),
    m_pInputDSW1      ( NULL ),
    m_pInputIN0       ( NULL ),
    m_pInputIN1       ( NULL ),
    m_pInputIN2       ( NULL ),
    m_pSpaceGame      ( NULL ),
    m_pSpaceSound     ( NULL ),
    m_pBufferChar     ( NULL ),
    m_pBufferSprite   ( NULL ),
    m_pCPUGame        ( NULL ),
    m_pCPUSound       ( NULL ),
    m_pDirty          ( NULL ),
    m_pbVideoRAM      ( NULL ),
    m_pbColourRAM     ( NULL ),
    m_pbSprite1RAM    ( NULL ),
    m_pbSprite2RAM    ( NULL ),
    m_pbSprite3RAM    ( NULL ),
    m_pbSoundRAM      ( NULL ),
    m_pGfxSetChar     ( NULL ),
    m_pColTableChar   ( NULL ),
    m_pColTableSprite ( NULL ),
    m_pWaveForm       ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameBallyMidway6809
//
//  Description:
//
//      This is the destructor for a game object based on dual 6809 Bally
//      Midway hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBallyMidway6809::~GameBallyMidway6809(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of a game based on dual 6809 Bally Midway hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBallyMidway6809::startUpSpace(
)
{
    //  Create the address spaces for the CPUs.
    m_pSpaceGame  = m_spaceList.add( new AddrSpace( "Game" ) );
    m_pSpaceSound = m_spaceList.add( new AddrSpace( "Sound" ) );

    //  Create temporary buffers for decoding the graphics.
    m_pBufferChar   = m_tempBufferList.add( new Buffer( "Char", 0x1000 ) );
    m_pBufferSprite = m_tempBufferList.add( new Buffer( "Sprite", 0x2000 ) );

    //  Some convenient pointers.
    m_pbVideoRAM   = m_pSpaceGame->getBuffer( ) + 0x0000;
    m_pbColourRAM  = m_pSpaceGame->getBuffer( ) + 0x0400;
    m_pbSprite1RAM = m_pSpaceGame->getBuffer( ) + 0x0f80;
    m_pbSprite2RAM = m_pSpaceGame->getBuffer( ) + 0x1780;
    m_pbSprite3RAM = m_pSpaceGame->getBuffer( ) + 0x1f80;
    m_pbSoundRAM   = m_pSpaceSound->getBuffer( ) + 0x0000;
}

void
GameBallyMidway6809::startUpCPU(
)
{
    //  We interleave the CPUs at the rate of 100 times per frame to allow
    //  proper communication.
    m_nCPUInterleave = 100;

    //  This hardware uses two 6809's, one for the game and one for the Sound.
    //  Note, that we don't identify the second as a sound CPU since the
    //  game will not run if the sound CPU is disabled.
    m_pCPUGame = m_cpuList.add( 
        createCPU( "Game",  CPU::CPU_6809, m_pSpaceGame )
    );
    m_pCPUSound = m_cpuList.add( 
        createCPU( "Sound", CPU::CPU_6809, m_pSpaceSound )
    );
    
    //  The CPU's both run at 1.1Mhz.
    m_pCPUGame->setClockSpeed( 1100000 );
    m_pCPUSound->setClockSpeed( 1100000 );
     
    //  The first CPU uses IRQ interrupts, the second CPU does not have
    //  interrupts.
    m_pCPUGame->interruptHandler( s_irqInterrupt );
    m_pCPUSound->interruptHandler( s_noInterrupt );
}

void
GameBallyMidway6809::startUpInput(
)
{
    //  Create the inputs needed.
    m_pInputDSW0 = m_inputList.add( new Input( "DSW0", 0x00 ) );
    m_pInputDSW1 = m_inputList.add( new Input( "DSW1", 0x00 ) );
    m_pInputIN0  = m_inputList.add( new Input( "IN0",  0x00 ) );
    m_pInputIN1  = m_inputList.add( new Input( "IN1",  0x00 ) );
    m_pInputIN2  = m_inputList.add( new Input( "IN2",  0x00 ) );

    //  The controls are common throughout the games.
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Up",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_UP, 
            m_pInputIN0,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Right",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_RIGHT, 
            m_pInputIN0,
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Down",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_DOWN, 
            m_pInputIN0,
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Left",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_LEFT, 
            m_pInputIN0,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Action 1",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION1, 
            m_pInputIN0,
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Action 2",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION2, 
            m_pInputIN0,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Up",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_UP, 
            m_pInputIN2,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Right",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_RIGHT, 
            m_pInputIN2,
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Down",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_DOWN, 
            m_pInputIN2,
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Left",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_LEFT, 
            m_pInputIN2,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Action 1",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION1, 
            m_pInputIN2,
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Action 2",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION2, 
            m_pInputIN2,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin Left",
            CtrlMap::PLAYER1,
            CtrlMap::COIN2,
            m_pInputIN1,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin Right",
            CtrlMap::PLAYER1,
            CtrlMap::COIN1,
            m_pInputIN1,
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START,
            m_pInputIN1,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_pInputIN1,
            0x20
        )
    );

    //  The dip switches hooked to the custom I/O are common.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "IN1", m_pInputIN1->getLocation( ), FALSE )
    );
    pDip->addSetting( "Orientation", 0x40, "Upright", "Cocktail" );
    pDip->addSetting( "Service", 0x80, "Service Mode Off", "Service Mode On" );
}

void
GameBallyMidway6809::startUpGraphics(
)
{
    //  Bally Midway games must have a 224x288 screen.  We also request a 
    //  double buffer of the same size for storing characters.
    m_pScreen->setScreenSize( 224, 288, 1, 1 );

    //  Allocate a buffer that will be used to indicate which
    //  characters need to be redrawn.
    m_pDirty = m_dirtyList.add( new Buffer( "Dirty", 0x0400, TRUE ) );

    //  Map the static character positions on the temporary bitmap.
    m_pScreen->getTempBitmap( )->setMapSize( 0x400 );
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        if( ( dwI / 32 ) < 2 )
        {
            m_pScreen->getTempBitmap( )->setMapping( 
                dwI, ( 29 - ( dwI % 32 ) ) * 8, ( ( dwI / 32 ) + 34 ) * 8
            );
        }
        else
        if( ( dwI / 32 ) > 29 )
        {
            m_pScreen->getTempBitmap( )->setMapping(
                dwI, ( 29 - ( dwI % 32 ) ) * 8, ( ( dwI / 32 ) - 30 ) * 8
            );
        }
        else
        {
            m_pScreen->getTempBitmap( )->setMapping(
                dwI, ( 29 - ( dwI / 32 ) ) * 8, ( ( dwI % 32 ) + 2 ) * 8
            );
        } 
    }

    //  Create the gfxset to hold the characters and set the information.
    m_pGfxSetChar = m_gfxSetList.add( new GfxSet( "Characters" ) );
    m_pGfxSetChar->setNumber( 256 );
    m_pGfxSetChar->setDimensions( 8, 8 );
    m_pGfxSetChar->setBPP( 2 );
    m_pGfxSetChar->setBitPlanes( 0, 4 );
    m_pGfxSetChar->setXBits(  56, 48, 40, 32, 24, 16,  8,  0 );
    m_pGfxSetChar->setYBits(  64, 65, 66, 67,  0,  1,  2,  3 );
    m_pGfxSetChar->setIncrement( 128 );

    //  Decode the foreground characters.
    m_pGfxSetChar->decode( m_pBufferChar->getBuffer( ) );
    

    //  Create the gfxset to hold the sprites and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprites" ) );
    m_pGfxSetSprite->setNumber( 128 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 2 );
    m_pGfxSetSprite->setBitPlanes( 0, 4 );
    m_pGfxSetSprite->setXBits(  
        312, 304, 296, 288, 280, 272, 264, 256, 
         56,  48,  40,  32,  24,  16,   8,   0 
    );
    m_pGfxSetSprite->setYBits(  
          0,   1,   2,   3,  64,  65,  66,  67, 
        128, 129, 130, 131, 192, 193, 194, 195
    );
    m_pGfxSetSprite->setIncrement( 512 );

    //  Decode the foreground characters.
    m_pGfxSetSprite->decode( m_pBufferSprite->getBuffer( ) );
}

void
GameBallyMidway6809::startUpColour(
)
{
    ASSERT( getColourPROM( ) != NULL );
    Byte* pbColourPROM = getColourPROM( )->getBuffer( );

    //  Bally Midway hardware games use a PROM to indicate the static palette.
    //  The lookup table PROMs for the characters and sprites are appended
    //  to the end of the colour PROM.
    //
    //  The PROM is hooked up to RGB output in the following manner:
    //
    //  Bit       Resistor               Colour     State
    //   0  /\/\/   1 kohm  \/\/\/\/\/\/\  Red      normal
    //   1  /\/\/ 470  ohm  \/\/\/\/\/\/\  Red      normal
    //   2  /\/\/ 220  ohm  \/\/\/\/\/\/\  Red      normal
    //   3  /\/\/   1 kohm  \/\/\/\/\/\/\  Green    normal
    //   4  /\/\/ 470  ohm  \/\/\/\/\/\/\  Green    normal
    //   5  /\/\/ 220  ohm  \/\/\/\/\/\/\  Green    normal
    //   6  /\/\/ 470  ohm  \/\/\/\/\/\/\  Blue     normal
    //   7  /\/\/ 220  ohm  \/\/\/\/\/\/\  Blue     normal
    //
    for( int32 nI = 31 ; nI >= 0 ; nI -= 1 )
    {
        //  Used to calculate the palette entries.
        Byte bBits[ 3 ];

        //  The RGB components.
        DWord dwRed; 
        DWord dwGreen; 
        DWord dwBlue; 

        //  Calculate the colour components.
        bBits[ 0 ] = ( *pbColourPROM >> 0 ) & 0x01;
        bBits[ 1 ] = ( *pbColourPROM >> 1 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 2 ) & 0x01;
        dwRed      = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

        bBits[ 0 ] = ( *pbColourPROM >> 3 ) & 0x01;
        bBits[ 1 ] = ( *pbColourPROM >> 4 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 5 ) & 0x01;
        dwGreen    = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

        bBits[ 0 ] = 0;
        bBits[ 1 ] = ( *pbColourPROM >> 6 ) & 0x01;
        bBits[ 2 ] = ( *pbColourPROM >> 7 ) & 0x01;
        dwBlue     = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];
 
        //  Set the colour.
        m_pReplay->getCanvas( )->setColour( nI, dwRed, dwGreen, dwBlue );
        pbColourPROM += 1;
    }

    //  The character lookup table consists of 64 entries of 4 bytes (2bpp).
    m_pColTableChar = 
        m_colourTableList.add( new ColourTable( "ColTableChar", 64, 4 ) );
    for( DWord dwI = 0 ; dwI < 64 * 4 ; dwI += 1 )
    {
        ( *m_pColTableChar )[ dwI ] = *pbColourPROM & 0x0f;
        pbColourPROM += 1;
    }

    //  The sprite lookup table consists of 64 entries of 4 bytes (2bpp).
    m_pColTableSprite = 
        m_colourTableList.add( new ColourTable( "ColTableSprite", 64, 4 ) );
    for( DWord dwI = 0 ; dwI < 64 * 4 ; dwI += 1 )
    {
        ( *m_pColTableSprite )[ dwI ] = 0x1f - ( *pbColourPROM & 0x0f );
        pbColourPROM += 1;
    }
}

void
GameBallyMidway6809::startUpSound(
)
{
    //  Dual 6809 Bally Midway hardware uses a waveform sound device.
    //  It uses a sample rate of 23920, 8 voices and 48 gain.
    m_pWaveForm = ( WaveForm* )m_soundDeviceList.add( 
        new WaveForm( "WF", 23920, 8, 48 ) 
    );
}



                        
///////////////////////////////////////////////////////////////////////////////
//  The following macro is a convenience for drawing a character in
//  updateDisplay().
///////////////////////////////////////////////////////////////////////////////
#define DRAW( SPRITE, X, Y )                                                  \
    pScreen->blit(                                                            \
        ( *m_pGfxSetSprite )[ SPRITE ],                                       \
        X,                                                                    \
        Y,                                                                    \
        m_pColTableSprite->getEntry( nColour ),                               \
        bFlipX,                                                               \
        bFlipY,                                                               \
        pScreen->getFullClipping( ),                                          \
        Bitmap::TRANSPARENCY_MAP,                                             \
        0x10                                                                  \
    );
///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBallyMidway6809::updateDisplay(
    Bitmap* pScreen,
    Bitmap* pTempScreen
)
{
    //  A pointer to the dirty buffer.
    Byte* pbDirty = m_pDirty->getBuffer( );


    //  Draw any dirty characters.
    //  There are 28x36=1008 characters that make up the background,
    //  However, the video RAM is stored in a contiguous 1024 block.  For
    //  simplicity we go ahead and draw the other 16 characters which will 
    //  end up off screen and therefore omitted by the blitting routines.
    for( DWord dwI = 0 ; dwI < 0x0400 ; dwI += 1 )
    {
        //  Draw the characters onto the temporary bitmap if it is dirty.
        if( pbDirty[ dwI ] )
        {
            //  Reset the dirty flag.
            pbDirty[ dwI ] = FALSE;

            //  Draw the character into the double buffer.
            if( m_bScreenFlippedX )
            {
                pTempScreen->blit(
                    ( *m_pGfxSetChar )[ m_pbVideoRAM[ dwI ] ],
                    pTempScreen->getMappingX( 0x3ff - dwI ),
                    pTempScreen->getMappingY( 0x3ff - dwI ),
                    m_pColTableChar->getEntry( m_pbColourRAM[ dwI ] & 0x3f ),
                    TRUE,
                    TRUE,
                    pTempScreen->getFullClipping( ),
                    Bitmap::TRANSPARENCY_NONE
                );
            }
            else
            {
                pTempScreen->blit(
                    ( *m_pGfxSetChar )[ m_pbVideoRAM[ dwI ] ],
                    dwI, 
                    m_pColTableChar->getEntry( m_pbColourRAM[ dwI ] & 0x3f )
                );
            }
        }
    }

    //  Copy the temporary bitmap onto the screen.
    pScreen->copy( pTempScreen );

    //  Draw the sprites.  There are 64 sprites each using 6 bytes.  The
    //  6 bytes are split into pairs and grouped in three separate RAM areas.
    //
    //  As an added bonus, the sprites can be doubled in the horizontal
    //  direction, the vertical direction or both.
    for( DWord dwI = 0 ; dwI < 64 * 2 ; dwI += 2 )
    {
        //  We only draw it if it's enabled.
        if( ( m_pbSprite3RAM[ dwI + 1 ] & 0x02 ) == 0 )
        {
            //  For convenience, calculate the attributes now.
            int  nSprite = m_pbSprite1RAM[ dwI ] & 0x7f;
            int  nColour = m_pbSprite1RAM[ dwI + 1 ] & 0x3f;
            int  nX      = m_pbSprite2RAM[ dwI ] - 17;
            int  nY      = 
                ( m_pbSprite2RAM[ dwI + 1 ] - 40 ) + 0x100 * 
                ( m_pbSprite3RAM[ dwI + 1 ] & 0x01 );
            Byte bFlipX  = m_pbSprite3RAM[ dwI ] & 0x02;
            Byte bFlipY  = m_pbSprite3RAM[ dwI ] & 0x01;
            if( m_bScreenFlippedX )
            {
                bFlipX ^= 0x02; 
                bFlipY ^= 0x01; 
            }

            //  Check the sizing of the sprite.
            switch( m_pbSprite3RAM[ dwI ] & 0x0c )
            {
                //  Your regular sized, every-day sprite.
                case 0x00:
                {
                    DRAW( nSprite, nX, nY );
                    break;
                }
                  
                //  Double vertical.
                case 0x04:
                {
                    nSprite &= 0xfe;
                    if( !bFlipY )
                    {
                        DRAW( nSprite, nX, nY );
                        DRAW( nSprite + 1, nX, nY + 16 );
                    }
                    else
                    {
                        DRAW( nSprite, nX, nY + 16 );
                        DRAW( nSprite + 1, nX, nY );
                    }
                    break; 
                }

                //  Double horizontal.
                case 0x08:
                {
                    nSprite &= 0xfd;
                    if( bFlipX )
                    {
                        DRAW( nSprite + 2, nX, nY );
                        DRAW( nSprite, nX + 16, nY );
                    }
                    else
                    {
                        DRAW( nSprite, nX, nY );
                        DRAW( nSprite + 2, nX + 16, nY );
                    }
                    break;
                }

                //  Double horizontal & vertical.
                case 0x0c:
                {
                    nSprite &= 0xfc;
                    if( !bFlipX && !bFlipY )
                    {
                        DRAW( nSprite + 2, nX, nY );
                        DRAW( nSprite + 3, nX, nY + 16 );
                        DRAW( nSprite,     nX + 16, nY );
                        DRAW( nSprite + 1, nX + 16, nY + 16 );
                    }
                    else
                    if( bFlipX && bFlipY )
                    {
                        DRAW( nSprite + 1, nX, nY );
                        DRAW( nSprite, nX, nY + 16 );
                        DRAW( nSprite + 3, nX + 16, nY );
                        DRAW( nSprite + 2, nX + 16, nY + 16 );
                    }
                    else
                    if( bFlipX )
                    {
                        DRAW( nSprite, nX, nY );
                        DRAW( nSprite + 1, nX, nY + 16 );
                        DRAW( nSprite + 2, nX + 16, nY );
                        DRAW( nSprite + 3, nX + 16, nY + 16 );
                    }
                    else
                    {
                        DRAW( nSprite + 3, nX, nY );
                        DRAW( nSprite + 2, nX, nY + 16 );
                        DRAW( nSprite + 1, nX + 16, nY );
                        DRAW( nSprite, nX + 16, nY + 16 );
                    }
                    break;
                }
            }
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSoundEnable
//
//  Description:
//
//      This function is called when the sound is enabled or disabled.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GameBallyMidway6809::s_writeSoundEnable(
    DWord         dwAddress, 
    Byte          /* bValue is unused */, 
    WriteHandler* pHandler
)
{
    //  The waveform object is held in the handler data.
    WaveForm* pWaveForm = ( WaveForm* )( pHandler->getData( ) );

    pWaveForm->enable( dwAddress & 0x01 );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSoundRegister
//
//  Description:
//
//      This function is called when one of the sound registers is written to.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GameBallyMidway6809::s_writeSoundRegister(
    DWord         dwAddress, 
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameBallyMidway6809* pThis = 
        ( GameBallyMidway6809* )( pHandler->getData( ) );

    //  Determine the voice and the register.
    DWord dwVoice    = ( dwAddress & 0x3f ) / 8;
    Byte* pbVoiceRAM = pThis->m_pbSoundRAM + dwVoice * 8;
    Byte  bReg       = dwAddress & 0x07;

    pbVoiceRAM[ bReg ] = bValue;

    //  Update the affected registers.
    switch( bReg )
    {
        //  Volume.
        case 0x03:
        {
            pThis->m_pWaveForm->setVolume( dwVoice, bValue );
            break;
        }

        //  Waveform.
        case 0x06:
        {
            pThis->m_pWaveForm->setWave( 
                dwVoice,
                pThis->getSoundPROM( )->getBuffer( ) + 
                    32 * ( ( bValue >> 4 ) & 0x07 )
            );
            //  Fall through.
        }

        //  Frequency.
        case 0x04:
        case 0x05:
        {
            DWord dwFrequency = pbVoiceRAM[ 6 ] & 0x0f;
            dwFrequency = dwFrequency * 256 + pbVoiceRAM[ 5 ];
            dwFrequency = dwFrequency * 256 + pbVoiceRAM[ 4 ];
            pThis->m_pWaveForm->setFrequency( dwVoice, dwFrequency );
            break;
        }

        //  The rest are not used.
        default:
        {
            break;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This is called when a value is written to the location 
//      that causes the screen to rotate.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBallyMidway6809::s_writeScreenRotate(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameBallyMidway6809* pThis = 
        ( GameBallyMidway6809* )( pHandler->getData( ) );

    //  If the status of the screen has changed then flip it.
    if( 
        ( bValue && !( pThis->m_bScreenFlippedX ) ) || 
        ( !bValue && pThis->m_bScreenFlippedX ) 
    )
    {
        //  Mark the screen as dirty.
        pThis->m_pDirty->clear( TRUE ); 

        //  Remember the state of the screen.
        pThis->m_bScreenFlippedX = bValue;
        pThis->m_bScreenFlippedY = bValue;
    }
    
    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}
