//////////////////////////////////////////////////////////////////////////////
//
//  File:    _bnj.cpp
//
//  Class:   GameBumpNJumpBase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for the various 
//      versions of Bump'N'Jump.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "_bnj.h"
#include "space.h"
#include "input.h"
#include "dip.h"
#include "gfxset.h"
#include "hiscore.h"
#include "ay8910.h"
#include "bitmap.h"
#include "ctable.h"
#include "ctrlmap.h"
#include "screen.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameBumpNJumpBase
//
//  Description:
//
//      This is the main constructor for a Bump'n'Jump game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBumpNJumpBase::GameBumpNJumpBase(
    const KString& iName
)
:
    GameDataEastIGS   ( iName, TRUE, -0x0800, FALSE ),
    m_pDirty          ( NULL ),
    m_pbBGRAM         ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameBumpNJumpBase
//
//  Description:
//
//      This is the destructor for the Bump'n'Jump game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBumpNJumpBase::~GameBumpNJumpBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of Bump'n'Jump.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBumpNJumpBase::startUpSpace(
)
{
    //  Let the base class create our Game & Sound maps.
    GameDataEastIGS::startUpSpace( );

    //  Assign convenience pointers for various items.
    m_pbVideoLoRAM = m_pSpaceGame->getBuffer( ) + 0x4000;
    m_pbVideoHiRAM = m_pSpaceGame->getBuffer( ) + 0x4400;
    m_pbBGRAM      = m_pSpaceGame->getBuffer( ) + 0x5000;
    m_pbScrollHi   = m_pSpaceGame->getBuffer( ) + 0x5400;
    m_pbScrollLo   = m_pSpaceGame->getBuffer( ) + 0x5800;
}

void
GameBumpNJumpBase::startUpInput(
)
{
    //  The base class will create the inputs and setup the controls.
    GameDataEastIGS::startUpInput( );

    //  The start buttons are mapped to different bits depending on the game.
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START,
            m_pInputCoin,
            0x08
        )
    );
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_pInputCoin,
            0x10
        )
    );


    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW1", m_pInputDip1->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Coin Slot 1", 
        0x03,
        "1 Coin/1 Credit (Slot 1)",
        "1 Coin/2 Credits (Slot 1)",
        "1 Coin/3 Credits (Slot 1)",
        "2 Coins/1 Credit (Slot 1)"
    );
    pDip->addSetting(
        "Coin Slot 2", 
        0x0c,
        "1 Coin/1 Credit (Slot 2)",
        "1 Coin/2 Credits (Slot 2)",
        "1 Coin/3 Credits (Slot 2)",
        "2 Coins/1 Credit (Slot 2)"
    );
    pDip->addSetting(
        "Diagnostic",
        0x30,
        "Game Mode",
        "Normal Test Procedure",
        "No Effect",
        "Video PCB Test"
    );
    pDip->addSetting(
        "Cabinet",
        0x40,
        "Cocktail Cabinet",
        "Upright Cabinet"
    );
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW2", m_pInputDip2->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Cars",
        0x01,
        "3 Cars",
        "5 Cars"
    );
    pDip->addSetting(
        "Bonus Car",
        0x06,
        "Bonus Every 30000",
        "Bonus Every 70000",
        "Bonus At 20000",
        "Bonus At 30000"
    );
    pDip->addSetting(
        "Continue",
        0x08,
        "Normal Mode",
        "Continue Mode"
    );
    pDip->addSetting(
        "Difficulty",
        0x10,
        "Easy",
        "Hard"
    );


    //  The Bump'n'Jump high scores are stored at 0x0500 and occupy
    //  640 bytes.  The current high score is stored from 0x000a-0x000c.
    //  Bump'n'Jump first clears the area and then writes
    //  the power-up scores in, so we trigger on two writes.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 2 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x000a,   3 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0500, 640 );
}

void
GameBumpNJumpBase::startUpGraphics(
)
{
    //  The base class sets the screen size and creates the default clipping
    //  region.
    GameDataEastIGS::startUpGraphics( );


    //  Allocate a buffer that will be used to indicate which background
    //  characters need to be redrawn.
    m_pDirty = m_dirtyList.add( new Buffer( "BG Dirty", 0x0200, TRUE ) );


    //  Map the static character positions on the temporary bitmap.
    m_pScreen->getTempBitmap( )->setMapSize( 0x200 );
    for( DWord dwI = 0 ; dwI < 0x200 ; dwI += 1 )
    {
        //  Calculate the coordinates of the character.
        m_pScreen->getTempBitmap( )->setMapping(
            dwI,
            ( ( ( dwI % 0x100 ) < 0x80 ) ? dwI % 8 : ( dwI % 8 ) + 8 ) * 16,
            ( 
                ( dwI < 0x100 ) ? 
                ( ( dwI % 0x80 ) / 8 ) : ( ( dwI % 0x80 ) / 8 ) + 16 
            ) * 16
        );
    }
    

    //  Create the gfxset to hold the foreground characters and set the
    //  information.
    m_pGfxSetFG = m_gfxSetList.add( new GfxSet( "Foreground Chars" ) );
    m_pGfxSetFG->setNumber( 1024 );
    m_pGfxSetFG->setDimensions( 8, 8 );
    m_pGfxSetFG->setBPP( 3 );
    m_pGfxSetFG->setBitPlanes( 131072, 65536, 0 );
    m_pGfxSetFG->setXBits(  0,  8, 16, 24, 32, 40, 48, 56 );
    m_pGfxSetFG->setYBits(  7,  6,  5,  4,  3,  2,  1,  0 );
    m_pGfxSetFG->setIncrement( 64 );

    //  Decode the foreground characters.
    m_pGfxSetFG->decode( m_pBufferFG->getBuffer( ) );
    

    //  Create the gfxset to hold the background characters and set the
    //  information.
    m_pGfxSetBG = m_gfxSetList.add( new GfxSet( "Background Chars" ) );
    m_pGfxSetBG->setNumber( 64 );
    m_pGfxSetBG->setDimensions( 16, 16 );
    m_pGfxSetBG->setBPP( 3 );
    m_pGfxSetBG->setBitPlanes( 32772, 0, 4 );
    m_pGfxSetBG->setXBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetBG->setYBits(  
          3,   2,   1,   0, 131, 130, 129, 128,
        259, 258, 257, 256, 387, 386, 385, 384
    );
    m_pGfxSetBG->setIncrement( 512 );

    //  Decode the background characters.
    m_pGfxSetBG->decode( m_pBufferBG->getBuffer( ) );
    

    //  Create the gfxset to hold the sprites and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprites" ) );
    m_pGfxSetSprite->setNumber( 256 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 3 );
    m_pGfxSetSprite->setBitPlanes( 131072, 65536, 0 );
    m_pGfxSetSprite->setXBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetSprite->setYBits(  
          7,   6,   5,   4,   3,   2,   1,   0,
        135, 134, 133, 132, 131, 130, 129, 128
    );
    m_pGfxSetSprite->setIncrement( 256 );

    //  Decode the background characters.
    m_pGfxSetSprite->decode( m_pBufferFG->getBuffer( ) );
}

void
GameBumpNJumpBase::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x0000, 0x07ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Vid/Col RAM",    0x4000, 0x47ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Mirror V/C RAM", 0x4800, 0x4fff, s_readMirror,       this
    );
    m_pCPUGame->readMemHandler( 
        "ROM",            0xa000, 0xffff, s_readROM,          NULL 
    );
    m_pCPUGame->readMemHandler( 
        "Dip Switch 1",   0x1000, 0x1000, s_readInput,        m_pInputDip1
    );
    m_pCPUGame->readMemHandler( 
        "Dip Switch 2",   0x1001, 0x1001, s_readInput,        m_pInputDip2
    );
    m_pCPUGame->readMemHandler( 
        "P1 Controls",    0x1002, 0x1002, s_readInput,        m_pInputP1
    );
    m_pCPUGame->readMemHandler( 
        "P2 Controls",    0x1003, 0x1003, s_readInput,        m_pInputP2
    );
    m_pCPUGame->readMemHandler( 
        "Coin Controls",  0x1004, 0x1004, s_readInput,        m_pInputCoin
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x0640, 0x0640, s_writeHiScore,      m_pHiScore
    );
    m_pCPUGame->writeMemHandler(
        "Palette RAM",    0x5c00, 0x5c0f, s_writePalette,      NULL
    );
    m_pCPUGame->writeMemHandler(
        "Scratch RAM",    0x0000, 0x07ff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Vid/Col RAM",    0x4000, 0x47ff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Mirror V/C RAM", 0x4800, 0x4fff, s_writeMirror,       this
    );
    m_pCPUGame->writeMemHandler(
        "BG RAM",         0x5000, 0x51ff, s_writeDirtyByte,    
        m_pDirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "Scroll Hi",      0x5400, 0x5400, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Scroll Lo",      0x5800, 0x5800, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Screen Rotate",  0x1001, 0x1001, s_writeScreenRotate, this
    );
    m_pCPUGame->writeMemHandler(
        "Sound Command",  0x1002, 0x1002, s_writeSoundCmd,     this
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0xb000, 0xffff, s_writeROM,          NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "Scratch RAM",    0x0000, 0x03ff, s_readRAM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "ROM",            0xf000, 0xffff, s_readROM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "Sound Command",  0xa000, 0xa000, s_readCommand,      m_pSoundCommand
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "RAM",            0x0000, 0x03ff, s_writeRAM,         NULL
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Write",   0x2000, 0x2000, s_write8910Write,   m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Control", 0x4000, 0x4000, s_write8910Control, m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Write",   0x6000, 0x6000, s_write8910Write,   m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Control", 0x8000, 0x8000, s_write8910Control, m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "IEnable",        0xc000, 0xc000, s_writeIntEnable,   m_pCPUSound
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0xf000, 0xffff, s_writeROM,         NULL
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBumpNJumpBase::updateDisplay(
    Bitmap* pScreen,
    Bitmap* pTempScreen
)
{
    //  If the high scroll register has a value then draw the background 
    //  taking into account the scroll registers, otherwise just clear the
    //  background.
    if( *m_pbScrollHi )
    {
        //  A pointer to the dirty buffer.
        Byte* pbDirty = m_pDirty->getBuffer( );
    
    
        //  Draw any dirty background characters.
        //  There are 16x32=512 characters that make up the background.
        for( DWord dwI = 0 ; dwI < 0x0200 ; dwI += 1 )
        {
            //  Draw the characters onto the temporary bitmap if it is dirty.
            if( pbDirty[ dwI ] )
            {
                //  Reset the dirty flag.
                pbDirty[ dwI ] = FALSE;
    
                //  Draw the character.
                pTempScreen->blit(
                    ( *m_pGfxSetBG )[ 
                        ( m_pbBGRAM[ dwI ] >> 4 ) + ( dwI & 0x80 ? 48 : 32 ) 
                    ],
                    dwI,
                    *m_pColTableBG
                );
            }
        }

        pScreen->blitScrollFull( 
            pTempScreen, 
            0, 
            ( *m_pbScrollHi & 0x02 ) * 128 + 511 - *m_pbScrollLo,
            pScreen->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        ); 
    }
    else
    {
        pScreen->clear( );
    }

    //  Draw the sprites and then the foreground characters.
    updateSprites( pScreen );
    updateCharacters( pScreen );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This is called when a value is written to the location 
//      that causes the screen to rotate.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBumpNJumpBase::s_writeScreenRotate(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameBumpNJumpBase* pThis = ( GameBumpNJumpBase* )( pHandler->getData( ) );

    //  For some reason that I'm obviously overlooking, it seems like 
    //  Bump'n'Jump *always* rotates the screen when in two player mode, 
    //  whether cocktail mode is on or not. 
    //  Therefore, we first check whether we are in cocktail mode before
    //  calling the default handler.
    if( pThis->m_pInputDip1->getValue( ) & 0x40 )
    {
        Game::s_writeScreenRotate( dwAddress, bValue, pHandler );
    }
}
