///////////////////////////////////////////////////////////////////////////////
//
//  File:    _btime.cpp
//
//  Class:   GameBurgerTimeBase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for the various
//      versions of Burgertime.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "_btime.h"
#include "space.h"
#include "input.h"
#include "dip.h"
#include "gfxset.h"
#include "hiscore.h"
#include "bitmap.h"
#include "ctable.h"
#include "appfile.h"
#include "ctrlmap.h"
#include "screen.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameBurgerTimeBase
//
//  Description:
//
//      This is the main constructor for a BurgerTime game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pwAddresses (input)
//          A pointer to the addresses that contain encrypted opcodes.
//
//      dwNumAddresses (input)
//          The number of encrypted opcode addresses contained in pwAddresses.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBurgerTimeBase::GameBurgerTimeBase(
    const KString& iName,
    Word*          pwAddresses,
    DWord          dwNumAddresses
)
:
    GameDataEastIGS  ( iName, FALSE, -0x0800, TRUE ),
    m_pwAddresses    ( pwAddresses ),
    m_dwNumAddresses ( dwNumAddresses ),
    m_pBufferMap     ( NULL ),
    m_bMapNumber     ( 7 ),
    m_bMapDirty      ( TRUE )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameBurgerTimeBase
//
//  Description:
//
//      This is the destructor for the BurgerTime game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameBurgerTimeBase::~GameBurgerTimeBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of Burgertime.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBurgerTimeBase::startUpSpace(
)
{
    //  Let the base class create our Game & Sound maps.
    GameDataEastIGS::startUpSpace( );

    //  Create the buffer for the background map graphics.
    m_pBufferMap = m_bufferList.add( new Buffer( "Map", 0x800 ) );


    //  Assign convenience pointers for important items.
    m_pbVideoLoRAM = m_pSpaceGame->getBuffer( ) + 0x1000;
    m_pbVideoHiRAM = m_pSpaceGame->getBuffer( ) + 0x1400;
}

void
GameBurgerTimeBase::startUpInput(
)
{
    //  The base class will create the inputs and setup the controls.
    GameDataEastIGS::startUpInput( );

    //  We need to adjust a couple of the initial values for inputs that 
    //  do not initialize to 0xff.
    m_pInputCoin->setInitial( 0x3f );
    m_pInputDip1->setInitial( 0x3f );
    
    //  The start buttons are mapped to different bits depending on the game.
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START,
            m_pInputCoin,
            0x01
        )
    );
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_pInputCoin,
            0x02
        )
    );

    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW1", m_pInputDip1->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Coin Slot 1", 
        0x03,
        "1 Coin/1 Credit (Slot 1)",
        "1 Coin/2 Credits (Slot 1)",
        "1 Coin/3 Credits (Slot 1)",
        "2 Coins/1 Credit (Slot 1)"
    );
    pDip->addSetting(
        "Coin Slot 2", 
        0x0c,
        "1 Coin/1 Credit (Slot 2)",
        "1 Coin/2 Credits (Slot 2)",
        "1 Coin/3 Credits (Slot 2)",
        "2 Coins/1 Credit (Slot 2)"
    );
    pDip->addSetting(
        "Diagnostic",
        0x30,
        "Game Mode",
        "Self Test",
        "Cross Hatch",
        "Sound Check"
    );
    pDip->addSetting(
        "Cabinet",
        0x40,
        "Cocktail Cabinet",
        "Upright Cabinet"
    );

    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW2", m_pInputDip2->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Chefs",
        0x01,
        "3 Chefs",
        "5 Chefs"
    );
    pDip->addSetting(
        "Bonus Chef",
        0x06,
        "Bonus Chef at 10000",
        "Bonus Chef at 15000",
        "Bonus Chef at 20000",
        "Bonus Chef at 30000"
    );
    pDip->addSetting(
        "Pursuers",
        0x08,
        "4 Pursuers",
        "6 Pursuers"
    );
    pDip->addSetting(
        "End Level Pepper",
        0x10,
        "No End of Level Pepper",
        "End of Level Pepper"
    );



    //  The Burgertime high scores are stored at:
    //      0x0033: 1 top score * 3 bytes/score = 3 bytes.
    //      0x0036: 5 names * 3 bytes/name = 15 bytes.
    //      0x0048: 5 scores * 3 bytes/score = 15 bytes.
    //  Burgertime writes to this area twice before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 2 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0033, 3 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0036, 15 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0048, 15 );
}

void
GameBurgerTimeBase::startUpGraphics(
)
{
    //  The base class sets the screen size and creates the default clipping
    //  region.
    GameDataEastIGS::startUpGraphics( );


    //  Map the static tile positions on the temporary bitmap.
    m_pScreen->getTempBitmap( )->setMapSize( 0x100 );
    for( DWord dwI = 0 ; dwI < 0x100 ; dwI += 1 )
    {
        m_pScreen->getTempBitmap( )->setMapping( 
            dwI, ( dwI % 16 ) * 16, ( dwI / 16 ) * 16, m_pClipping
        ); 
    }

    //  Create the gfxset to hold the foreground characters and set the
    //  information.
    m_pGfxSetFG = m_gfxSetList.add( new GfxSet( "Foreground Chars" ) );
    m_pGfxSetFG->setNumber( 1024 );
    m_pGfxSetFG->setDimensions( 8, 8 );
    m_pGfxSetFG->setBPP( 3 );
    m_pGfxSetFG->setBitPlanes( 0, 65536, 131072 );
    m_pGfxSetFG->setXBits(  0,  8, 16, 24, 32, 40, 48, 56 );
    m_pGfxSetFG->setYBits(  7,  6,  5,  4,  3,  2,  1,  0 );
    m_pGfxSetFG->setIncrement( 64 );

    //  Decode the foreground characters.
    m_pGfxSetFG->decode( m_pBufferFG->getBuffer( ) );
    

    //  Create the gfxset to hold the background characters and set the
    //  information.
    m_pGfxSetBG = m_gfxSetList.add( new GfxSet( "Background Chars" ) );
    m_pGfxSetBG->setNumber( 64 );
    m_pGfxSetBG->setDimensions( 16, 16 );
    m_pGfxSetBG->setBPP( 3 );
    m_pGfxSetBG->setBitPlanes( 0, 16384, 32768 );
    m_pGfxSetBG->setXBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetBG->setYBits(  
          7,   6,   5,   4,   3,   2,   1,   0,
        135, 134, 133, 132, 131, 130, 129, 128
    );
    m_pGfxSetBG->setIncrement( 256 );

    //  Decode the background characters.
    m_pGfxSetBG->decode( m_pBufferBG->getBuffer( ) );
    

    //  Create the gfxset to hold the sprites and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprites" ) );
    m_pGfxSetSprite->setNumber( 128 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 3 );
    m_pGfxSetSprite->setBitPlanes( 0, 65536, 131072 );
    m_pGfxSetSprite->setXBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetSprite->setYBits(  
          7,   6,   5,   4,   3,   2,   1,   0,
        135, 134, 133, 132, 131, 130, 129, 128
    );
    m_pGfxSetSprite->setIncrement( 256 );

    //  Decode the sprites.
    m_pGfxSetSprite->decode( m_pBufferFG->getBuffer( ) );
}

void
GameBurgerTimeBase::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x0000, 0x07ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Vid/Col RAM",    0x1000, 0x17ff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Mirror V/C RAM", 0x1800, 0x1fff, s_readMirror,       this
    );
    m_pCPUGame->readMemHandler( 
        "ROM",            0xb000, 0xffff, s_readROM,          NULL 
    );
    m_pCPUGame->readMemHandler( 
        "P1 Controls",    0x4000, 0x4004, s_readInputs,       this
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x0036, 0x0036, s_writeHiScore,      m_pHiScore
    );
    m_pCPUGame->writeMemHandler(
        "Palette RAM",    0x0c00, 0x0c0f, s_writePalette,      NULL
    );
    m_pCPUGame->writeMemHandler(
        "Scratch RAM",    0x0000, 0x07ff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Gfx RAM",        0x1000, 0x17ff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler( 
        "Mirror V/C RAM", 0x1800, 0x1fff, s_writeMirror,       this
    );
    m_pCPUGame->writeMemHandler(
        "Screen Rotate",  0x4002, 0x4002, s_writeScreenRotate, this
    );
    m_pCPUGame->writeMemHandler(
        "Sound Command",  0x4003, 0x4003, s_writeSoundCmd,     this
    );
    m_pCPUGame->writeMemHandler(
        "Map Number",     0x4004, 0x4004, s_writeMapNumber,    this
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0xb000, 0xffff, s_writeROM,          NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "Scratch RAM",    0x0000, 0x03ff, s_readRAM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "ROM",            0xf000, 0xffff, s_readROM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "Sound Command",  0xa000, 0xa000, s_readCommand,      m_pSoundCommand
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "RAM",            0x0000, 0x03ff, s_writeRAM,         NULL
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Write",   0x2000, 0x2000, s_write8910Write,   m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Control", 0x4000, 0x4000, s_write8910Control, m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Write",   0x6000, 0x6000, s_write8910Write,   m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Control", 0x8000, 0x8000, s_write8910Control, m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "IEnable",        0xc000, 0xc000, s_writeIntEnable,   m_pCPUSound
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0xf000, 0xffff, s_writeROM,         NULL
    );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: decryptROMs
//
//  Description:
//
//      This function is used to decrypt Burgertime Game ROMs.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBurgerTimeBase::decryptROMs(
)
{
    //  The source byte.
    Byte bSource;


    //  Burgertime uses opcode decryption.
    m_pSpaceGame->setEncryptType( AddrSpace::CRYPT_OP );


    //  The source and destination buffers.
    Byte* pbSource = m_pSpaceGame->getBuffer( );
    Byte* pbDest   = m_pSpaceGame->getDecryptBuffer( );


    //  Loop through each encrypted address.
    for( DWord dwI = 0 ; dwI < m_dwNumAddresses ; dwI += 1 )
    {
        //  BurgerTime uses a tricky encryption scheme that I haven't totally
        //  figured out yet.  Only opcodes are affected and only some of them.
        //  The encryption is fairly simple it's just a bit swap using the 
        //  following pattern:
        //      76543210 -> 65342710
        //  However, figuring out when to decrypt is the hard part.  All 
        //  encrypted opcodes fall on addresses with the following pattern:
        //      _______1 _____1__
        //  But not all opcodes on such addresses are encrypted.  Therefore, 
        //  the only way I am currently able to decrypt the ROMs is by 
        //  hand :-(.  If anyone can figure out the last piece of the puzzle, 
        //  please let me know.
        bSource = pbSource[ m_pwAddresses[ dwI ] ];
        pbDest[ m_pwAddresses[ dwI ] ] = 
            ( bSource & 0x13 ) |
            ( ( bSource & 0x80 ) >> 5) |
            ( ( bSource & 0x64 ) << 1) |
            ( ( bSource & 0x08 ) << 2);
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBurgerTimeBase::updateDisplay(
    Bitmap* pScreen,
    Bitmap* pTempScreen
)
{
    //  If the background is dirty then redraw it.
    if( m_bMapDirty )
    {
        //  Point to the start of the map ROM.
        Byte* pbMapStart = m_pBufferMap->getBuffer( ) + m_bMapNumber * 256;

        //  There are 16x16=256 characters that make up the map background.
        for( DWord dwI = 0 ; dwI < 0x0100 ; dwI += 1 )
        {
            //  Draw the characters onto the temporary bitmap.
            pTempScreen->blit(
                ( *m_pGfxSetBG )[ *pbMapStart ], dwI, *m_pColTableBG
            );

            //  Move on to the next character.
            pbMapStart += 1;
        }

        //  The map is no longer dirty.
        m_bMapDirty = FALSE;
    }

    //  Blast the temporary screen to the real screen.
    pScreen->copy( pTempScreen );

    //  Draw the foreground characters and sprites.
    updateCharacters( pScreen );
    updateSprites( pScreen );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member is called to save the current state of the game in 
//      a save game file.  This can be retrieved with the load member.
//
//  Parameters:
//
//      pSaveFile (input)
//          A pointer to the file to save to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
GameBurgerTimeBase::save( 
    AppFile* pSaveFile
)
{
    //  We always allow the base class to save itself first.
    if( Game::save( pSaveFile ) )
    {
        return( TRUE );
    }

    //  Write the map number.
    pSaveFile->write( ( Byte* )&m_bMapNumber, sizeof( m_bMapNumber ) );
   
    //  If there were errors encountered then indicate this to the client.
    return( pSaveFile->count( ) != sizeof( m_bMapNumber ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member is called to load the current state of the game from
//      a save game file that was created in the save member.
//
//  Parameters:
//
//      pLoadFile (input)
//          A pointer to the file to load from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
GameBurgerTimeBase::load( 
    AppFile* pLoadFile
)
{
    //  We need to mark the map as dirty so that it will be redrawn after
    //  the load.
    m_bMapDirty = TRUE;

    //  We always allow the base class to load itself first.
    if( Game::load( pLoadFile ) )
    {
        return( TRUE );
    }

    //  Load the map number.
    pLoadFile->read( ( Byte* )&m_bMapNumber, sizeof( m_bMapNumber ) );

    //  If there were errors encountered then indicate this to the client.
    return( pLoadFile->count( ) != sizeof( m_bMapNumber ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeMapNumber
//
//  Description:
//
//      This function is called when the map number is written to.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameBurgerTimeBase::s_writeMapNumber(
    DWord         /* dwAddress is unused */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameBurgerTimeBase* pThis = ( GameBurgerTimeBase* )( pHandler->getData( ) );

    //  The map number.
    Byte bMapNumber;


    //  The map values written need to be converted to the real values.  They
    //  are also different between the two players when in cocktail mode.
    if( pThis->m_bScreenFlippedX )
    {
        switch( bValue )
        {
            case 0x10: bMapNumber = 0x00; break;
            case 0x11: bMapNumber = 0x01; break;
            case 0x12: bMapNumber = 0x02; break;
            case 0x13: bMapNumber = 0x03; break;
            case 0x14: bMapNumber = 0x04; break;
            case 0x15: bMapNumber = 0x05; break;
            default:   bMapNumber = 0x07; break;
        }
    }
    else
    {
        switch( bValue )
        {
            case 0x13: bMapNumber = 0x00; break;
            case 0x10: bMapNumber = 0x01; break;
            case 0x11: bMapNumber = 0x02; break;
            case 0x12: bMapNumber = 0x03; break;
            case 0x17: bMapNumber = 0x04; break;
            case 0x14: bMapNumber = 0x05; break;
            default:   bMapNumber = 0x07; break;
        }
    }

    //  If the map number has changed then record the new map number.
    if( bMapNumber != pThis->m_bMapNumber )
    {
        //  The map should be redrawn.
        pThis->m_bMapDirty  = TRUE;
        pThis->m_bMapNumber = bMapNumber;
        
        //  If there is no map then we disable the sprites by placing
        //  a blank (0x3f) value in each sprite.
        if( bMapNumber == 0x07 )
        {
            for( DWord dwI = 0x01801 ; dwI < 8 * 4 ; dwI += 4 )
            {
                ( *( pThis->m_pSpaceGame ) )[ dwI ] = 0x3f;
            }
        }
    }
}
