///////////////////////////////////////////////////////////////////////////////
//
//  File:    _de-0.cpp
//
//  Class:   GameDataEast0
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for a series of
//      games by Data East that run on what is known as DE-0 hardware.
//      This hardware contains a 68000 for the game, a 6502 for sound,
//      a YM2203, YM3812 and an Oki M6295.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "_de-0.h"
#include "space.h"
#include "buffer.h"
#include "input.h"
#include "ctrlmap.h"
#include "gfxset.h"
#include "ctable.h"
#include "bitmap.h"
#include "hiscore.h"
#include "screen.h"
#include "cqueue.h"
#include "ym3812.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameDataEast0
//
//  Description:
//
//      This is the main constructor for a DE-0 game object.
//      
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameDataEast0::GameDataEast0(
    const KString& iName
)
:
    Game               ( iName ),
    m_pSpaceGame       ( NULL ),
    m_pSpaceSound      ( NULL ),
    m_pBufferChar      ( NULL ),
    m_pBufferTile1     ( NULL ),
    m_pBufferTile2     ( NULL ),
    m_pBufferSprite    ( NULL ),
    m_pCharBitmap      ( NULL ),
    m_pCharDirty       ( NULL ),
    m_pbCharScroll     ( NULL ),
    m_pbCharRAM        ( NULL ),
    m_pTile1Bitmap     ( NULL ),
    m_pTile1Dirty      ( NULL ),
    m_pbTile1Scroll    ( NULL ),
    m_pbTile1RAM       ( NULL ),
    m_pTile2Bitmap     ( NULL ),
    m_pTile2Dirty      ( NULL ),
    m_pbTile2Scroll    ( NULL ),
    m_pbTile2RAM       ( NULL ),
    m_pbRedRAM         ( NULL ),
    m_pbGreenRAM       ( NULL ),
    m_pbBlueRAM        ( NULL ),
    m_pbSpriteRAM      ( NULL ),
    m_pbPriorityRAM    ( NULL ),
    m_pCPUGame         ( NULL ),
    m_pCPUSound        ( NULL ),
    m_pGfxSetChar      ( NULL ),
    m_pGfxSetTile1     ( NULL ),
    m_pGfxSetTile2     ( NULL ),
    m_pGfxSetSprite    ( NULL ),
    m_pColTableChar    ( NULL ),
    m_pColTableTile1   ( NULL ),
    m_pColTableTile2   ( NULL ),
    m_pColTableSprite  ( NULL ),
    m_pSoundCommand    ( NULL ),
    m_pYM2203          ( NULL ),
    m_pYM3812          ( NULL ),
    m_pOkiM6295        ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameDataEast0
//
//  Description:
//
//      This is the destructor for a DE-0 game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameDataEast0::~GameDataEast0(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of the DE-0 hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEast0::startUpSpace(
)
{
    //  Create a 24 bit address space for the Game CPU.  Set this space up
    //  with pages that are used by the CPU.
    m_pSpaceGame  = m_spaceList.add( new AddrSpace( "Game", 24 ) );
    m_pSpaceGame->setSegments(
        MEM_START, 
            MEM_RANGE( 0x000000, 0x05ffff ), 
            MEM_RANGE( 0x180000, 0x18ffff ), 
            MEM_RANGE( 0x240000, 0x24ffff ), 
            MEM_RANGE( 0x300000, 0x31ffff ), 
            MEM_RANGE( 0xff0000, 0xffffff ), 
        MEM_END 
    );

    //  Create an address space for the sound CPU.
    m_pSpaceSound = m_spaceList.add( new AddrSpace( "Sound" ) );

    //  Create temporary buffers for decoding the graphics.
    m_pBufferChar   = m_tempBufferList.add( new Buffer( "Char",   0x20000 ) );
    m_pBufferTile1  = m_tempBufferList.add( new Buffer( "Tile1",  0x40000 ) );
    m_pBufferTile2  = m_tempBufferList.add( new Buffer( "Tile2",  0x20000 ) );
    m_pBufferSprite = m_tempBufferList.add( new Buffer( "Sprite", 0x60000 ) );

    //  Some convenient pointers.
    m_pbCharScroll     = m_pSpaceGame->getPointer( 0x240010 );
    m_pbCharRAM        = m_pSpaceGame->getPointer( 0x244000 );
    m_pbTile1Scroll    = m_pSpaceGame->getPointer( 0x246010 );
    m_pbTile1RAM       = m_pSpaceGame->getPointer( 0x24a000 );
    m_pbTile2Scroll    = m_pSpaceGame->getPointer( 0x24c010 );
    m_pbTile2RAM       = m_pSpaceGame->getPointer( 0x24d000 );
    m_pbRedRAM         = m_pSpaceGame->getPointer( 0x310001 );
    m_pbGreenRAM       = m_pSpaceGame->getPointer( 0x310000 );
    m_pbBlueRAM        = m_pSpaceGame->getPointer( 0x314001 );
    m_pbSpriteRAM      = m_pSpaceGame->getPointer( 0xffc000 );
    m_pbPriorityRAM    = m_pSpaceGame->getPointer( 0x30c011 );

    //  Create a command queue for communication between the CPUs.
    m_pSoundCommand = new CommandQueue( "DE0Sound", 1, TRUE );
    m_bufferList.add( m_pSoundCommand );
}

void
GameDataEast0::startUpCPU(
)
{
    //  DE-0 games use a 68000 for the game and a 6502 for the sound.
    m_pCPUGame = m_cpuList.add( 
        createCPU( "Game",  CPU::CPU_68000, m_pSpaceGame )
    );
    m_pCPUSound = m_cpuList.add( 
        createCPU( "Sound",  CPU::CPU_6502, m_pSpaceSound )
    );
    
    //  The Game CPU runs at 10Mhz and the Sound CPU runs at 1Mhz.
    m_pCPUGame->setClockSpeed( 10000000 );
    m_pCPUSound->setClockSpeed( 1250000 );
     
    //  The game CPU performs an interrupt level 6 each time through. 
    //  The sound CPU performs an IRQ 32 times per frame.
    m_pCPUGame->setIntVector( 6 );
    m_pCPUGame->interruptHandler( s_vectorInterrupt );
    m_pCPUSound->setIntsPerFrame( 16 );
    m_pCPUSound->interruptHandler( s_irqInterrupt );
}

void
GameDataEast0::startUpInput(
)
{
    //  Create the inputs needed.
    m_inputList.add( new Input( "P2 Controls", 0xff ) );
    m_inputList.add( new Input( "P1 Controls", 0xff ) );
    m_inputList.add( new Input( "Coin Mech", 0xff ) );
    m_inputList.add( new Input( "Coin Mech", 0x7f ) );
    m_inputList.add( new Input( "DSW0", 0xff ) );
    m_inputList.add( new Input( "DSW1", 0x7f ) );

    //  Setup the control maps.
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Up",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_UP, 
            m_inputList[ 1 ],
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Down",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_DOWN, 
            m_inputList[ 1 ],
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Left",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_LEFT, 
            m_inputList[ 1 ],
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Right",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_RIGHT, 
            m_inputList[ 1 ],
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Fire",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION1, 
            m_inputList[ 1 ],
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Jump",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION2, 
            m_inputList[ 1 ],
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Up",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_UP, 
            m_inputList[ 0 ],
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Down",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_DOWN, 
            m_inputList[ 0 ],
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Left",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_LEFT, 
            m_inputList[ 0 ],
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Right",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_RIGHT, 
            m_inputList[ 0 ],
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Fire",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION1, 
            m_inputList[ 0 ],
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Jump",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION2, 
            m_inputList[ 0 ],
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START,
            m_inputList[ 3 ],
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_inputList[ 3 ],
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 1",
            CtrlMap::PLAYER1,
            CtrlMap::COIN1,
            m_inputList[ 3 ],
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 2",
            CtrlMap::PLAYER1,
            CtrlMap::COIN2,
            m_inputList[ 3 ],
            0x20
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 3",
            CtrlMap::PLAYER1,
            CtrlMap::COIN3,
            m_inputList[ 3 ],
            0x40
        )
    );


    //  The high scores are stored at 0xff8ed0.  There are 176 bytes.
    //  This area is written to twice before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 2 );
    m_pHiScore->addRange( m_pSpaceGame->getPointer( 0xff8ed0 ), 176 );
}

void
GameDataEast0::startUpGraphics(
)
{
    //  DE-0 games run on a lo-res 256x256 screen.  The FPS and VBlank duration
    //  are taken from the DECO IGS games.
    m_pScreen->setScreenSize( 256, 256 );
    m_pScreen->setFPS( 57 );
    m_pScreen->setVBlank( 1536 );

    //  DE-0 games have a scrollable character layer and two scrollable 
    //  tile layers which are twice as tall and wide as the screen.
    m_pCharBitmap  = m_bitmapList.add( createBitmap( "CharBmp",  512, 512 ) );
    m_pTile1Bitmap = m_bitmapList.add( createBitmap( "Tile1Bmp", 512, 512 ) );
    m_pTile2Bitmap = m_bitmapList.add( createBitmap( "Tile2Bmp", 512, 512 ) );

    //  The character layer consists of 4096 characters and the tile layers 
    //  consist of 1024 tiles.  We allocate a dirty buffer for each layer.
    m_pCharDirty  = m_dirtyList.add( new Buffer( "CharDirty",  0x1000, TRUE ) );
    m_pTile1Dirty = m_dirtyList.add( new Buffer( "Tile1Dirty", 0x0400, TRUE ) );
    m_pTile2Dirty = m_dirtyList.add( new Buffer( "Tile2Dirty", 0x0400, TRUE ) );

    //  The characters and tiles are stored in memory as follows:
    //       +--------+--------+      +--------+--------+
    //       | 0x000- | 0x800- |      | 0x000- | 0x200- |
    //       | 0x3ff  | 0xbff  |      | 0x0ff  | 0x2ff  |
    //  Char +--------+--------+ Tile +--------+--------+
    //       | 0x400- | 0xc00- |      | 0x100- | 0x300- |
    //       | 0x7ff  | 0xfff  |      | 0x1ff  | 0x3ff  |
    //       +--------+--------+      +--------+--------+
    //  Map the static character positions on the layers.
    m_pCharBitmap->setMapSize( 0x1000 );
    m_pTile1Bitmap->setMapSize( 0x400 );
    m_pTile2Bitmap->setMapSize( 0x400 );
    for( DWord dwI = 0 ; dwI < 0x1000 ; dwI += 1 )
    {
        m_pCharBitmap->setMapping(
            dwI, 
            ( ( dwI & 0x800 ) ? 256 : 0 ) + ( ( dwI & 0x3ff ) % 32 ) * 8,
            ( ( dwI & 0x400 ) ? 256 : 0 ) + ( ( dwI & 0x3ff ) / 32 ) * 8 
        );
    }
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        m_pTile1Bitmap->setMapping(
            dwI, 
            ( ( dwI & 0x200 ) ? 0x100 : 0 ) + ( ( dwI & 0xff ) % 16 ) * 16,
            ( ( dwI & 0x100 ) ? 0x100 : 0 ) + ( ( dwI & 0xff ) / 16 ) * 16
        );
        m_pTile2Bitmap->setMapping(
            dwI, 
            ( ( dwI & 0x200 ) ? 0x100 : 0 ) + ( ( dwI & 0xff ) % 16 ) * 16,
            ( ( dwI & 0x100 ) ? 0x100 : 0 ) + ( ( dwI & 0xff ) / 16 ) * 16
        );
    }
        
    //  Create the gfxset to hold the characters and set the information.
    //  I'm pretty sure the planes are in the right order.
    m_pGfxSetChar = m_gfxSetList.add( new GfxSet( "Characters" ) );
    m_pGfxSetChar->setNumber( 4096 );
    m_pGfxSetChar->setDimensions( 8, 8 );
    m_pGfxSetChar->setBPP( 4 );
    m_pGfxSetChar->setBitPlanes( 524288, 0, 786432, 262144 );
    m_pGfxSetChar->setXBits(   0,  1,  2,  3,  4,  5,  6,  7 );
    m_pGfxSetChar->setYBits(   0,  8, 16, 24, 32, 40, 48, 56 );
    m_pGfxSetChar->setIncrement( 64 );

    //  Decode the foreground characters.
    m_pGfxSetChar->decode( m_pBufferChar->getBuffer( ) );
    

    //  Create the gfxset to hold the first tile set and set the information.
    //  I'm pretty sure the planes are in the right order.
    m_pGfxSetTile1 = m_gfxSetList.add( new GfxSet( "Tile1" ) );
    m_pGfxSetTile1->setNumber( 2048 );
    m_pGfxSetTile1->setDimensions( 16, 16 );
    m_pGfxSetTile1->setBPP( 4 );
    m_pGfxSetTile1->setBitPlanes( 1572864, 524288, 1048576, 0 );
    m_pGfxSetTile1->setXBits(  
        128, 129, 130, 131, 132, 133, 134, 135, 
          0,   1,   2,   3,   4,   5,   6,   7 
    );
    m_pGfxSetTile1->setYBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetTile1->setIncrement( 256 );

    //  Decode the first tile set.
    m_pGfxSetTile1->decode( m_pBufferTile1->getBuffer( ) );


    //  Create the gfxset to hold the second tile set and set the information.
    //  I'm pretty sure the planes are in the right order.
    m_pGfxSetTile2 = m_gfxSetList.add( new GfxSet( "Tile2" ) );
    m_pGfxSetTile2->setNumber( 1024 );
    m_pGfxSetTile2->setDimensions( 16, 16 );
    m_pGfxSetTile2->setBPP( 4 );
    m_pGfxSetTile2->setBitPlanes( 262144, 786432, 0, 524288 );
    m_pGfxSetTile2->setXBits(  
        128, 129, 130, 131, 132, 133, 134, 135, 
          0,   1,   2,   3,   4,   5,   6,   7 
    );
    m_pGfxSetTile2->setYBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetTile2->setIncrement( 256 );

    //  Decode the second tile set.
    m_pGfxSetTile2->decode( m_pBufferTile2->getBuffer( ) );



    //  Create the gfxset to hold the sprite set and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprite" ) );
    m_pGfxSetSprite->setNumber( 3072 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 4 );
    m_pGfxSetSprite->setBitPlanes( 1572864, 2359296, 0, 786432 );
    m_pGfxSetSprite->setXBits(  
        128, 129, 130, 131, 132, 133, 134, 135, 
          0,   1,   2,   3,   4,   5,   6,   7 
    );
    m_pGfxSetSprite->setYBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetSprite->setIncrement( 256 );

    //  Decode the foreground characters.
    m_pGfxSetSprite->decode( m_pBufferSprite->getBuffer( ) );
}

void
GameDataEast0::startUpColour(
)
{
    //  Each layer uses its own colour table of 16 entries of 16 values (4bpp).
    //  We add them to the colour table list in the order they are stored
    //  in memory so that we can use the address as an index into the array
    //  to get the correct table.

    //      310000-3101ff--Character.
    //      314000-3141ff/
    m_pColTableChar =
        m_colourTableList.add( new ColourTable( "CTChar", 16, 16 ) );

    //      310200-3103ff--Sprite.
    //      314200-3143ff/
    m_pColTableSprite = 
        m_colourTableList.add( new ColourTable( "CTSprite", 16, 16 ) );

    //      310400-3105ff--Tile1.
    //      314400-3145ff/
    m_pColTableTile1 = 
        m_colourTableList.add( new ColourTable( "CTTile1", 16, 16 ) );

    //      310600-3107ff--Tile2.
    //      314600-3147ff/
    m_pColTableTile2 = 
        m_colourTableList.add( new ColourTable( "CTTile2", 16, 16 ) );

    //  Initially, we set colour 0 to be transparent black and colour 1 to
    //  be non-transparent black.
    Replay::s_instance( ).getCanvas( )->setColour( 0, 0x00, 0x00, 0x00 );
    Replay::s_instance( ).getCanvas( )->setColour( 1, 0x00, 0x00, 0x00 );

    //  Initially, assign all first table entries to 0 and the remainder to 1.
    //  Lock each one down.
    for( Byte bI = 0 ; bI < m_colourTableList.entries( ) ; bI += 1 )
    {
        //  16 entries.
        for( DWord dwI = 0 ; dwI < 16 * 16 ; dwI += 1 )
        {
            DWord dwColour = ( dwI % 16 == 0 ) ? 0 : 1;
            ( *( m_colourTableList[ bI ] ) )[ dwI ] = dwColour;
            Replay::s_instance( ).getCanvas( )->lockColour( dwColour ); 
        }
    }
}

void
GameDataEast0::startUpSound(
)
{
    //  DE-0 games use a YM2203 for effects, a YM3812 for music and an
    //  Oki M6295 for samples. 
    //m_pYM2203 = m_soundDeviceList.add( new YM2203( XX, XX, XX ) );
    //m_pOkiM6295 = m_soundDeviceList.add( new OkiM6295( XX, XX, XX ) );
    m_pYM3812 = m_soundDeviceList.add( new YM3812( "Music" ) );
}

void
GameDataEast0::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "ROM",            0x000000, 0x05ffff, s_readROM,       NULL
    );
    m_pCPUGame->readMemHandler( 
        "Inputs",         0x30c000, 0x30c005, s_readInputs,    this
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Char Dirty",     0x244000, 0x245fff, s_writeDirtyWord,
        m_pCharDirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "Tile 1 Dirty",   0x24a000, 0x24a7ff, s_writeDirtyWord,
        m_pTile1Dirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "Tile 2 Dirty",   0x24d000, 0x24d7ff, s_writeDirtyWord,
        m_pTile2Dirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "Palette (G/R)",  0x310000, 0x3107ff, s_writePalette,  this
    );
    m_pCPUGame->writeMemHandler(
        "Palette (B)",    0x314000, 0x3147ff, s_writePalette,  this
    );
    m_pCPUGame->writeMemHandler(
        "Sound Cmd",      0x30c014, 0x30c015, s_writeSoundCmd, this
    );
    m_pCPUGame->writeMemHandler(
        "Screen Rotate",  0xff800c, 0xff800d, s_writeScreenRotate, this
    );
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0xff8ed4, 0xff8ed5, s_writeHiScore,  m_pHiScore
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0x000000, 0x05ffff, s_writeROM,      NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "RAM",            0x0000, 0x05ff, s_readRAM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "Sound Command",  0x3000, 0x3000, s_readCommand,      m_pSoundCommand
    );
    m_pCPUSound->readMemHandler( 
        "ROM",            0x8000, 0xffff, s_readROM,          NULL
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "RAM",            0x0000, 0x05ff, s_writeRAM,         NULL
    );
    m_pCPUSound->writeMemHandler(
        "YM3812 Control", 0x1000, 0x1000, s_write3812Control, m_pYM3812
    );
    m_pCPUSound->writeMemHandler(
        "YM3812 Write",   0x1001, 0x1001, s_write3812Write,   m_pYM3812
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0x008000, 0x00ffff, s_writeROM,     NULL
    );
}



                        
///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEast0::updateDisplay(
    Bitmap* pScreen,
    Bitmap* /* pTempScreen is unused */
)
{
    //  Convenient pointers to the dirty buffers.
    Byte* pbCharDirty  = m_pCharDirty->getBuffer( );
    Byte* pbTile1Dirty = m_pTile1Dirty->getBuffer( );
    Byte* pbTile2Dirty = m_pTile2Dirty->getBuffer( );

    //  A character number and its colour.
    DWord dwCharNum;
    DWord dwColour;

    
    //  Update the character layer.  There are 64x64=4096 characters and each
    //  character is described in two bytes CCC---- NNNNNNNN where:
    //      CCCC     = character colour.
    //      NNNNNNNN = character number.
    for( DWord dwI = 0 ; dwI < 0x1000 ; dwI += 1 )
    {
        dwColour = m_pbCharRAM[ dwI << 1 ] >> 4;

        //  Only draw the character if it's dirty or its colour is dirty.
        if( m_pColTableChar->getDirty( dwColour ) || pbCharDirty[ dwI ] )
        {
            pbCharDirty[ dwI ] = FALSE;

            DWord dwCharNum = m_pbCharRAM[ ( dwI << 1 ) + 1 ];

            m_pCharBitmap->blit(
                ( *m_pGfxSetChar )[ dwCharNum ],
                dwI,
                m_pColTableChar->getEntry( dwColour )
            );
        }
    }
    m_pColTableChar->clearDirty( );

    //  Update the first tile layer.  There are 32x32=1024 tiles and each tile
    //  is described in two bytes CCCCNNNN NNNNNNNN where:
    //      CCCC          = tile colour.
    //      NNNN NNNNNNNN = tile number.
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        dwColour = m_pbTile1RAM[ dwI << 1 ] >> 4; 

        //  Only draw the tile if it's dirty or its colour is dirty.
        if( m_pColTableTile1->getDirty( dwColour ) || pbTile1Dirty[ dwI ] )
        {
            pbTile1Dirty[ dwI ] = FALSE;

            dwCharNum =
                ( ( m_pbTile1RAM[ dwI << 1 ] & 0x07 ) << 8 ) + 
                m_pbTile1RAM[ ( dwI << 1 ) + 1 ];

            m_pTile1Bitmap->blit(
                ( *m_pGfxSetTile1 )[ dwCharNum ],
                dwI,
                m_pColTableTile1->getEntry( dwColour )
            );
        }
    }
    m_pColTableTile1->clearDirty( );

    //  Update the second tile layer.  There are 32x32=1024 tiles and each tile
    //  is described in two bytes CCCCNNNN NNNNNNNN where:
    //      CCCC          = tile colour.
    //      NNNN NNNNNNNN = tile number.
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        dwColour = m_pbTile2RAM[ dwI << 1 ] >> 4; 

        //  Only draw the tile if it's dirty or its colour is dirty.
        if( m_pColTableTile2->getDirty( dwColour ) || pbTile2Dirty[ dwI ] )
        {
            pbTile2Dirty[ dwI ] = FALSE;

            DWord dwCharNum =
                ( ( m_pbTile2RAM[ dwI << 1 ] & 0x03 ) << 8 ) + 
                m_pbTile2RAM[ ( dwI << 1 ) + 1 ];

            m_pTile2Bitmap->blit(
                ( *m_pGfxSetTile2 )[ dwCharNum ],
                dwI,
                m_pColTableTile2->getEntry( dwColour )
            );
        }
    }
    m_pColTableTile2->clearDirty( );

    //  Blit the layers onto the screen.
    if( *m_pbPriorityRAM & 0x02 )
    {
        pScreen->blitScrollFull(
            m_pTile2Bitmap,
            -( ( m_pbTile2Scroll[ 0 ] << 8 ) + ( m_pbTile2Scroll[ 1 ] ) ),
            -( ( m_pbTile2Scroll[ 2 ] << 8 ) + ( m_pbTile2Scroll[ 3 ] ) ),
            pScreen->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
        updateSprites( pScreen );
        pScreen->blitScrollFull(
            m_pTile1Bitmap,
            -( ( m_pbTile1Scroll[ 0 ] << 8 ) + ( m_pbTile1Scroll[ 1 ] ) ),
            -( ( m_pbTile1Scroll[ 2 ] << 8 ) + ( m_pbTile1Scroll[ 3 ] ) ),
            pScreen->getFullClipping( ),
            Bitmap::TRANSPARENCY_RAW
        );
    }
    else
    {
        pScreen->blitScrollFull(
            m_pTile2Bitmap,
            -( ( m_pbTile2Scroll[ 0 ] << 8 ) + ( m_pbTile2Scroll[ 1 ] ) ),
            -( ( m_pbTile2Scroll[ 2 ] << 8 ) + ( m_pbTile2Scroll[ 3 ] ) ),
            pScreen->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
        pScreen->blitScrollFull(
            m_pTile1Bitmap,
            -( ( m_pbTile1Scroll[ 0 ] << 8 ) + ( m_pbTile1Scroll[ 1 ] ) ),
            -( ( m_pbTile1Scroll[ 2 ] << 8 ) + ( m_pbTile1Scroll[ 3 ] ) ),
            pScreen->getFullClipping( ),
            Bitmap::TRANSPARENCY_RAW
        );
        updateSprites( pScreen );
    }
  
    //  Now the character layer.
    pScreen->blitScrollFull(
        m_pCharBitmap,
        -( ( m_pbCharScroll[ 0 ] << 8 ) + ( m_pbCharScroll[ 1 ] ) ),
        -( ( m_pbCharScroll[ 2 ] << 8 ) + ( m_pbCharScroll[ 3 ] ) ),
        pScreen->getFullClipping( ),
        Bitmap::TRANSPARENCY_RAW
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateSprites
//
//  Description:
//
//      This member is called to allow the game to update the sprite display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEast0::updateSprites(
    Bitmap* pScreen
)
{
    //  Draw the sprite layer.
    for( DWord dwI = 0 ; dwI < 256 * 8 ; dwI += 8 )
    {
        //  If the sprite is disabled then continue to the next one.
        if( !( m_pbSpriteRAM[ dwI ] & 0x80 ) )
        {
            continue;
        }

        //  Calculate some sprite attributes that are used repeatedly.
        DWord dwCharNum     =
            ( ( m_pbSpriteRAM[ dwI + 2 ] & 0x0f ) << 8 ) + 
            m_pbSpriteRAM[ dwI + 3 ];
        DWord dwColour      = m_pbSpriteRAM[ dwI + 4 ] >> 4;
        int32 nX            = 
            240 - m_pbSpriteRAM[ dwI + 5 ] + 
            ( ( m_pbSpriteRAM[ dwI + 4 ] & 0x01 ) << 8 );
        int32 nY            = 
            240 - m_pbSpriteRAM[ dwI + 1 ] +
            ( ( m_pbSpriteRAM[ dwI ] & 0x01 ) << 8 );
        Byte  bHFlip        = m_pbSpriteRAM[ dwI ] & 0x20;
        Byte  bVFlip        = m_pbSpriteRAM[ dwI ] & 0x40;
        Byte  bDoubleHeight = m_pbSpriteRAM[ dwI ] & 0x08;
        Byte  bQuadHeight   = m_pbSpriteRAM[ dwI ] & 0x10;

        if( bDoubleHeight )
        {
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum ],
                nX,
                nY - 16,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum + 1 ],
                nX,
                nY,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
        }
        else
        if( bQuadHeight )
        {
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum ],
                nX,
                nY - 48,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum + 1 ],
                nX,
                nY - 32,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum + 2 ],
                nX,
                nY - 16,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum + 3 ],
                nX,
                nY,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
        }
        else
        {
            pScreen->blit(
                ( *m_pGfxSetSprite )[ dwCharNum ],
                nX,
                nY,
                m_pColTableSprite->getEntry( dwColour ),
                bHFlip,
                bVFlip,
                pScreen->getFullClipping( ),
                Bitmap::TRANSPARENCY_RAW
            );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writePalette
//
//  Description:
//
//      This function is called when a value is written to the palette RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEast0::s_writePalette(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  Robocop uses RAM to indicate the current palette.  This 
    //  allows dynamic palette manipulation.

    //  The canvas.
    Canvas* pCanvas = Replay::s_instance( ).getCanvas( );

    //  The game object.
    GameDataEast0* pThis = ( GameDataEast0* )( pHandler->getData( ) );
    ASSERT( pThis != NULL );

    //  If the value hasn't changed then nothing to do.
    if( pThis->m_pSpaceGame->getByte( dwAddress ) == bValue )
    {
        return;
    }

    //  The colour table affected.
    ColourTable& colTable = 
        *( pThis->m_colourTableList[ ( ( dwAddress & 0x700 ) >> 8 ) / 2 ] );

    //  The colour affected.
    DWord dwColour = ( dwAddress & 0x1ff ) / 2;

    //  The offset of the colour components in memory.
    DWord dwOffset = dwAddress & 0x7fe;

    //  Write the value to memory.
    pThis->m_pSpaceGame->setByte( dwAddress, bValue );

    //  Get the RGB components.
    DWord dwR = pThis->m_pbRedRAM[ dwOffset ];
    DWord dwG = pThis->m_pbGreenRAM[ dwOffset ];
    DWord dwB = pThis->m_pbBlueRAM[ dwOffset ];

    //  Unlock the old value.
    pCanvas->unlockColour( colTable[ dwColour ] );

    //  The very first colour is used for transparency.  To prevent non-
    //  transparent black from matching transparent black, we start searching
    //  at the second colour unless this is a modification of the first
    //  value of a colour table entry (16 values per entry).
    DWord dwStart = dwColour % 16 ? 1 : 0;

    //  If the colour already exists then use it.
    DWord dwIdx;
    if( pCanvas->getColour( &dwIdx, dwR, dwG, dwB, dwStart ) )
    {
    }
    //  Otherwise, if there's a free colour then use that.
    else
    if( pCanvas->getFreeColour( &dwIdx ) )
    {
        pCanvas->setColour( dwIdx, dwR, dwG, dwB );
    }
    //  Otherwise, get the closest colour we can.
    else
    if( pCanvas->getClosestColour( &dwIdx, dwR, dwG, dwB, dwStart ) )
    {
    }
        
    //  Assign the new colour value.
    colTable.setValue( dwColour, dwIdx );

    //  Lock it down.
    pCanvas->lockColour( dwIdx );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSoundCmd
//
//  Description:
//
//      This function is called when the game CPU writes a command to
//      the sound CPU.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GameDataEast0::s_writeSoundCmd(
    DWord         dwAddress,
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameDataEast0* pThis = ( GameDataEast0* )( pHandler->getData( ) );

    //  Only the low byte contains the command.
    if( dwAddress & 0x01 )
    {
        //  Add the command.
        pThis->m_pSoundCommand->addCommand( bValue );

        //  Interrupt the game CPU so the sound CPU can handle the command.
        pThis->m_pCPUSound->interrupt( CPU::INT_NMI );
        pThis->m_pCPUGame->setCyclesLeft( 0 );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This is a memory handler for writing a value to a location
//      that causes the screen to rotate 180.  
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEast0::s_writeScreenRotate( 
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  The low byte contains the flipping info.
    if( dwAddress & 0x01 )
    {
        Game::s_writeScreenRotate( dwAddress, bValue >> 7, pHandler );
    }

    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}



