//////////////////////////////////////////////////////////////////////////////
//
//  File:    _decoigs.cpp
//
//  Class:   GameDataEastIGS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for a series of
//      games by Data East that were part of the Data East Interchangeable
//      Game System.  The hardware allowed a new game to be loaded simply
//      by changing a *tape* containing the program code and the marquee.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "_decoigs.h"
#include "space.h"
#include "input.h"
#include "ctrlmap.h"
#include "cqueue.h"
#include "gfxset.h"
#include "screen.h"
#include "clip.h"
#include "ctable.h"
#include "cqueue.h"
#include "ay8910.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameDataEastIGS
//
//  Description:
//
//      This is the main constructor for a Data East IGS game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      bScrollingBG (input)
//          Indicates whether or not the game uses a scrolling background.
//
//      nMirrorOffset (input)
//          Indicates the difference between the video RAM and the 
//          mirror video RAM.
//
//      bReverseCoin (input)
//          Indicates that the coin mechs use reversed bits.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameDataEastIGS::GameDataEastIGS(
    const KString& iName,
    const Byte     bScrollingBG,
    const int32    nMirrorOffset,
    const Byte     bReverseCoin
)
:
    Game              ( iName ),
    m_pInputP1        ( NULL ),
    m_pInputP2        ( NULL ),
    m_pInputCoin      ( NULL ),
    m_pInputDip1      ( NULL ),
    m_pInputDip2      ( NULL ),
    m_pSpaceGame      ( NULL ),
    m_pSpaceSound     ( NULL ),
    m_pBufferFG       ( NULL ),
    m_pBufferBG       ( NULL ),
    m_pSoundCommand   ( NULL ),
    m_pCPUGame        ( NULL ),
    m_pCPUSound       ( NULL ),
    m_pGfxSetFG       ( NULL ),
    m_pGfxSetBG       ( NULL ),
    m_pGfxSetSprite   ( NULL ),
    m_pColTableFG     ( NULL ),
    m_pColTableBG     ( NULL ),
    m_pAY8910One      ( NULL ),
    m_pAY8910Two      ( NULL ),
    m_pbVideoLoRAM    ( NULL ),
    m_pbVideoHiRAM    ( NULL ),
    m_pbScrollHi      ( NULL ),
    m_pbScrollLo      ( NULL ),
    m_pbSoundControl  ( NULL ),
    m_bCoin           ( FALSE ),
    m_pClipping       ( NULL ),
    m_bScrollingBG    ( bScrollingBG ),
    m_nMirrorOffset   ( nMirrorOffset ),
    m_bReverseCoin    ( bReverseCoin )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameDataEastIGS
//
//  Description:
//
//      This is the destructor for a Data East IGS game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameDataEastIGS::~GameDataEastIGS(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of a Data East IGS game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::startUpSpace(
)
{
    //  Create the address spaces for the CPUs.
    m_pSpaceGame  = m_spaceList.add( new AddrSpace( "Game" ) );
    m_pSpaceSound = m_spaceList.add( new AddrSpace( "Sound" ) );

    //  Create temporary buffers for decoding the graphics.
    m_pBufferFG = m_tempBufferList.add( new Buffer( "FG Chars", 0x6000 ) );
    m_pBufferBG = m_tempBufferList.add( new Buffer( "BG Chars", 0x2000 ) );

    //  Create a command queue for communication between the CPUs.
    m_pSoundCommand = new CommandQueue( "DECOSound", 1, TRUE );
    m_bufferList.add( m_pSoundCommand );
}

void
GameDataEastIGS::startUpCPU(
)
{
    //  DECO IGS games use two 6502s, one for the game and one for sound.
    m_pCPUGame = m_cpuList.add( 
        createCPU( "Game",  CPU::CPU_6502, m_pSpaceGame )
    );
    m_pCPUSound = m_cpuList.add( 
        createCPU( "Sound", CPU::CPU_6502, m_pSpaceSound, CPU::SOUND )
    );
    
    //  The game CPU runs at 750KHz and the sound CPU runs at 
    //  500KHz.  This may not be correct but looks pretty good.
    m_pCPUGame->setClockSpeed( 750000 );
    m_pCPUSound->setClockSpeed( 500000 );
     
    //  We interrupt the Game once per frame for vblank.
    //  The Sound CPU is interrupted 16 times per frame.  Not sure if this is 
    //  correct.
    m_pCPUGame->setIntsPerFrame( 1 );
    m_pCPUSound->setIntsPerFrame( 16 );

    //  Set the interrupt handlers for the CPUs.
    m_pCPUGame->interruptHandler( s_gameInterrupt );
    m_pCPUSound->interruptHandler( s_soundInterrupt );
}

void
GameDataEastIGS::startUpInput(
)
{
    //  Create the inputs needed.
    m_pInputP1   = m_inputList.add( new Input( "Player 1 Controls", 0xff ) );
    m_pInputP2   = m_inputList.add( new Input( "Player 2 Controls", 0xff ) );
    m_pInputCoin = m_inputList.add( new Input( "Coin Controls",     0xff ) );
    m_pInputDip1 = m_inputList.add( new Input( "Dip Switch 1",      0xff ) );
    m_pInputDip2 = m_inputList.add( new Input( "Dip Switch 2",      0xff ) );

    //  Setup the controls.
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Right",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_RIGHT, 
            m_pInputP1,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Left",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_LEFT, 
            m_pInputP1,
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Up",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_UP, 
            m_pInputP1,
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Down",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_DOWN, 
            m_pInputP1,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 1 Action",
            CtrlMap::UPRIGHT1 | CtrlMap::COCKTAIL1 | CtrlMap::UPRIGHT2,
            CtrlMap::P1_ACTION1, 
            m_pInputP1,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Right",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_RIGHT, 
            m_pInputP2,
            0x01
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Left",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_LEFT, 
            m_pInputP2,
            0x02
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Up",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_UP, 
            m_pInputP2,
            0x04
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Down",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_DOWN, 
            m_pInputP2,
            0x08
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Player 2 Action",
            CtrlMap::COCKTAIL2,
            CtrlMap::P2_ACTION1, 
            m_pInputP2,
            0x10
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 1 Insert", 
            CtrlMap::PLAYER1,
            CtrlMap::COIN1, 
            m_pInputCoin,
            0x40
        )
    );
    m_ctrlMapList.add( 
        new CtrlMap( 
            "Coin 2 Insert", 
            CtrlMap::PLAYER1,
            CtrlMap::COIN2, 
            m_pInputCoin,
            0x80
        )
    );
}

void
GameDataEastIGS::startUpGraphics(
)
{
    //  Data East IGS games must have a 256x256 screen.  The background
    //  may be 1 or two times as tall as the screen depending on whether or
    //  not the game supports scrolling backgrounds.
    //  The game runs at 57 FPS based on the following info from 
    //  the MAME driver:
    //  Main clock: XTAL = 12 MHz
    //  Horizontal video frequency: HSYNC = XTAL/768?? = 15.625 kHz ??
    //  Video frequency: VSYNC = HSYNC/272 = 57.44 Hz ?
    //  VBlank duration: 1/VSYNC * (24/272) = 1536 us ?
    m_pScreen->setScreenSize( 256, 256, 1, m_bScrollingBG ? 2 : 1 );
    m_pScreen->setFPS( 57 );
    m_pScreen->setVBlank( 1536, m_pInputDip1->getLocation( ), 0x80 );

    //  The main clipping area chops the first and last columns.
    m_pClipping = m_clippingList.add( new Clipping( 8, 247, 0, 255 ) );
}

void
GameDataEastIGS::startUpColour(
)
{
    //  The characters and sprites use the first 8 colours in the palette and
    //  the background characters use the next 8 colours in the palette.
    m_pColTableFG = 
        m_colourTableList.add( new ColourTable( "FG ColTable", 1, 8 ) );
    m_pColTableBG = 
        m_colourTableList.add( new ColourTable( "BG ColTable", 1, 8 ) );

    //  Assign the values for the colour tables.
    for( DWord dwI = 0 ; dwI < 8 ; dwI += 1 )
    {
        ( *m_pColTableFG )[ dwI ] = dwI;
    }
    for( DWord dwI = 0 ; dwI < 8 ; dwI += 1 )
    {
        ( *m_pColTableBG )[ dwI ] = dwI + 8;
    }
}

void
GameDataEastIGS::startUpSound(
)
{
    //  Data East IGS games use two AY-3-8910s hooked up to the sound CPU.
    m_pAY8910One = m_soundDeviceList.add( new AY8910( "1", 1500000, 0x20ff ) );
    m_pAY8910Two = m_soundDeviceList.add( new AY8910( "2", 1500000, 0x20ff ) );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: decryptROMs
//
//  Description:
//
//      This function is the default method used to decrypt the game ROMs.
//      All DECO IGS games seem to have encrypted opcodes.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::decryptROMs(
)
{
    //  Used to traverse the source buffer.
    DWord dwAddress;


    //  Set to opcode encryption.
    m_pSpaceGame->setEncryptType( AddrSpace::CRYPT_OP );
    

    //  Pointers to the source and destinations.
    Byte* pbSource = m_pSpaceGame->getBuffer( );
    Byte* pbDest   = m_pSpaceGame->getDecryptBuffer( );


    //  Loop through each location in the address space.
    for( 
        dwAddress = 0x0000 ; 
        dwAddress < m_pSpaceGame->getSize( ) ; 
        dwAddress += 1 
    )
    {
        //  The majority of DECO IGS games use a nice brain-dead encryption 
        //  scheme.  Just shuffle a couple of bits.
        *pbDest = 
            ( ( *pbSource & 0x40 ) >> 1 ) | 
            ( ( *pbSource & 0x20 ) << 1 ) | 
            ( *pbSource & 0x9f );
        pbSource += 1;
        pbDest   += 1;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateCharacters
//
//  Description:
//
//      This member is called to update the display of the foreground
//      characters.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::updateCharacters(
    Bitmap* pScreen
)
{
    //  The number of a character being drawn.
    DWord dwCharNum;


    //  Draw the foreground characters.  There are 32x32=1024 characters.
    for( DWord dwI = 0 ; dwI < 0x0400 ; dwI += 1 )
    {
        //  Calculate the character number.  It is made up of two bytes.
        dwCharNum = ( m_pbVideoHiRAM[ dwI ] << 8 ) | m_pbVideoLoRAM[ dwI ];

        //  Draw the character transparently onto the screen only if
        //  it isn't a blank character ( dwCharNum == 0 ).
        if( dwCharNum )
        {
            pScreen->blit(
                ( *m_pGfxSetFG )[ dwCharNum ],
                ( dwI & 0x1f ) * 8,
                ( dwI >> 5 ) * 8,
                *m_pColTableFG,
                FALSE,
                FALSE,
                m_pClipping,
                Bitmap::TRANSPARENCY_RAW
            );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateSprites
//
//  Description:
//
//      This member is called to update the display of the sprites.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::updateSprites(
    Bitmap* pScreen
)
{
    //  Draw the sprites.  There are 8 sprites each using 4 bytes.
    //  The 32 bytes can be found in the first column of video RAM.  Since
    //  this column is clipped, the sprite values will not cause junk 
    //  characters on the screen.
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 4 * 0x20 )
    {
        //  Draw the sprites if they are enabled.
        //  Note that when the screen is flipped, the sprites are offset 2
        //  pixels in the X direction which we have to compensate for.
        if( m_pbVideoLoRAM[ dwI + 0 * 0x20 ] & 0x01 )
        {
            pScreen->blit(
                ( *m_pGfxSetSprite )[ m_pbVideoLoRAM[ dwI + 0x20 ] ],
                ( m_bScreenFlippedX ? 241 : 239 ) - 
                    m_pbVideoLoRAM[ dwI + 2 * 0x20 ],
                m_pbVideoLoRAM[ dwI + 3 * 0x20 ],
                *m_pColTableFG,
                m_pbVideoLoRAM[ dwI + 0 * 0x20 ] & 0x02,
                m_pbVideoLoRAM[ dwI + 0 * 0x20 ] & 0x04,
                m_pClipping,
                Bitmap::TRANSPARENCY_RAW
            );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_gameInterrupt 
//
//  Description:
//
//      This function is the interrupt handler for the game CPU.  It updates
//      the vblank handler and then returns an appropriate interrupt value.
//
//  Parameters:
//
//      pCPU (input)
//          The CPU being interrupted.
//
//      pGame (input)
//          A pointer to the game.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
GameDataEastIGS::s_gameInterrupt( 
    CPU*  /* pCPU is unused */,
    Game* pGame
)
{
    //  The game.
    GameDataEastIGS* pThis = ( GameDataEastIGS* )pGame;

 
    //  We check if a coin is inserted.  The two coin mechs use the two high
    //  bits of the coin input.  The bits are either on or off depending
    //  on the game.
    if( 
        ( *( pThis->m_pInputCoin->getLocation( ) ) & 0xc0 ) != 
        ( pThis->m_bReverseCoin ? 0x00 : 0xc0 )
    )
    {
        //  Check the coin semaphore.
        if( pThis->m_bCoin == FALSE )
        {
            //  Normally return an NMI when a coin is inserted, however
            //  games that have reversed coin bits use IRQ's.
            pThis->m_bCoin = TRUE;
            return( pThis->m_bReverseCoin ? CPU::INT_IRQ : CPU::INT_NMI );
        }
    }
    else
    {
        //  Reset the coin semaphore.
        pThis->m_bCoin = FALSE;
    }

    return( CPU::INT_NONE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_soundInterrupt
//
//  Description:
//
//      This function is the interrupt handler for the sound CPU.
//
//  Parameters:
//
//      pCPU (input)
//          The sound CPU.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
GameDataEastIGS::s_soundInterrupt(
    CPU*  pCPU,
    Game* pGame
)
{
    //  A pointer to the sound command.
    GameDataEastIGS* pThis = ( GameDataEastIGS* )pGame;

    //  If the interrupts are enabled then return the regular NMI.
    if( 
        pCPU->intsEnabled( ) && 
        ( !pThis->m_pbSoundControl || ( *( pThis->m_pbSoundControl ) & 0x08 ) ) 
    )
    {
        return( CPU::INT_NMI );
    }

    //  No interrupt this time.
    return( CPU::INT_NONE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readMirror
//
//  Description:
//
//      This function is called when the mirror video RAM is read from.
//      The X/Y coordinates are swapped in the mirror RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DWord
GameDataEastIGS::s_readMirror(
    DWord        dwAddress,
    ReadHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameDataEastIGS* pThis = ( GameDataEastIGS* )( pHandler->getData( ) );

    //  We swap the X & Y (32x32) locations and then read from the required 
    //  location.
    DWord dwOffset = dwAddress & 0x03ff;
    DWord dwX      = dwOffset / 32;
    DWord dwY      = dwOffset % 32;
    
    //  Mask out the X & Y bits from the address.
    dwAddress &= 0xfc00;

    //  Now return the value of from the mirror RAM.
    return( 
        pThis->m_pCPUGame->readMem( 
            dwAddress + pThis->m_nMirrorOffset + 32 * dwY + dwX 
        ) 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writePalette
//
//  Description:
//
//      This function is called when a value is written to the palette RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::s_writePalette(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* /* pHandler is not used */
)
{
    //  Data East IGS games use RAM to indicate the current palette.  This 
    //  allows dynamic palette manipulation.
    //
    //  The RAM is hooked up in the following manner (from the Burgertime
    //  schematics):
    //
    //  Bit       Resistor               Colour     State
    //   0  /\/\/ 47 kohm  \/\/\/\/\/\/\  Red      inverted
    //   1  /\/\/ 33 kohm  \/\/\/\/\/\/\  Red      inverted
    //   2  /\/\/ 15 kohm  \/\/\/\/\/\/\  Red      inverted
    //   3  /\/\/ 47 kohm  \/\/\/\/\/\/\  Green    inverted
    //   4  /\/\/ 33 kohm  \/\/\/\/\/\/\  Green    inverted
    //   5  /\/\/ 15 kohm  \/\/\/\/\/\/\  Green    inverted
    //   6  /\/\/ 33 kohm  \/\/\/\/\/\/\  Blue     inverted
    //   7  /\/\/ 15 kohm  \/\/\/\/\/\/\  Blue     inverted
    //

    //  Used to calculate the palette entries.
    Byte bBits[ 3 ];

    //  The RGB components.
    DWord dwRed; 
    DWord dwGreen; 
    DWord dwBlue; 

    //  The bits are inverted.
    bValue = ~bValue;

    //  Calculate the colour components.
    bBits[ 0 ] = ( bValue >> 0 ) & 0x01;
    bBits[ 1 ] = ( bValue >> 1 ) & 0x01;
    bBits[ 2 ] = ( bValue >> 2 ) & 0x01;
    dwRed      = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

    bBits[ 0 ] = ( bValue >> 3 ) & 0x01;
    bBits[ 1 ] = ( bValue >> 4 ) & 0x01;
    bBits[ 2 ] = ( bValue >> 5 ) & 0x01;
    dwGreen    = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

    bBits[ 0 ] = 0;
    bBits[ 1 ] = ( bValue >> 6 ) & 0x01;
    bBits[ 2 ] = ( bValue >> 7 ) & 0x01;
    dwBlue     = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];

    //  Set the colours.
    Replay::s_instance( ).getCanvas( )->setColour( 
        dwAddress & 0x0f, dwRed, dwGreen, dwBlue 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeMirror
//
//  Description:
//
//      This function is called when the mirror video RAM is written to.
//      The X/Y coordinates are swapped in the mirror RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameDataEastIGS::s_writeMirror(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameDataEastIGS* pThis = ( GameDataEastIGS* )( pHandler->getData( ) );

    //  We swap the X & Y (32x32) locations and then read from the required 
    //  location.
    DWord dwOffset = dwAddress & 0x03ff;
    DWord dwX      = dwOffset / 32;
    DWord dwY      = dwOffset % 32;
    
    //  Mask out the X & Y bits from the address.
    dwAddress &= 0xfc00;

    //  Set the value in mirror RAM.
    pThis->m_pCPUGame->writeMem( 
        dwAddress + pThis->m_nMirrorOffset + 32 * dwY + dwX, bValue 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSoundCmd
//
//  Description:
//
//      This function is called when the game CPU writes a command to
//      the sound CPU.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GameDataEastIGS::s_writeSoundCmd(
    DWord         /* dwAddress is unused */, 
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameDataEastIGS* pThis = ( GameDataEastIGS* )( pHandler->getData( ) );

    //  Add the command.
    pThis->m_pSoundCommand->addCommand( bValue );

    //  Interrupt the game CPU so the sound CPU can handle the command.
    pThis->m_pCPUSound->interrupt( CPU::INT_IRQ );
    pThis->m_pCPUGame->setCyclesLeft( 0 );
}
