///////////////////////////////////////////////////////////////////////////////
//
//  File:    _turtles.cpp
//
//  Class:   GameTurtlesBase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract class that serves as a base for all Turtle games.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////
//  System Headers.

//  Application Headers.
#include "_turtles.h"
#include "hiscore.h"
#include "space.h"
#include "input.h"
#include "dip.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameTurtlesBase
//
//  Description:
//
//      This is the main constructor for a Turtles game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameTurtlesBase::GameTurtlesBase(
    const KString& iName
)
:
    GameAmidarHW      ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameTurtlesBase
//
//  Description:
//
//      This is the destructor for a Turtles game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameTurtlesBase::~GameTurtlesBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of a Turtles game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////

//  NOTE:  This function can be removed once Z80NBA is made to work with
//         the Turtles games.  For now, we override to indicate that we
//         want to use the 'C' CPU core for the game.
void
GameTurtlesBase::startUpCPU(
)
{
    //  We interleave the CPUs at the rate of 10 times per frame to allow
    //  proper communication.
    m_nCPUInterleave = 10;

    //  Amidar hardware uses two Z80's, one for the game and one for the Sound.
    m_pCPUGame = m_cpuList.add( 
        createCPU( "Game",  CPU::CPU_Z80, m_pSpaceGame, CPU::GAME, TRUE )
    );
    m_pCPUSound = m_cpuList.add( 
        createCPU( "Sound", CPU::CPU_Z80, m_pSpaceSound, CPU::SOUND )
    );
    
    //  The game CPU runs at 3.072Mhz.  The sound CPU runs at 1.78975Mhz.
    m_pCPUGame->setClockSpeed( 3072000 );
    m_pCPUSound->setClockSpeed( 1789750 );
     
    //  The first CPU uses NMI interrupts, the second has interrupts triggered
    //  from the first CPU.
    m_pCPUGame->interruptHandler( s_nmiInterrupt );
    m_pCPUSound->interruptHandler( s_noInterrupt );
}

void
GameTurtlesBase::startUpInput(
)
{
    //  Allow the base class to create the inputs and set up the controls.
    GameAmidarHW::startUpInput( );

    //  We don't want to default to infinite lives.
    m_pInputIN1->setInitial( 0xfd );

    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "IN1", m_pInputIN1->getLocation( ), TRUE )
    );
    pDip = m_dipSwitchList.add(
        new DipSwitch( "IN2", m_pInputIN2->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Cabinet",
        0x08,
        "Cocktail Cabinet",
        "Upright Cabinet"
    );
    pDip->addSetting(
        "Unknown1",
        0x20,
        "??? Off",
        "??? On"
    );
    pDip->addSetting(
        "Unknown2",
        0x80,
        "??? Off",
        "??? On"
    );

    //  The Amidar high scores are stored at:
    //      0x8200: 10 scores * 3 bytes/score = 30 bytes.
    //      0x80a8: 1 top score * 3 bytes/score = 3 bytes.
    //  Amidar writes to this area twice before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 2 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x8200, 30 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x80a8,  3 );
}

void
GameTurtlesBase::startUpMemMap(
)
{
    //  Register the high score handler first.
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x80aa, 0x80aa, s_writeHiScore,      m_pHiScore
    ); 

    //  Allow the base class to register the remainder of the handlers.
    GameAmidarHW::startUpMemMap( );
}

void
GameTurtlesBase::startUpColour(
)
{
    //  The amidar colour PROM.
    static Byte abColourPROM[ ] =
    {
        0x00, 0xc0, 0x57, 0xff, 0x00, 0x66, 0xf2, 0xfe, 
        0x00, 0x2d, 0x12, 0xbf, 0x00, 0x2f, 0x7d, 0xb8,
        0x00, 0x72, 0xd2, 0x06, 0x00, 0x94, 0xff, 0xe8,
        0x00, 0x54, 0x2f, 0xf6, 0x00, 0x24, 0xbf, 0xc6
    };

    //  Assign the colour PROM.
    m_pColourPROM = new Buffer( 
        "Colour PROM", 
        sizeof( abColourPROM ) / sizeof( abColourPROM[ 0 ] ),
        abColourPROM
    );

    //  The base class will initialize the colours.
    GameAmidarHW::startUpColour( );
}
