//////////////////////////////////////////////////////////////////////////////
//
//  File:    lnc.cpp
//
//  Class:   GameLockNChase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the class for the Lock'N'Chase driver.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "lnc.h"
#include "gameinfo.h"
#include "space.h"
#include "buffer.h"
#include "input.h"
#include "hiscore.h"
#include "dip.h"
#include "gfxset.h"
#include "ctable.h"
#include "screen.h"


///////////////////////////////////////////////////////////////////////////////
//  Static declaration.
///////////////////////////////////////////////////////////////////////////////

//  Game Information.
GAME_INFO

    ASSIGN_GAME_ID,
        "lnc",

    ASSIGN_GAME_NAME,
        "Lock'n'Chase",

    ASSIGN_GAME_VERSION,      
        REPLAY_MAJOR_REVISION, REPLAY_MINOR_REVISION, "1", "0",

    ASSIGN_GAME_CONTRIBUTORS,
        "Kevin Brisley",

    ASSIGN_GAME_REQD_FILES,
        "s0-3a",
        "s1-3b",
        "s2-3c",
        "s3-3d",
        "s4-11l",
        "s5-11m",
        "s6-13l",
        "s7-13m",
        "s8-15l",
        "s9-15m",
        "sa-1h",

FOR( lnc, GameLockNChase )


///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameLockNChase
//
//  Description:
//
//      This is the main constructor for a Lock'n'Chase game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameLockNChase::GameLockNChase(
    const KString& iName
)
:
    GameDataEastIGS   ( iName, FALSE, -0x4000, FALSE ),
    m_pbCharSetRAM    ( NULL ),
    m_pDirty          ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameLockNChase
//
//  Description:
//
//      This is the destructor for the Lock'n'Chase game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameLockNChase::~GameLockNChase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Lock'n'Chase game class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GameLockNChase::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GameLockNChase" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of Lock'n'Chase.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameLockNChase::startUpSpace(
)
{
    //  Let the base class create our Game & Sound maps.
    GameDataEastIGS::startUpSpace( );

    //  Assign convenience pointers for various items.
    m_pbVideoLoRAM   = m_pSpaceGame->getBuffer( ) + 0x3c00;
    m_pbVideoHiRAM   = m_pSpaceGame->getBuffer( ) + 0x7800;
    m_pbSoundControl = m_pSpaceGame->getBuffer( ) + 0x8001;
    m_pbCharSetRAM   = m_pSpaceGame->getBuffer( ) + 0x8003;
}

void
GameLockNChase::startUpROMs(
)
{
    //  Load in the Game ROMs.
    m_pSpaceGame->loadFile( 0x0c000, m_pGameInfo->getGameId( ), "s3-3d" );
    m_pSpaceGame->loadFile( 0x0d000, m_pGameInfo->getGameId( ), "s2-3c" );
    m_pSpaceGame->loadFile( 0x0e000, m_pGameInfo->getGameId( ), "s1-3b" );
    m_pSpaceGame->loadFile( 0x0f000, m_pGameInfo->getGameId( ), "s0-3a" );

    //  Load in the Sound ROMs.
    m_pSpaceSound->loadFile( 0x0f000, m_pGameInfo->getGameId( ), "sa-1h" );
    
    //  Load in the foreground graphic ROMs.
    m_pBufferFG->loadFile( 0x0000, m_pGameInfo->getGameId( ), "s8-15l" );
    m_pBufferFG->loadFile( 0x1000, m_pGameInfo->getGameId( ), "s9-15m" );
    m_pBufferFG->loadFile( 0x2000, m_pGameInfo->getGameId( ), "s6-13l" );
    m_pBufferFG->loadFile( 0x3000, m_pGameInfo->getGameId( ), "s7-13m" );
    m_pBufferFG->loadFile( 0x4000, m_pGameInfo->getGameId( ), "s4-11l" );
    m_pBufferFG->loadFile( 0x5000, m_pGameInfo->getGameId( ), "s5-11m" );

    //  Decrypt the game ROMs.
    decryptROMs( );
}

void
GameLockNChase::startUpInput(
)
{
    //  The base class will create the inputs and setup the controls.
    GameDataEastIGS::startUpInput( );

    //  The start buttons are mapped to different bits depending on the game.
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 1 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P1_START,
            m_pInputCoin,
            0x01
        )
    );
    m_ctrlMapList.add(
        new CtrlMap(
            "Player 2 Start",
            CtrlMap::PLAYER1,
            CtrlMap::P2_START,
            m_pInputCoin,
            0x02
        )
    );


    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW1", m_pInputDip1->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Coin Slot 1", 
        0x0c,
        "1 Coin/1 Credit (Slot 1)",
        "1 Coin/2 Credits (Slot 1)",
        "1 Coin/3 Credits (Slot 1)",
        "2 Coins/1 Credit (Slot 1)"
    );
    pDip->addSetting(
        "Coin Slot 2", 
        0x03,
        "1 Coin/1 Credit (Slot 2)",
        "1 Coin/2 Credits (Slot 2)",
        "1 Coin/3 Credits (Slot 2)",
        "2 Coins/1 Credit (Slot 2)"
    );
    pDip->addSetting(
        "Diagnostic",
        0x30,
        "Game Mode",
        "Watchdog Test",
        "Full Test",
        "RAM Test"
    );
    pDip->addSetting(
        "Cabinet",
        0x40,
        "Cocktail Cabinet",
        "Upright Cabinet"
    );
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW2", m_pInputDip2->getLocation( ), TRUE )
    );
    pDip->addSetting(
        "Lives",
        0x01,
        "3 Lives",
        "5 Lives"
    );
    pDip->addSetting(
        "Bonus Life",
        0x06,
        "Bonus Life At $15,000",
        "Bonus Life At $20,000",
        "Bonus Life At $30,000",
        "No Bonus Life"
    );
    pDip->addSetting(
        "Speed",
        0x08,
        "Slow Speed",
        "Fast Speed"
    );


    //  The Lock'n'Chase high scores are stored at:
    //      0x0008: 1 top score * 3 bytes/score = 3 bytes.
    //      0x0294: 5 scores * 3 bytes/score    = 15 bytes.
    //      0x02a6: 5 names * 3 bytes/name      = 15 bytes.
    //  Lock'n'Chase writes to this area once before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 1 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0008, 3 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x0294, 15 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x02a6, 15 );
}

void
GameLockNChase::startUpGraphics(
)
{
    //  The base class sets the screen size and creates the default clipping
    //  region.
    GameDataEastIGS::startUpGraphics( );

    //  Map the static character positions on the temporary bitmap.
    m_pScreen->getTempBitmap( )->setMapSize( 0x400 );
    for( DWord dwI = 0 ; dwI < 0x400 ; dwI += 1 )
    {
        m_pScreen->getTempBitmap( )->setMapping( 
            dwI, ( dwI % 32 ) * 8, ( dwI / 32 ) * 8, m_pClipping
        ); 
    }

    //  Allocate a buffer that will be used to indicate which background
    //  characters need to be redrawn.
    m_pDirty = m_dirtyList.add( new Buffer( "Dirty", 0x0400, TRUE ) );

    //  Create the gfxset to hold the foreground characters and set the
    //  information.
    m_pGfxSetFG = m_gfxSetList.add( new GfxSet( "Foreground Chars" ) );
    m_pGfxSetFG->setNumber( 1024 );
    m_pGfxSetFG->setDimensions( 8, 8 );
    m_pGfxSetFG->setBPP( 3 );
    m_pGfxSetFG->setBitPlanes( 0, 65536, 131072 );
    m_pGfxSetFG->setXBits(  0,  8, 16, 24, 32, 40, 48, 56 );
    m_pGfxSetFG->setYBits(  7,  6,  5,  4,  3,  2,  1,  0 );
    m_pGfxSetFG->setIncrement( 64 );

    //  Decode the foreground characters.
    m_pGfxSetFG->decode( m_pBufferFG->getBuffer( ) );
    

    //  Create the gfxset to hold the sprites and set the information.
    m_pGfxSetSprite = m_gfxSetList.add( new GfxSet( "Sprites" ) );
    m_pGfxSetSprite->setNumber( 256 );
    m_pGfxSetSprite->setDimensions( 16, 16 );
    m_pGfxSetSprite->setBPP( 3 );
    m_pGfxSetSprite->setBitPlanes( 0, 65536, 131072 );
    m_pGfxSetSprite->setXBits(  
          0,   8,  16,  24,  32,  40,  48,  56, 
         64,  72,  80,  88,  96, 104, 112, 120
    );
    m_pGfxSetSprite->setYBits(  
          7,   6,   5,   4,   3,   2,   1,   0,
        135, 134, 133, 132, 131, 130, 129, 128
    );
    m_pGfxSetSprite->setIncrement( 256 );

    //  Decode the background characters.
    m_pGfxSetSprite->decode( m_pBufferFG->getBuffer( ) );
}

void
GameLockNChase::startUpColour(
)
{
    //  Unlike the other Data East IGS games, Lock'n'Chase uses a colour PROM
    //  instead of RAM to generate the palette.  There are 8 colours used for
    //  the whole game.
    static Byte abColourPROM[ ] =
    {
        0x00, 0xdf, 0x51, 0x1c, 0xa7, 0xe0, 0xfc, 0xff
    };
        
    //  The base class will allocate the colour tables for us.
    GameDataEastIGS::startUpColour( );

    //  Lock'n'Chase doesn't modify the palette dynamically during run-time
    //  so we simply set the colours here on startup and never worry about
    //  them again.
    // 
    //  I'm not sure if the following is correct.  The colours look a bit
    //  off.  Can anyone with a real machine let me know whether the colours
    //  are at all accurate and if not, what they should be?
    //
    //  Bit       Resistor               Colour     State
    //   0  /\/\/ 33 kohm  \/\/\/\/\/\/\  Blue     normal
    //   1  /\/\/ 15 kohm  \/\/\/\/\/\/\  Blue     normal
    //   2  /\/\/ 47 kohm  \/\/\/\/\/\/\  Green    normal
    //   3  /\/\/ 33 kohm  \/\/\/\/\/\/\  Green    normal
    //   4  /\/\/ 15 kohm  \/\/\/\/\/\/\  Green    normal
    //   5  /\/\/ 47 kohm  \/\/\/\/\/\/\  Red      normal
    //   6  /\/\/ 33 kohm  \/\/\/\/\/\/\  Red      normal
    //   7  /\/\/ 15 kohm  \/\/\/\/\/\/\  Red      normal
    //

    //  Used to calculate the palette entries.
    Byte bBits[ 3 ];

    //  The RGB components.
    DWord dwRed; 
    DWord dwGreen; 
    DWord dwBlue; 

    //  Calculate the colour components.
    for( Byte bI = 0 ; bI < 8 ; bI += 1 )
    {
        Byte bValue = abColourPROM[ bI ];

        bBits[ 0 ] = ( bValue >> 5 ) & 0x01;
        bBits[ 1 ] = ( bValue >> 6 ) & 0x01;
        bBits[ 2 ] = ( bValue >> 7 ) & 0x01;
        dwRed      = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];
    
        bBits[ 0 ] = ( bValue >> 2 ) & 0x01;
        bBits[ 1 ] = ( bValue >> 3 ) & 0x01;
        bBits[ 2 ] = ( bValue >> 4 ) & 0x01;
        dwGreen    = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];
    
        bBits[ 0 ] = 0;
        bBits[ 1 ] = ( bValue >> 0 ) & 0x01;
        bBits[ 2 ] = ( bValue >> 1 ) & 0x01;
        dwBlue     = 0x21 * bBits[ 0 ] + 0x47 * bBits[ 1 ] + 0x97 * bBits[ 2 ];
    
        //  Set the colours.
        Replay::s_instance( ).getCanvas( )->setColour( 
            bI, dwRed, dwGreen, dwBlue 
        );
    }
}

void
GameLockNChase::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x0000, 0x3fff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Mirror Vid RAM", 0x7c00, 0x7fff, s_readMirror,       this
    );
    m_pCPUGame->readMemHandler( 
        "ROM",            0xc000, 0xffff, s_readROM,          NULL 
    );
    m_pCPUGame->readMemHandler( 
        "Dip Switch 1",   0x8000, 0x8000, s_readInput,        m_pInputDip1
    );
    m_pCPUGame->readMemHandler( 
        "Dip Switch 2",   0x8001, 0x8001, s_readInput,        m_pInputDip2
    );
    m_pCPUGame->readMemHandler( 
        "P1 Controls",    0x9000, 0x9000, s_readInput,        m_pInputP1
    );
    m_pCPUGame->readMemHandler( 
        "P2 Controls",    0x9001, 0x9001, s_readInput,        m_pInputP2
    );
    m_pCPUGame->readMemHandler( 
        "Coin Controls",  0x9002, 0x9002, s_readInput,        m_pInputCoin
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Player Switch",  0x0006, 0x0006, s_writeScreenRotate, this
    );
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x0294, 0x0294, s_writeHiScore,      m_pHiScore
    );
    m_pCPUGame->writeMemHandler(
        "RAM",            0x0000, 0x3bff, s_writeRAM,          NULL
    );
    m_pCPUGame->writeMemHandler(
        "Video RAM",      0x3c00, 0x3fff, s_writeVideoRAM,     this
    );
    m_pCPUGame->writeMemHandler(
        "Mirror Vid RAM", 0x7c00, 0x7fff, s_writeMirror,       this
    );
    m_pCPUGame->writeMemHandler(
        "Sound Command",  0x9002, 0x9002, s_writeSoundCmd,     this
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0xc000, 0xffff, s_writeROM,          NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "Scratch RAM",    0x0000, 0x03ff, s_readRAM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "ROM",            0xf000, 0xffff, s_readROM,          NULL
    );
    m_pCPUSound->readMemHandler( 
        "Sound Command",  0xa000, 0xa000, s_readCommand,      m_pSoundCommand
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "RAM",            0x0000, 0x03ff, s_writeRAM,         NULL
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Write",   0x2000, 0x2000, s_write8910Write,   m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#1 Control", 0x4000, 0x4000, s_write8910Control, m_pAY8910One
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Write",   0x6000, 0x6000, s_write8910Write,   m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "8910#2 Control", 0x8000, 0x8000, s_write8910Control, m_pAY8910Two
    );
    m_pCPUSound->writeMemHandler(
        "IEnable",        0xc000, 0xc000, s_writeIntEnable,   m_pCPUSound
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0xf000, 0xffff, s_writeROM,         NULL
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called to allow the game to update the display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameLockNChase::updateDisplay(
    Bitmap* pScreen,
    Bitmap* pTempScreen
)
{
    //  A pointer to the dirty buffer.
    Byte* pbDirty = m_pDirty->getBuffer( );

    //  The number of a character being drawn.
    DWord dwCharNum;


    //  Since Lock'n'Chase doesn't have a background layer, we use the
    //  foreground as the background.  This allows us to use a dirty,
    //  double-buffer scheme.
    for( DWord dwI = 0 ; dwI < 0x0400 ; dwI += 1 )
    {
        //  Draw the characters onto the temporary bitmap if it is dirty.
        if( pbDirty[ dwI ] )
        {
            //  Reset the dirty flag.
            pbDirty[ dwI ] = FALSE;

           //  Calculate the character number.  It is made up of two bytes.
           dwCharNum = ( m_pbVideoHiRAM[ dwI ] << 8 ) | m_pbVideoLoRAM[ dwI ];

            //  Draw the character.
            pTempScreen->blit(
                ( *m_pGfxSetFG )[ dwCharNum ], dwI, *m_pColTableFG
            );
        }
    }

    //  Blast the temporary screen to the real screen.
    pScreen->copy( pTempScreen );

    //  Now update the sprites.
    updateSprites( pScreen );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeVideoRAM
//
//  Description:
//
//      This function is called when the video RAM is written to.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GameLockNChase::s_writeVideoRAM(
    DWord         dwAddress, 
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameLockNChase* pThis = ( GameLockNChase* )( pHandler->getData( ) );

    //  The offset of the item.
    DWord dwOffset = dwAddress & 0x3ff;

    //  The value in the hi video RAM location should be the character set that
    //  is to be used to draw the character.
    pThis->m_pbVideoHiRAM[ dwOffset ] = *( pThis->m_pbCharSetRAM );

    //  Set the character as dirty.
    ( *( pThis->m_pDirty ) )[ dwOffset ] = TRUE;

    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This is called when a value is written to the location
//      that causes the screen to rotate.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameLockNChase::s_writeScreenRotate(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The game object is held in the handler data.
    GameLockNChase* pThis = ( GameLockNChase* )( pHandler->getData( ) );

    //  For some reason I can't find the memory location that rotates
    //  the screen.  Searching for places that read mask 0x40 from 0x8000
    //  (i.e. the cocktail/upright dip) yields no hints.  
    //  Therefore, as a hack, this will be called when a value is written
    //  to the location that indicates which player is active.  And if
    //  we've got a cocktail table then we'll rotate the screen.
    if( pThis->m_pInputDip1->getValue( ) & 0x40 )
    {
        Game::s_writeScreenRotate( dwAddress, bValue, pHandler );
    }

    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}
