///////////////////////////////////////////////////////////////////////////////
//
//  File:    pacnpal.cpp
//
//  Class:   GamePacNPal
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the game object for Bally Midway's Pac'n'Pal.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "pacnpal.h"
#include "gameinfo.h"
#include "space.h"
#include "input.h"
#include "dip.h"
#include "hiscore.h"


///////////////////////////////////////////////////////////////////////////////
//  Static declaration.
///////////////////////////////////////////////////////////////////////////////

//  Game Information.
GAME_INFO

    ASSIGN_GAME_ID,
        "pacnpal",

    ASSIGN_GAME_NAME,
        "Pac'n'Pal",

    ASSIGN_GAME_VERSION,      
        REPLAY_MAJOR_REVISION, REPLAY_MINOR_REVISION, "1", "0",

    ASSIGN_GAME_CONTRIBUTORS,
        "Kevin Brisley",
        "Lawnmower Man (sprites)",

    ASSIGN_GAME_REQD_FILES,
        "pap11b.cpu",
        "pap12b.cpu",
        "pap13b.cpu",
        "pap14.cpu",
        "pap15.vid",
        "pap16.cpu",

FOR( pacnpal, GamePacNPal )



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GamePacNPal
//
//  Description:
//
//      This is the main constructor for Pac'n'Pal.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GamePacNPal::GamePacNPal(
    const KString& iName
)
:
    Game6809Pac      ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GamePacNPal
//
//  Description:
//
//      This is the destructor for Pac'n'Pal.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GamePacNPal::~GamePacNPal(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Pac'n'Pal game class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GamePacNPal::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GamePacNPal" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of Pac'n'Pal.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GamePacNPal::startUpCPU(
)
{
    //  Let the base class initialize the CPUs.
    Game6809Pac::startUpCPU( );

    m_nCPUInterleave = 100;

    //  The only difference in Pac'n'Pal is that the sound CPU performs an
    //  IRQ each vblank.
    m_pCPUSound->interruptHandler( s_irqInterrupt );
}

void
GamePacNPal::startUpInput(
)
{
    //  The base class will create the inputs and setup the controls.
    Game6809Pac::startUpInput( );

    //  We need to adjust the initial values of one of the dip switches
    //  to get the appropriate default settings.
    m_pInputDSW1->setInitial( 0x98 );

    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW0", m_pInputDSW0->getLocation( ), FALSE )
    );
    pDip->addSetting(
        "Difficulty",
        0x03,
        "Left 1 Coin/1 Credit",
        "Left 1 Coin/2 Credits",
        "Left 2 Coins/1 Credit",
        "Left 2 Coins/3 Credits"
    );
    pDip->addSetting(
        "Difficulty",
        0x0c,
        "Rank A",
        "Rank B",
        "Rank C",
        "Rank D"
    );
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW1", m_pInputDSW1->getLocation( ), FALSE )
    );
    pDip->addSetting(
        "Left Coin",
        0x07,
        "Right 1 Coin/1 Credit",
        "Right 1 Coin/2 Credits",
        "Right 1 Coin/3 Credits",
        "Right 1 Coin/6 Credits",
        "Right 1 Coin/7 Credits",
        "Right 2 Coins/1 Credit",
        "Right 2 Coins/3 Credits",
        "Right 3 Coins/1 Credit"
    );
    pDip->addSetting(
        "Bonus Life",
        0x38,
        "No Bonus Lives",
        "Bonus Lives at 20k/70k/70k",
        "Bonus Lives at 30k/80k/80k",
        "Bonus Lives at 20k/70k",
        "Bonus Lives at 30k/70k",
        "Bonus Lives at 30k/80k",
        "Bonus Lives at 30k/100k",
        "Bonus Lives at 30k"
    );
    pDip->addSetting(
        "Lives",
        0xc0,
        "1 Life",
        "2 Lives",
        "3 Lives",
        "5 Lives"
    );


    //  The Pac'n'Pal high scores are stored at:
    //      0x104c: High Score table = 40 bytes.
    //      0x116d: Current High Score = 3 bytes.
    //  Pac'n'Pal writes to this area 22 times before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 22 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x104c, 40 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x116d,  3 );
}

void
GamePacNPal::startUpColour(
)
{
    //  The Pac'n'Pal colour PROM.  It contains the palette in the first
    //  32 bytes and then contains the character colour lookup table in the 
    //  next 256 bytes followed by the sprite colour lookup table in the final
    //  256 bytes.
    static Byte abColourPROM[ ] =
    {
        0xf6,0xc0,0x3f,0x3c,0x8f,0x07,0xe7,0xf0,
        0x2f,0x28,0xc9,0xed,0x5d,0xa4,0xf6,0x00,
        0x00,0xf6,0x5d,0xc9,0xe0,0xf0,0xba,0x38,
        0x3c,0x3f,0x27,0xb7,0xef,0x8f,0x07,0x00,

        0x00,0x06,0x05,0x01,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x01,0x0e,0x00,
        0x00,0x07,0x0e,0x02,0x00,0x01,0x0e,0x0d,
        0x00,0x06,0x0e,0x02,0x00,0x00,0x00,0x00,
        0x00,0x03,0x00,0x04,0x05,0x03,0x00,0x04,
        0x00,0x0b,0x00,0x09,0x0c,0x0b,0x00,0x09,
        0x00,0x02,0x00,0x03,0x00,0x0b,0x00,0x0d,
        0x00,0x0d,0x00,0x0e,0x00,0x09,0x00,0x0c,
        0x00,0x00,0x0e,0x00,0x00,0x00,0x01,0x00,
        0x00,0x00,0x02,0x00,0x00,0x00,0x03,0x00,
        0x00,0x00,0x04,0x00,0x00,0x00,0x05,0x00,
        0x00,0x00,0x06,0x00,0x00,0x00,0x07,0x00,
        0x00,0x00,0x08,0x00,0x00,0x00,0x09,0x00,
        0x00,0x00,0x0d,0x00,0x00,0x00,0x0b,0x00,
        0x00,0x00,0x0c,0x00,0x00,0x00,0x0d,0x00,
        0x0f,0x0f,0x0f,0x0f,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,

        0x0f,0x0f,0x0f,0x0f,0x0f,0x02,0x0f,0x0f,
        0x0f,0x03,0x0e,0x04,0x0f,0x05,0x00,0x01,
        0x0f,0x06,0x00,0x01,0x0f,0x07,0x00,0x01,
        0x0f,0x08,0x00,0x01,0x0f,0x01,0x07,0x0e,
        0x0f,0x0d,0x00,0x0b,0x0f,0x04,0x00,0x0d,
        0x00,0x00,0x00,0x00,0x0f,0x0d,0x00,0x02,
        0x0f,0x00,0x00,0x00,0x0f,0x07,0x07,0x00,
        0x0f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x05,0x02,0x0d,0x0f,0x05,0x0b,0x0e,
        0x0f,0x02,0x08,0x0e,0x0f,0x0e,0x05,0x08,
        0x00,0x00,0x00,0x00,0x0f,0x0e,0x05,0x05,
        0x0f,0x05,0x00,0x01,0x00,0x00,0x00,0x00,
        0x0f,0x05,0x0e,0x0c,0x0f,0x05,0x0e,0x03,
        0x0f,0x08,0x03,0x09,0x0f,0x03,0x0e,0x09,
        0x0f,0x02,0x0e,0x09,0x0f,0x08,0x0c,0x03,
        0x0f,0x07,0x0d,0x0e,0x00,0x00,0x00,0x00,
        0x0f,0x0e,0x05,0x0d,0x0f,0x0b,0x05,0x0c,
        0x0f,0x0b,0x05,0x09,0x0f,0x0b,0x08,0x03,
        0x0f,0x0b,0x09,0x03,0x0f,0x0b,0x02,0x03,
        0x0f,0x0b,0x08,0x03,0x0f,0x0b,0x0d,0x0d,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x07,0x0b,0x0e,0x0f,0x0e,0x0d,0x0b,
        0x0f,0x0e,0x0b,0x07,0x0f,0x02,0x0f,0x0e,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x0f,0x0f,0x0f,0x0f,
        0x0f,0x0f,0x00,0x0f,0x0f,0x0f,0x00,0x00,
        0x0f,0x00,0x0f,0x0f,0x0f,0x00,0x00,0x0f,
        0x0f,0x0f,0x0f,0x00,0x0f,0x00,0x00,0x00
    };

    //  Assign the colour PROM.
    m_pColourPROM = new Buffer( 
        "Colour PROM", 
        sizeof( abColourPROM ) / sizeof( abColourPROM[ 0 ] ),
        abColourPROM
    );

    //  The base class will initialize the colours.
    Game6809Pac::startUpColour( );
}

void
GamePacNPal::startUpSound(
)
{
    //  The Super Pac-Man sound PROM. 
    static Byte abSoundPROM[ ] =
    {
        0x07,0x0d,0x0e,0x0f,0x0c,0x0d,0x0d,0x0e,
        0x0e,0x0e,0x0d,0x0d,0x0c,0x0f,0x0e,0x0d,
        0x07,0x05,0x04,0x07,0x06,0x05,0x05,0x04,
        0x04,0x04,0x05,0x05,0x06,0x07,0x04,0x05,
        0x0c,0x0e,0x0e,0x0e,0x0e,0x0d,0x0d,0x0e,
        0x0e,0x0d,0x0e,0x0d,0x0c,0x06,0x06,0x07,
        0x0c,0x0c,0x06,0x05,0x04,0x05,0x04,0x04,
        0x05,0x05,0x04,0x04,0x04,0x04,0x06,0x07,
        0x0c,0x0e,0x0e,0x0e,0x0e,0x0d,0x0f,0x0c,
        0x0e,0x0e,0x0c,0x0e,0x0c,0x0e,0x06,0x05,
        0x0c,0x0c,0x06,0x04,0x06,0x05,0x04,0x04,
        0x07,0x05,0x04,0x04,0x04,0x04,0x06,0x05,
        0x07,0x0e,0x0c,0x0d,0x0e,0x0d,0x0c,0x0e,
        0x07,0x04,0x06,0x05,0x04,0x05,0x06,0x04,
        0x07,0x0f,0x0d,0x0e,0x0d,0x0f,0x07,0x07,
        0x05,0x04,0x05,0x07,0x07,0x0e,0x07,0x04,
        0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
        0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
        0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
        0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
        0x0e,0x0c,0x07,0x0e,0x0c,0x07,0x05,0x0c,
        0x0e,0x0d,0x0c,0x0e,0x0f,0x06,0x06,0x07,
        0x0c,0x0c,0x07,0x04,0x06,0x05,0x04,0x07,
        0x0d,0x07,0x06,0x04,0x07,0x06,0x04,0x07,
        0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0c,0x0d,
        0x0c,0x0e,0x07,0x0c,0x05,0x07,0x05,0x05,
        0x0c,0x0f,0x0e,0x0e,0x0c,0x04,0x06,0x05,
        0x05,0x04,0x05,0x07,0x05,0x04,0x05,0x07,
        0x0c,0x0e,0x0e,0x0e,0x0e,0x0c,0x0f,0x0d,
        0x07,0x06,0x05,0x05,0x05,0x06,0x0c,0x0e,
        0x0f,0x0f,0x0d,0x06,0x04,0x06,0x06,0x06,
        0x07,0x05,0x04,0x03,0x02,0x01,0x03,0x07
    };

    //  Assign the sound PROM.
    m_pSoundPROM = new Buffer( 
        "Sound PROM", 
        sizeof( abSoundPROM ) / sizeof( abSoundPROM[ 0 ] ),
        abSoundPROM
    );

    //  Call the base class.
    Game6809Pac::startUpSound( );
}

void
GamePacNPal::startUpMemMap(
)
{
    //  Set some specific handlers before setting the defaults.
    m_pCPUGame->readMemHandler( 
        "Screen Rotate",  0x2000, 0x2000, s_readScreenRotate,  this
    );
    m_pCPUGame->writeMemHandler(
        "Screen Rotate",  0x2000, 0x2000, s_writeScreenRotate, this
    );
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x116e, 0x116e, s_writeHiScore,      m_pHiScore
    );

    //  Allow the base class to set the remainder of the handlers.
    Game6809Pac::startUpMemMap( );

    //  We have a special shared memory handler.  This is set after the
    //  base class because it overrides the default.
    m_pCPUSound->writeMemHandler(
        "Shared RAM",     0x0040, 0x03ff, s_writeSharedRAM,     this
    );
}

void
GamePacNPal::startUpROMs(
)
{
    //  Load in the Game ROMs.
    m_pSpaceGame->loadFile( 0xa000, m_pGameInfo->getGameId( ), "pap13b.cpu" );
    m_pSpaceGame->loadFile( 0xc000, m_pGameInfo->getGameId( ), "pap12b.cpu" );
    m_pSpaceGame->loadFile( 0xe000, m_pGameInfo->getGameId( ), "pap11b.cpu" );

    //  Load in the Sound ROMs.
    m_pSpaceSound->loadFile( 0xf000, m_pGameInfo->getGameId( ), "pap14.cpu" );
    
    //  Load in the graphic ROMs.
    m_pBufferChar->loadFile( 0x0000, m_pGameInfo->getGameId( ), "pap16.cpu" );
    m_pBufferSprite->loadFile( 0x0000, m_pGameInfo->getGameId( ), "pap15.vid" );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readScreenRotate 
//
//  Description:
//
//      This memory handler is invoked when the game reads from the
//      screen rotation memory location.  A read means flip the screen
//      (for the 2nd player in cocktail mode).
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the timer.
//
///////////////////////////////////////////////////////////////////////////////
DWord 
GamePacNPal::s_readScreenRotate( 
    DWord        dwAddress,
    ReadHandler* pHandler
)
{
    //  The data of the handler contains the pointer to our game.
    ASSERT( pHandler->getData( ) != NULL );
    GamePacNPal* pThis = ( GamePacNPal* )( pHandler->getData( ) );

    //  Flip the screen if needed.
    if( !( pThis->m_bScreenFlippedX ) )
    {
        //  Mark the screen as dirty.
        pThis->m_pDirty->clear( TRUE );

        //  Remember the state of the screen.
        pThis->m_bScreenFlippedX = 0x01;
        pThis->m_bScreenFlippedY = 0x01;
    }

    //  Get the value from the regular address space.
    return( CPU::sm_pSpace->getByte( dwAddress ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This memory handler is invoked when the game writes to the
//      screen rotation memory location.  A write means unflip the screen
//      (for normal operation).
//
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GamePacNPal::s_writeScreenRotate(
    DWord         dwAddress, 
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The data of the handler contains the pointer to our game.
    ASSERT( pHandler->getData( ) != NULL );
    GamePacNPal* pThis = ( GamePacNPal* )( pHandler->getData( ) );

    //  Unflip the screen if needed.
    if( pThis->m_bScreenFlippedX )
    {
        //  Mark the screen as dirty.
        pThis->m_pDirty->clear( TRUE );

        //  Remember the state of the screen.
        pThis->m_bScreenFlippedX = 0x00;
        pThis->m_bScreenFlippedY = 0x00;
    }

    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSharedRAM
//
//  Description:
//
//      This function is called when a value is written to the shared memory
//      from the sound CPU.  Pac'n'Pal uses a special handler since
//      we need to check for the sound CPU going into a tight loop.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
GamePacNPal::s_writeSharedRAM(
    DWord         dwAddress, 
    Byte          bValue, 
    WriteHandler* pHandler
)
{
    //  The data of the handler contains the pointer to our game.
    GamePacNPal* pThis = ( GamePacNPal* )( pHandler->getData( ) );

    //  We check for a command into the shared memory that indicates that
    //  the sound CPU is in a type loop.  In such a case, we give the
    //  game CPU a chance to perform.
    if( ( dwAddress == 0x41 ) && ( bValue == 2 ) )
    {
        CPU::sm_pCPU->setCyclesLeft( 0 );
    }

    //  Now write the value to shared RAM.
    pThis->m_pbSharedRAM[ dwAddress - pHandler->getStart( ) ] = bValue;
}
