///////////////////////////////////////////////////////////////////////////////
//
//  File:    robocop.cpp
//
//  Class:   GameRobocop
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the game class for Data East's Robocop.  This supports the
//      original board.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "robocop.h"
#include "gameinfo.h"
#include "space.h"
#include "buffer.h"



///////////////////////////////////////////////////////////////////////////////
//  Static declaration.
///////////////////////////////////////////////////////////////////////////////

//  Game Information.
GAME_INFO

    ASSIGN_GAME_ID,
        "robocop",

    ASSIGN_GAME_NAME,
        "Robocop",

    ASSIGN_GAME_VERSION,      
        REPLAY_MAJOR_REVISION, REPLAY_MINOR_REVISION, "1", "0",

    ASSIGN_GAME_CONTRIBUTORS,
        "Kevin Brisley",

    ASSIGN_GAME_REQD_FILES,
        "roboc_00.rom",
        "roboc_01.rom",
        "roboc_02.rom",
        "roboc_03.rom",
        "roboc_04.rom",
        "roboc_05.rom",
        "roboc_06.rom",
        "roboc_07.rom",
        "roboc_08.rom",
        "roboc_09.rom",
        "roboc_10.rom",
        "roboc_11.rom",
        "roboc_12.rom",
        "roboc_13.rom",
        "roboc_14.rom",
        "roboc_15.rom",
        "roboc_16.rom",
        "roboc_17.rom",
        "roboc_18.rom",
        "roboc_19.rom",
        "roboc_20.rom",
        "roboc_21.rom",
        "roboc_22.rom",
        "roboc_23.rom",

FOR( robocop, GameRobocop )


///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameRobocop
//
//  Description:
//
//      This is the main constructor for a Robocop game object.
//      
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameRobocop::GameRobocop(
    const KString& iName
)
:
    GameRobocopBase    ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameRobocop
//
//  Description:
//
//      This is the destructor for a Robocop game object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameRobocop::~GameRobocop(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Robocop game class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GameRobocop::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GameRobocop" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of the Robocop hardware.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameRobocop::startUpROMs(
)
{
    //  Load in the Game ROMs.  The odd & even bytes are split up so we 
    //  load skipping every other byte.
    m_pSpaceGame->loadFileSkip( 0x000000, GAME_ID( ), "roboc_05.rom" );
    m_pSpaceGame->loadFileSkip( 0x000001, GAME_ID( ), "roboc_01.rom" );
    m_pSpaceGame->loadFileSkip( 0x020000, GAME_ID( ), "roboc_04.rom" );
    m_pSpaceGame->loadFileSkip( 0x020001, GAME_ID( ), "roboc_00.rom" );

    //  Load in the Sound ROMs.
    m_pSpaceSound->loadFile( 0x8000, GAME_ID( ), "roboc_03.rom" );

    //  Load in the ROMs for the character set.
    m_pBufferChar->loadFile( 0x00000, GAME_ID( ), "roboc_22.rom" );
    m_pBufferChar->loadFile( 0x10000, GAME_ID( ), "roboc_23.rom" );

    //  Load in the ROMs for the first tile set.
    m_pBufferTile1->loadFile( 0x00000, GAME_ID( ), "roboc_18.rom" );
    m_pBufferTile1->loadFile( 0x10000, GAME_ID( ), "roboc_19.rom" );
    m_pBufferTile1->loadFile( 0x20000, GAME_ID( ), "roboc_20.rom" );
    m_pBufferTile1->loadFile( 0x30000, GAME_ID( ), "roboc_21.rom" );

    //  Load in the ROMs for the second tile set.
    m_pBufferTile2->loadFile( 0x00000, GAME_ID( ), "roboc_14.rom" );
    m_pBufferTile2->loadFile( 0x08000, GAME_ID( ), "roboc_15.rom" );
    m_pBufferTile2->loadFile( 0x10000, GAME_ID( ), "roboc_16.rom" );
    m_pBufferTile2->loadFile( 0x18000, GAME_ID( ), "roboc_17.rom" );

    //  Load in the ROMs for the sprite set.
    m_pBufferSprite->loadFile( 0x00000, GAME_ID( ), "roboc_07.rom" );
    m_pBufferSprite->loadFile( 0x10000, GAME_ID( ), "roboc_06.rom" );
    m_pBufferSprite->loadFile( 0x18000, GAME_ID( ), "roboc_09.rom" );
    m_pBufferSprite->loadFile( 0x28000, GAME_ID( ), "roboc_08.rom" );
    m_pBufferSprite->loadFile( 0x30000, GAME_ID( ), "roboc_11.rom" );
    m_pBufferSprite->loadFile( 0x40000, GAME_ID( ), "roboc_10.rom" );
    m_pBufferSprite->loadFile( 0x48000, GAME_ID( ), "roboc_13.rom" );
    m_pBufferSprite->loadFile( 0x58000, GAME_ID( ), "roboc_12.rom" );
}

void
GameRobocop::startUpMemMap(
)
{
    //  Let the base class perform the common initialization.
    GameRobocopBase::startUpMemMap( );

    //  The original Robocop board has a mystery chip which controls the
    //  main loop of the game.  Add a read/write handler for the mystery ASIC.
    m_pCPUGame->readMemHandler( 
        "Mystery ASIC",   0x180000, 0x18000f, s_readASIC,      this
    );
    m_pCPUGame->writeMemHandler(
        "Mystery ASIC",   0x180000, 0x18000f, s_writeASIC,     this
    );

    //  Initialize the mystery ASIC.
    for( DWord dwI = 0 ; dwI < 16 ; dwI += 1 )
    {
        m_abASIC[ dwI ] = 0;
    }
    m_abASIC[ 5 ] = 1;
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeASIC
//
//  Description:
//
//      This is a memory handler called when the game writes to the
//      mystery ASIC.
//
//  Parameters:
//
//      dwAddress (input)
//          The address written to.
//
//      bValue (input)
//          The value written.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameRobocop::s_writeASIC(
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The handler data contains a pointer to the game object.
    GameRobocop* pThis = ( GameRobocop* )( pHandler->getData( ) );

    //  Check for a special location.
    switch( dwAddress )
    {
        //  Writing to 0x180007 seems to modify the value in the ASIC (or
        //  at least it's a convenient spot to do so).
        case 0x180007: 
        {
            //  A value of 0xff is written during the main loop.
            if( bValue == 0xff ) 
            {
                //  Move to the next value skipping over invalid addresses
                //  in the jump table and cycling around after 27.
                pThis->m_abASIC[ 5 ] += 1;
                if( 
                    ( pThis->m_abASIC[ 5 ] ==  0 ) || 
                    ( pThis->m_abASIC[ 5 ] ==  4 ) || 
                    ( pThis->m_abASIC[ 5 ] == 10 ) || 
                    ( pThis->m_abASIC[ 5 ] == 12 ) || 
                    ( pThis->m_abASIC[ 5 ] == 14 ) ||
                    ( pThis->m_abASIC[ 5 ] == 18 ) || 
                    ( pThis->m_abASIC[ 5 ] == 21 )
                )
                {
                    pThis->m_abASIC[ 5 ] += 1;
                }
                else
                if( pThis->m_abASIC[ 5 ] == 17 )
                {
                    pThis->m_abASIC[ 5 ] += 2;
                }
                else
                if( pThis->m_abASIC[ 5 ] > 27 )
                {
                    pThis->m_abASIC[ 5 ] = 1;
                }
            }
            break;
        }
    }

    //  If not handled then just set the value in memory.
    pThis->m_pSpaceGame->setByte( dwAddress, bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readASIC 
//
//  Description:
//
//      This memory handler is invoked when the game reads from the mystery
//      ASIC.
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the timer.
//
///////////////////////////////////////////////////////////////////////////////
DWord 
GameRobocop::s_readASIC( 
    DWord        dwAddress,
    ReadHandler* pHandler
)
{
    //  The handler data contains a pointer to the game object.
    GameRobocop* pThis = ( GameRobocop* )( pHandler->getData( ) );

    //  Check for a special location.
    switch( dwAddress )
    {
        case 0x180001: 
            pThis->m_abASIC[ 1 ] += 1;   return( pThis->m_abASIC[ 1 ] );
        case 0x180003: 
            pThis->m_abASIC[ 3 ] += 1;   return( pThis->m_abASIC[ 3 ] );
        case 0x180005:                      
                                         return( pThis->m_abASIC[ 5 ] );
        case 0x180007: 
            pThis->m_abASIC[ 7 ] += 1;   return( pThis->m_abASIC[ 7 ] );
        case 0x180009: 
            pThis->m_abASIC[ 9 ] += 1;   return( pThis->m_abASIC[ 9 ] );
    }

    //  If not handled then just return the value from memory.
    return( pThis->m_pSpaceGame->getByte( dwAddress ) );
}
