///////////////////////////////////////////////////////////////////////////////
//
//  File:    repfactl.cpp
//
//  Class:   ReplayFactoryLinux
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is the concrete factory for creating Linux based
//      objects used in Replay.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repfactl.h"
#include "canvasl.h"
#include "keybx.h"
#include "joyl.h"
#include "mousex.h"
#include "soundl.h"
#include "diskx.h"
#include "networkx.h"
#include "clockx.h"
#ifdef RELEASE
#include "6502nba.h"
#include "6809pla.h"
#include "68000nca.h"
#include "z80nba.h"
#include "z80mdc.h"
#else
#include "6502mfc.h"
#include "6809kbc.h"
#include "68000bsc.h"
#include "z80mdc.h"
#endif
#ifdef GAMES_shared
#include "gamfactx.h"
#else
#include "gamfact.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  File Static Variables.
///////////////////////////////////////////////////////////////////////////////

//  The singleton instance.
ReplayFactoryLinux ReplayFactoryLinux::sm_replayFactoryLinux( "ReplayFactory" );


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ReplayFactoryLinux
//
//  Description:
//
//      This is the main constructor for the Linux Unix/X factory
//      object.  It is protected because it is a singleton and therefore
//      cannot be instantiated by anyone but itself or a derived class.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryLinux::ReplayFactoryLinux(
    const KString& iName
)
:
    ReplayFactory ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ReplayLinux
//
//  Description:
//
//      This is the main destructor for the Linux Unix/X Replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryLinux::~ReplayFactoryLinux(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ReplayFactoryLinux::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ReplayFactoryLinux" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCanvas
//
//  Description:
//
//      This creates the default canvas object for Linux.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the canvas.
//
///////////////////////////////////////////////////////////////////////////////
Canvas*
ReplayFactoryLinux::createCanvas(
    const KString& iName
)
{
    return( CanvasLinux::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createKeyboard
//
//  Description:
//
//      This creates the default keyboard object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the keyboard.
//
///////////////////////////////////////////////////////////////////////////////
Keyboard*
ReplayFactoryLinux::createKeyboard(
    const KString& iName
)
{
    return( KeyboardUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createJoystick
//
//  Description:
//
//      This creates the default joystick object for Linux.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the joystick.
//
///////////////////////////////////////////////////////////////////////////////
Joystick*
ReplayFactoryLinux::createJoystick(
    const KString& iName
)
{
    return( JoystickLinux::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createMouse
//
//  Description:
//
//      This creates the default mouse object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the mouse.
//
///////////////////////////////////////////////////////////////////////////////
Mouse*
ReplayFactoryLinux::createMouse(
    const KString& iName
)
{
    return( MouseUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createSound
//
//  Description:
//
//      This creates the default sound object for Linux.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the sound.
//
///////////////////////////////////////////////////////////////////////////////
Sound*
ReplayFactoryLinux::createSound(
    const KString& iName
)
{
    return( new SoundLinux( iName ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createDisk
//
//  Description:
//
//      This creates the default disk object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the disk.
//
///////////////////////////////////////////////////////////////////////////////
Disk*
ReplayFactoryLinux::createDisk(
    const KString& iName
)
{
    return( new DiskUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createNetwork
//
//  Description:
//
//      This creates the default network object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the network.
//
///////////////////////////////////////////////////////////////////////////////
Network*
ReplayFactoryLinux::createNetwork(
    const KString& iName
)
{
    return( new NetworkUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createClock
//
//  Description:
//
//      This creates the default clock object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the clock.
//
///////////////////////////////////////////////////////////////////////////////
Clock*
ReplayFactoryLinux::createClock(
    const KString& iName
)
{
    return( new ClockUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createGameFactory
//
//  Description:
//
//      This creates the default game factory object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the game factory.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory*
ReplayFactoryLinux::createGameFactory(
    const KString& iName
)
{
#ifdef GAMES_shared
    return( GameFactoryUnixX::s_build( iName ) );
#else
    return( GameFactory::s_build( iName ) );
#endif
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCPU
//
//  Description:
//
//      This creates CPUs for use with the Linux platform.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      nCPUType (input)
//          The type of CPU to create.
//
//      pGame (input)
//          The game that the CPU belongs to.
//
//      pSpace (input)
//          The address space to provide to the new CPU.
//
//      ePurpose (input)
//          The purpose of CPU to create.
//
//      bCONSERVATIVE (input)
//          A TEMPORARY FLAG USED TO ALLOW US TO PICK THE 'C' BASED Z80
//          CORE DUE TO THE INSTABILITY OF THE ASM.
//
//  Returns:
//
//      A pointer to the CPU.
//
///////////////////////////////////////////////////////////////////////////////
CPU*
ReplayFactoryLinux::createCPU(
    const KString&        iName,
    const CPU::CPUType    eCPUType,
    Game*                 pGame,
    AddrSpace*            pSpace,
    const CPU::CPUPurpose ePurpose      /* = CPU::GAME */,
    const Byte            bCONSERVATIVE /* = FALSE */
)
{
    ASSERT( eCPUType < CPU::NUM_CPU_TYPES );
 
    //  Create the appropriate CPU.  We can use the nice, speedy ASM versions
    //  of the core on Linux.
    switch( eCPUType )
    {
#ifdef RELEASE
        case CPU::CPU_6502:
        {
            return( CPU6502NBA::s_build( iName, pGame, pSpace, ePurpose ) );
        }
        
        case CPU::CPU_6809:
        {
            return( CPU6809PLA::s_build( iName, pGame, pSpace, ePurpose ) );
        } 

        case CPU::CPU_68000:
        {
            return( CPU68000NCA::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        case CPU::CPU_Z80:
        {
            if( bCONSERVATIVE )
            {
                return( CPUZ80MDC::s_build( iName, pGame, pSpace, ePurpose ) );
            }
            else
            {
                return( CPUZ80NBA::s_build( iName, pGame, pSpace, ePurpose ) );
            }
        }
#else
        case CPU::CPU_6502:
        {
            return( CPU6502MFC::s_build( iName, pGame, pSpace, ePurpose ) );
        }
        
        case CPU::CPU_6809:
        {
            return( CPU6809KBC::s_build( iName, pGame, pSpace, ePurpose ) );
        } 

        case CPU::CPU_68000:
        {
            return( CPU68000BSC::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        case CPU::CPU_Z80:
        {
            return( CPUZ80MDC::s_build( iName, pGame, pSpace, ePurpose ) );
        }
#endif

        default:
        {
            fatalError( 
                "CPUFactory::createCPU - unknown CPU Type (%d).", eCPUType
            );
        }
    }

    //  Should never arrive here.
    return( NULL );
}
