///////////////////////////////////////////////////////////////////////////////
//
//  File:    movie.cpp
//
//  Class:   MovieMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a base class for Meddlers that record and
//      play back movies of Replay games.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "movie.h"
#include "replay.h"
#include "game.h"
#include "gameinfo.h"
#include "disk.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: MovieMeddler
//
//  Description:
//
//      This is the main constructor for a movie meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//      bAllowCreate (input)
//          This indicates whether the movie file is allowed to be created 
//          (record) or not (play).
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MovieMeddler::MovieMeddler(
    const KString&  iName,
    Game*           pGame,
    const Byte      bAllowCreate
)
:
    Meddler           ( iName, pGame ),
    m_bAllowCreate    ( bAllowCreate ),
    m_dwTime          ( 0 ),
    m_frameList       ( 100 ),
    m_pFile           ( NULL ),
    m_pKeyboard       ( Replay::s_instance( ).getKeyboard( ) ),
    m_pJoystick       ( Replay::s_instance( ).getJoystick( ) ),
    m_rCtrlMapList    ( pGame->getCtrlMapList( ) )
{
    ASSERT( m_pGame     != NULL );
    ASSERT( m_pKeyboard != NULL );
    ASSERT( m_pJoystick != NULL );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a Meddler object.  It is to
//      be called from the build method and performs functions we would 
//      normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MovieMeddler::init(
)
{
    //  The disk object.
    Disk* pDisk = Replay::s_instance( ).getDisk( );

    //  Call the base class first.
    Meddler::init( );

    ASSERT( m_pGame != NULL );

    //  Create the name of the movie file.
    ASSERT( m_pGame->getGameInfo( ) != NULL );
    KString fileName  = pDisk->getPath( );
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += "movie";
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += m_pGame->getGameInfo( )->getGameId( );
    fileName         += ".mov";

    //  Ensure the directory components exist.
    pDisk->makeDirectoryComponents( fileName );

    //  Open the application file.
    m_pFile = new AppFile( fileName, m_bAllowCreate );

    //  If the file could not be opened then mark the meddler as complete.
    if( !m_pFile->isOpen( ) )
    {
        m_bComplete = TRUE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~MovieMeddler
//
//  Description:
//
//      This is the destructor for a movie meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MovieMeddler::~MovieMeddler(
)
{
    //  Free the list of frames.
    for( DWord dwI = 0 ; dwI < m_frameList.entries( ) ; dwI += 1 )
    {
        delete m_frameList[ dwI ];
    }
    m_frameList.clear( );

    //  Delete the movie file.
    delete m_pFile;
}
