///////////////////////////////////////////////////////////////////////////////
//
//  File:    play.cpp
//
//  Class:   MoviePlayMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to play a movie of the current game being 
//      emulated.  Movies to be played must have previously been recorded
//      using the MovieRecordMeddler class.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "play.h"
#include "game.h"
#include "ctrlmap.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a movie player meddler object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//      pGame (input)
//          The game the meddler is associated with.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
MoviePlayMeddler*
MoviePlayMeddler::s_build(
    const KString& iName,
    Game*          pGame
)
{
    //  Create the new object.
    MoviePlayMeddler* pThis = new MoviePlayMeddler( iName, pGame );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: MoviePlayMeddler
//
//  Description:
//
//      This is the main constructor for a movie player meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MoviePlayMeddler::MoviePlayMeddler(
    const KString&  iName,
    Game*           pGame
)
:
    MovieMeddler   ( iName, pGame, FALSE ),
    m_dwFrame      ( 0 ),
    m_dwEndTime    ( 0 )
{
    //  Initialization is done in init( ).
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a Movie Player Meddler object.  
//      It is to be called from the build method and performs functions we 
//      would normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MoviePlayMeddler::init(
)
{
    //  First let the base class initialize.
    MovieMeddler::init( );

    //  If the file was not opened successfully then just return.
    if( !m_pFile->isOpen( ) )
    {
        return;
    }

    //  Load the game's current state from the movie file.
    m_pGame->load( m_pFile );

    //  Read the end time of the movie.
    m_pFile->read( ( Byte* )&m_dwEndTime, sizeof( m_dwEndTime ) );
    CONFIRM( 
        m_pFile->count( ) == sizeof( m_dwEndTime ), "Error reading movie file."
    );
   
    //  Read the number of movie frames.
    DWord dwNumFrames; 
    m_pFile->read( ( Byte* )&dwNumFrames, sizeof( dwNumFrames ) );
    CONFIRM( 
        m_pFile->count( ) == sizeof( dwNumFrames ), "Error reading movie file."
    );
   
    //  Read in each frame and add it to the list of frames.
    for( DWord dwI = 0 ; dwI < dwNumFrames ; dwI += 1 )
    {
        //  A new frame to hold the frame read from the movie file.
        Frame* pFrame = new Frame;
        
        //  Read the frame from the file.
        m_pFile->clearTotal( );
        m_pFile->read( 
            ( Byte* )&( pFrame->m_dwTime ), sizeof( pFrame->m_dwTime )
        );
        m_pFile->read( 
            ( Byte* )&( pFrame->m_dwCtrlMapIndex ),
            sizeof( pFrame->m_dwCtrlMapIndex )
        );
        m_pFile->read( 
            &( pFrame->m_cModifier ), sizeof( pFrame->m_cModifier ) 
        );
        m_pFile->read( &( pFrame->m_bState ), sizeof( pFrame->m_bState ) );

        //  Make sure the read was successful.
        CONFIRM( 
            m_pFile->total( ) == 
                sizeof( pFrame->m_dwTime ) + 
                sizeof( pFrame->m_dwCtrlMapIndex ) + 
                sizeof( pFrame->m_cModifier ) + 
                sizeof( pFrame->m_bState ),
            "Error reading movie file frame %d.",
            dwI
        );

        //  Add it to the list.
        m_frameList.add( pFrame );
    }

    //  We're all done with the movie file so close it.
    m_pFile->close( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~MoviePlayMeddler
//
//  Description:
//
//      This is the destructor for a movie player meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MoviePlayMeddler::~MoviePlayMeddler(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the movie player meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
MoviePlayMeddler::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "MoviePlayMeddler" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: enableHardware
//
//  Description:
//
//      This member is called to either enable or disable the hardware input
//      of the control maps involved in the game so that the user can't 
//      interfere with the playback of the movie.
//
//  Parameters:
//
//      bEnable (input)
//          Enable or disable the hardware input.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MoviePlayMeddler::enableHardware(
    Byte bEnable
) const
{
    //  Run through the list of control maps.
    for( DWord dwI = 0 ; dwI < m_rCtrlMapList.entries( ) ; dwI += 1 )
    {
        m_pKeyboard->getSwitch( 
            m_rCtrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ) 
        )->enableHardware(  bEnable );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called periodically when the meddler is to update 
//      itself.  It checks if any of the control map values are to be
//      changed for the current time slice and if so, changes them.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MoviePlayMeddler::update(
)
{
    //  Update should not be called if the meddler has completed.
    ASSERT( !m_bComplete );

    //  Increment the timer.
    m_dwTime += 1;

    //  Change any control map key states for the current time slice.
    while( 
        ( m_dwFrame < m_frameList.entries( ) ) && 
        ( m_frameList[ m_dwFrame ]->m_dwTime == m_dwTime )
    )
    {
        //  The current frame of interest.
        Frame* pFrame = m_frameList[ m_dwFrame ];

        //  Right now we only have keyboard control.
        ASSERT( pFrame->m_cModifier = 'K' );

        //  Adjust the control map key.
        m_pKeyboard->getSwitch( 
            m_rCtrlMapList[ pFrame->m_dwCtrlMapIndex ]->get( CtrlMap::KEYBOARD )
        )->virtualOn( pFrame->m_bState );

        //  Move on to the next frame.
        m_dwFrame += 1;
    }

    //  If we've reached the end of the movie then terminate.
    if( m_dwTime > m_dwEndTime )
    {
        terminate( );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: activate
//
//  Description:
//
//      This member is called when the meddler is activated.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MoviePlayMeddler::activate(
)
{
    //  Disable hardware input to the control maps.
    enableHardware( FALSE );

    //  Call the base class.
    MovieMeddler::activate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deactivate
//
//  Description:
//
//      This member is called when the meddler is deactivated.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MoviePlayMeddler::deactivate(
)
{
    //  Reenable hardware input to the control maps.
    enableHardware( TRUE );

    //  Call the base class.
    MovieMeddler::deactivate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getDecoration
//
//  Description:
//
//      This member is to return the decoration that is to be used to indicate
//      to the user that the meddler is meddling. 
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The decoration.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
MoviePlayMeddler::getDecoration(
) const
{
    //  The name of the class.
    static const KString decoration( "\x02" );

    //  We want the decoration to flash every so often.
    static DWord dwCount = 0;

    
    //  Increment the count modulo 120.
    dwCount = ( dwCount + 1 ) % 120;

    //  Return either the decoration or nothing.
    if( dwCount < 60 )
    {
        return( decoration );
    }
    else
    {
        return( KStringNULL );
    }
}
