///////////////////////////////////////////////////////////////////////////////
//
//  File:    record.cpp
//
//  Class:   MovieRecordMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to record a movie of the current game being 
//      emulated.  It can be played back at a later date with the 
//      MoviePlayMeddler.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "record.h"
#include "game.h"
#include "ctrlmap.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a movie recorder meddler object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//      pGame (input)
//          The game the meddler is associated with.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
MovieRecordMeddler*
MovieRecordMeddler::s_build(
    const KString& iName,
    Game*          pGame
)
{
    //  Create the new object.
    MovieRecordMeddler* pThis = new MovieRecordMeddler( iName, pGame );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: MovieRecordMeddler
//
//  Description:
//
//      This is the main constructor for a movie recorder meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MovieRecordMeddler::MovieRecordMeddler(
    const KString&  iName,
    Game*           pGame
)
:
    MovieMeddler      ( iName, pGame, TRUE ),
    m_pbLastKeyStates ( NULL )
{
    //  Initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a Movie Recorder Meddler object.  
//      It is to be called from the build method and performs functions we 
//      would normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MovieRecordMeddler::init(
)
{
    //  First let the base class initialize.
    MovieMeddler::init( );

    //  If the file was not opened successfully then just return.
    if( !m_pFile->isOpen( ) )
    {
        return;
    }

    //  Save the game's current state to the movie file.
    m_pGame->save( m_pFile );

    //  Allocate a list that will contain the last values of each key 
    //  corresponding to the control maps for the game.
    ASSERT( m_rCtrlMapList.entries( ) > 0 );
    m_pbLastKeyStates = new Byte[ m_rCtrlMapList.entries( ) ];

    //  Record the initial values of the control map keys.  We treat 
    //  joystick movements as keys since we are dealing only with digital
    //  controllers at the moment.
    for( DWord dwI = 0 ; dwI < m_rCtrlMapList.entries( ) ; dwI += 1 )
    {
        m_pbLastKeyStates[ dwI ] = 
            m_pKeyboard->switchOn( 
                m_rCtrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ) 
            ) ||
            m_pJoystick->switchOn( 
                m_rCtrlMapList[ dwI ]->get( CtrlMap::JOYSTICK ) 
            );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~MovieRecordMeddler
//
//  Description:
//
//      This is the destructor for a movie recorder meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
MovieRecordMeddler::~MovieRecordMeddler(
)
{
    //  Free up the list of last key states
    delete [] m_pbLastKeyStates;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the movie recorder meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
MovieRecordMeddler::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "MovieRecordMeddler" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called periodically when the meddler is to update 
//      itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MovieRecordMeddler::update(
)
{
    //  Update should not be called if the meddler has completed.
    ASSERT( !m_bComplete );

    //  Increment the timer.
    m_dwTime += 1;

    //  If any of the control map key states have changed values then add a 
    //  frame to the list indicating the new value.
    for( DWord dwI = 0 ; dwI < m_rCtrlMapList.entries( ) ; dwI += 1 )
    {
        //  The current state of the control map key.
        Byte bPressed = 
            m_pKeyboard->switchOn( 
                m_rCtrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ) 
            ) ||
            m_pJoystick->switchOn( 
                m_rCtrlMapList[ dwI ]->get( CtrlMap::JOYSTICK ) 
            );

        //  Has the state changed?
        if( bPressed != m_pbLastKeyStates[ dwI ] )
        {
            //  A new frame to hold the changed value of the control map.
            Frame* pFrame = new Frame;
        
            //  Add it to the list.
            pFrame->m_dwTime         = m_dwTime;
            pFrame->m_dwCtrlMapIndex = dwI;
            pFrame->m_cModifier      = 'K';
            pFrame->m_bState         = bPressed;
            m_frameList.add( pFrame );

            //  Record the last state of the control map key.
            m_pbLastKeyStates[ dwI ] = bPressed;
        }
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: terminate
//
//  Description:
//
//      This member is called when the recording is to be stopped.  At this
//      time, the list of frames is written out to the movie file.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
MovieRecordMeddler::terminate(
)
{
    //  If we've already completed then just return.
    if( m_bComplete )
    {
        return;
    }

    //  Call the base class.
    MovieMeddler::terminate( );

    //  First write out the time value.
    m_pFile->write( ( Byte* )&m_dwTime, sizeof( m_dwTime ) );
    CONFIRM( 
        m_pFile->count( ) == sizeof( m_dwTime ), "Error writing movie file."
    );

    //  Next write out the number of frames.
    DWord dwNumFrames = m_frameList.entries( );
    m_pFile->write( ( Byte* )&dwNumFrames, sizeof( dwNumFrames ) );
    CONFIRM( 
        m_pFile->count( ) == sizeof( dwNumFrames ), "Error writing movie file."
    );
    
    //  OK, write the frames we've accumulated out to the movie file.
    for( DWord dwI = 0 ; dwI < dwNumFrames ; dwI += 1 )
    {
        //  A convenient pointer to the current frame.
        Frame* pFrame = m_frameList[ dwI ];

        //  Write the frame to the file.
        m_pFile->clearTotal( );
        m_pFile->write( 
            ( Byte* )&( pFrame->m_dwTime ), sizeof( pFrame->m_dwTime )
        );
        m_pFile->write( 
            ( Byte* )&( pFrame->m_dwCtrlMapIndex ),
            sizeof( pFrame->m_dwCtrlMapIndex )
        );
        m_pFile->write( 
            &( pFrame->m_cModifier ), sizeof( pFrame->m_cModifier ) 
        );
        m_pFile->write( &( pFrame->m_bState ), sizeof( pFrame->m_bState ) );

        //  Make sure the write was successful.
        CONFIRM( 
            m_pFile->total( ) == 
                sizeof( pFrame->m_dwTime ) + 
                sizeof( pFrame->m_dwCtrlMapIndex ) + 
                sizeof( pFrame->m_cModifier ) + 
                sizeof( pFrame->m_bState ),
            "Error writing movie file frame %d.",
            dwI
        );
    } 

    //  We're now done with the file so close it.
    m_pFile->close( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getDecoration
//
//  Description:
//
//      This member is to return the decoration that is to be used to indicate
//      to the user that the meddler is meddling. 
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The decoration.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
MovieRecordMeddler::getDecoration(
) const
{
    //  The name of the class.
    static const KString decoration( "\x01" );

    //  We want the decoration to flash every so often.
    static DWord dwCount = 0;

    
    //  Increment the count modulo 120.
    dwCount = ( dwCount + 1 ) % 120;

    //  Return either the decoration or nothing.
    if( dwCount < 60 )
    {
        return( decoration );
    }
    else
    {
        return( KStringNULL );
    }
}
