///////////////////////////////////////////////////////////////////////////////
//
//  File:    server.cpp
//
//  Class:   NetGameServerMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to handle the duties of a server during a net
//      game.  The server coordinates the execution of the game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "server.h"
#include "game.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a net game server meddler object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//      pGame (input)
//          The game the meddler is associated with.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
NetGameServerMeddler*
NetGameServerMeddler::s_build(
    const KString& iName,
    Game*          pGame
)
{
    //  Create the new object.
    NetGameServerMeddler* pThis = new NetGameServerMeddler( iName, pGame );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetGameServerMeddler
//
//  Description:
//
//      This is the main constructor for a net game server meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameServerMeddler::NetGameServerMeddler(
    const KString&  iName,
    Game*           pGame
)
:
    NetGameMeddler ( iName, pGame )
{
    //  Initialization is done in init( ).
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a net game server meddler object.  
//      It is to be called from the build method and performs functions we 
//      would normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameServerMeddler::init(
)
{
    //  First let the base class initialize.
    NetGameMeddler::init( );

    //  Since we're the server, we do the assignment of player numbers.
    assignPlayerNumbers( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetGameServerMeddler
//
//  Description:
//
//      This is the destructor for a net game server meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameServerMeddler::~NetGameServerMeddler(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the net game server meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetGameServerMeddler::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "NetGameServerMeddler" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: assignPlayerNumbers
//
//  Description:
//
//      This member is called to assign the player numbers for the clients
//      involved in the network game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameServerMeddler::assignPlayerNumbers(
)
{
    //  The player number message.
    Network::MsgPlayer* pMsgPlayer = ( Network::MsgPlayer* )m_pMsg;

    //  The server is always player 1.
    setPlayerNum( 1 );

    //  Send a message to each client with their player numbers.  
    //  It is important to leave the call to retrieve
    //  the number of connections inside the for( ) since the number 
    //  may change if connections are dropped during the loop.
    for( DWord dwI = 0 ; dwI < m_pNetwork->getNumConnections( ) ; dwI += 1 )
    {
        //  Create the message.
        pMsgPlayer->eType      = Network::MSG_PLAYER;        
        pMsgPlayer->bPlayerNum = ( Byte )( dwI + 2 );

        //  Send the message.
        m_pNetwork->writeMsg( dwI, m_pMsg );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called periodically when the meddler is to update 
//      itself.  It gathers messages from the clients and sends out the
//      current state of the controls.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameServerMeddler::update(
)
{
    //  The network control message.
    Network::MsgCtrl* pMsgCtrl = ( Network::MsgCtrl* )m_pMsg;

    //  Update should not be called if the meddler has completed.
    ASSERT( !m_bComplete );

    //  The state of the keys.
    DWord dwControlState;


    //  Get the current physical state of the controls.
    getKeyState( &dwControlState );

    //  We must receive a control message from each player before we
    //  can continue.  It is important to leave the call to retrieve
    //  the number of connections inside the for( ) since the number 
    //  may change if connections are dropped during the loop.
    for( DWord dwI = 0 ; dwI < m_pNetwork->getNumConnections( ) ; dwI += 1 )
    {
        //  Keep looping until the control message is obtained.
        for( ; ; )
        {
            //  Read the message for the current connection (player).
            m_pNetwork->readMsg( dwI, m_pMsg );

            //  Is the message the control message?
            if( m_pMsg->eType == Network::MSG_CTRL )
            {
                //  Add the state of the controls to the combined state.
                dwControlState |= pMsgCtrl->dwCtrlMask;

                //  Now that we've received the control message for this
                //  connection, we can move on to the next one.
                break;
            }
            //  Has the connection closed?
            else
            if( m_pMsg->eType == Network::MSG_CLOSE )
            {
                //  If there are no more players then terminate the meddler.
                if( m_pNetwork->getNumConnections( ) < 1 )
                {
                    terminate( );
                }

                //  If it was the last connection that dropped then get out
                //  of the loop.
                if( dwI >= m_pNetwork->getNumConnections( ) )
                {
                    break;
                }

                //  Now that a connection has dropped out, reassign the
                //  player numbers.
                assignPlayerNumbers( );
            }
        }
    }

    //  At this point, all control messages have been received.  Therefore,
    //  dwControlState contains the combined control state for all connections.
    //  Send out the combined state to all connections.
    pMsgCtrl->eType      = Network::MSG_CTRL;
    pMsgCtrl->dwCtrlMask = dwControlState;
    m_pNetwork->writeMsg( m_pMsg );

    //  Now set the state of our controls to the combined control state.
    setKeyState( &dwControlState );
}
