///////////////////////////////////////////////////////////////////////////////
//
//  File:    bitmap.cpp
//
//  Class:   Bitmap
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a bitmap which is of course just an array
//      of bits representing an image.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <string.h>

//  Application Headers.
#include "reptypes.h"
#include "bitmap.h"
#include "replay.h"
#include "appfile.h"
#include "canvas.h"
#include "matrix.h"
#include "clip.h"




///////////////////////////////////////////////////////////////////////////////
//  Static Data Declaration.
///////////////////////////////////////////////////////////////////////////////
Matrix<Byte> Bitmap::sm_matrix;



///////////////////////////////////////////////////////////////////////////////
//  Constants.
///////////////////////////////////////////////////////////////////////////////
const Word PCX_PALETTE_SIZE = 769;




///////////////////////////////////////////////////////////////////////////////
//
//  Function: Bitmap
//
//  Description:
//
//      This constructor creates a blank bitmap of the specified dimensions.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the bitmap.
//
//      dwHeight (input)
//          The height of the bitmap.
//
//      bScreen (input)
//          Indicates whether or not this bitmap is a screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap::Bitmap(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScreen   /* = FALSE */
)
:
    RepBase           ( iName ),
    m_pCanvas         ( Replay::s_instance( ).getCanvas( ) ),
    m_dwWidth         ( dwWidth ),
    m_dwHeight        ( dwHeight ),
    m_pbBuffer        ( NULL ),
    m_ppbRows         ( NULL ),
    m_ppbCols         ( NULL ),
    m_pFullClipping   ( NULL ),
    m_pnMapX          ( NULL ),
    m_pnMapY          ( NULL ),
    m_bScreen         ( bScreen )
{
    //  Initialization is to be done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called from the build method to initialize the object.
//      It allows virtual methods to be called during construction.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::init(
)
{
    //  Set the full clipping region.
    m_pFullClipping = new Clipping( 0, m_dwWidth - 1, 0, m_dwHeight - 1 );

    //  Allocate the buffer for the bitmap if it hasn't already been allocated
    //  by a sub-class.
    if( m_pbBuffer == NULL )
    {
        m_pbBuffer = new Byte[ m_dwWidth * m_dwHeight ];
    }

    //  The array of rows is used to conveniently access the rows of the
    //  bitmap.
    m_ppbRows = new Byte*[ m_dwHeight ];
    for( DWord dwRow = 0 ; dwRow < m_dwHeight ; dwRow += 1 )
    {
        m_ppbRows[ dwRow ] = &( m_pbBuffer[ dwRow * m_dwWidth ] );
    }

    //  The array of columns isn't very useful to access the actual columns
    //  since the memory is contiguous in the row direction.  However, it is
    //  a good place holder since when the screen is transposed, the columns
    //  become the rows.
    m_ppbCols = new Byte*[ m_dwWidth ];
    for( DWord dwCol = 0 ; dwCol < m_dwWidth ; dwCol += 1 )
    {
        m_ppbCols[ dwCol ] = &( m_pbBuffer[ dwCol * m_dwHeight ] );
    }

    //  If the screen is transposed then we need to swap the X/Y axis.  However,
    //  we don't need to transpose the bitmap since the bitmap was just
    //  initialized and doesn't yet contain any meaningful data.
    if( m_pCanvas->getTranspose( ) )
    {
        swapXY( );
    }

    //  Log on to the canvas so that we will be notified when the canvas
    //  orientation changes.
    m_pCanvas->logon( this );

    //  Clear the bitmap.
    clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Bitmap
//
//  Description:
//
//      This is the destructor for a bitmap.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap::~Bitmap(
)
{
    //  Log off from the canvas so that we will no longer be notified
    //  of canvas orientation changes.
    m_pCanvas->logoff( this );

    //  Free the clipping region.
    ASSERT( m_pFullClipping != NULL );
    delete m_pFullClipping;

    //  Get rid of the mappings.
    delete [] m_pnMapX;
    delete [] m_pnMapY;

    //  Free the buffers.
    delete [] m_pbBuffer;
    delete [] m_ppbRows;
    delete [] m_ppbCols;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the bitmap class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Bitmap::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Bitmap" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadPCX
//
//  Description:
//
//      This constructor loads the specified .PCX file into the bitmap.
//
//  Parameters:
//
//      pcxFileName (input)
//          The name of the PCX file to load.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::loadPCX(
    const KString&  pcxFileName
)
{
    //  The PCX Header.
    PCXHeader pcxHeader;

    //  The width & height of the PCX image.
    DWord dwPCXWidth;
    DWord dwPCXHeight;
 
    //  The length of the picture data.
    DWord dwLength;

    //  A value read from the PCX data.
    Byte bValue;

    //  The number of times the value repeats.
    Byte bCount;

    //  The current row & column of the image.
    DWord dwRow;
    DWord dwColumn;

    //  Open the input file.
    AppFile pcxFile( pcxFileName );


    //  If the file couldn't be opened then just return.
    if( pcxFile.isOpen( ) == FALSE )
    {
        return;
    }

    //  Load in the header of the .PCX.
    pcxFile.read( ( Byte* )&pcxHeader, sizeof( pcxHeader ) ); 
    CONFIRM( 
        pcxFile.count( ) == sizeof( pcxHeader ), 
        "Corrupt %s.", pcxFileName.data( ) 
    );

    //  Calculate the dimensions.
#ifdef BYTE_LS_FIRST
    dwPCXWidth  = pcxHeader.wMaxX - pcxHeader.wMinX + 1;
    dwPCXHeight = pcxHeader.wMaxY - pcxHeader.wMinY + 1;
#else
    dwPCXWidth  = 
        (
            ( ( pcxHeader.wMaxX & 0x00ff ) << 8 ) | 
            ( ( pcxHeader.wMaxX & 0xff00 ) >> 8 )
        ) - 
        (
            ( ( pcxHeader.wMinX & 0x00ff ) << 8 ) | 
            ( ( pcxHeader.wMinX & 0xff00 ) >> 8 )
        ) + 1;
    dwPCXHeight  = 
        (
            ( ( pcxHeader.wMaxY & 0x00ff ) << 8 ) | 
            ( ( pcxHeader.wMaxY & 0xff00 ) >> 8 )
        ) - 
        (
            ( ( pcxHeader.wMinY & 0x00ff ) << 8 ) | 
            ( ( pcxHeader.wMinY & 0xff00 ) >> 8 )
        ) + 1;
#endif

    //  Calculate the length of the data.
    dwLength = pcxFile.length( ) - sizeof( pcxHeader ) - PCX_PALETTE_SIZE;

    //  Decode the data.
    dwRow = 0;
    dwColumn = 0;
    while( dwLength > 0 )
    {
        //  Get the next byte.
        pcxFile.read( &bValue, sizeof( Byte ) );
        dwLength -= 1;
        
        //  If the top two bits are set then we have a repeating value,
        //  otherwise we have a single value.
        if( ( bValue & 0xc0 ) == 0xc0 )
        {
            //  The low 6 bits are the number of repeats.
            bCount = bValue & 0x3f;

            //  Get the value that is repeated.
            pcxFile.read( &bValue, sizeof( Byte ) );
            dwLength -= 1;

            //  Place the value into the buffer.
            while( bCount > 0 )
            {
                //  Clip it.  Remember to use the get() functions instead
                //  of the member data so that we get the non-transposed values.
                if( ( dwColumn < getWidth( ) ) && ( dwRow < getHeight( ) ) )
                {
                    setPixel( dwColumn, dwRow, bValue );
                }
                bCount -= 1;
                dwColumn += 1;
                if( dwColumn >= dwPCXWidth )
                {
                    dwRow += 1;
                    dwColumn = 0;
                }
            }
        }
        else
        {
            //  Clip it.  Remember to use the get() functions instead
            //  of the member data so that we get the non-transposed values.
            if( ( dwColumn < getWidth( ) ) && ( dwRow < getHeight( ) ) )
            {
                setPixel( dwColumn, dwRow, bValue );
            }
            dwColumn += 1;
            if( dwColumn >= dwPCXWidth )
            {
                dwRow += 1;
                dwColumn = 0;
            }
        }
    }
           
    //  Close the file.
    pcxFile.close( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: changeOrientation
//
//  Description:
//
//      This function is called when the orientation of the bitmap is to
//      be changed.  This can include a rotation of 90 degrees in a counter-
//      clockwise direction as well as a flip in either the X or Y direction.
//
//  Parameters:
//
//      bTranspose (input)
//          Should the image be transposed?
//
//      bFlipX (input)
//          Should the image be flipped in the X direction?
//
//      bFlipY (input)
//          Should the image be flipped in the Y direction?
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::changeOrientation(
    const Byte bTranspose,
    const Byte bFlipX,
    const Byte bFlipY
)
{
    ASSERT( bTranspose || bFlipX || bFlipY );

    //  Set the matrix for the current bitmap.
    sm_matrix.setup( m_pbBuffer, m_dwWidth, m_dwHeight );

    //  If the bitmap is to be transposed then swap the X/Y.
    if( bTranspose )
    {
        //  Transpose the bitmap.
        sm_matrix.transpose( );

        //  Swap the axis.
        swapXY( );

        //  Set the new dimensions in the matrix.
        sm_matrix.setup( m_pbBuffer, m_dwWidth, m_dwHeight );
    }

    //  If it is to be flipped then do it.
    if( bFlipX || bFlipY )
    {
        sm_matrix.flip( bFlipX, bFlipY );
    }
}





///////////////////////////////////////////////////////////////////////////////
//
//  Function: setMapSize
//
//  Description:
//
//      This function is called to set the size of the character map.  The
//      character map is a convenient way to map characters or tiles onto
//      this bitmap.  For example, if this bitmap is primarily used as a
//      buffer for holding a tile layer of 512 tiles, the map can be used 
//      to hold the X/Y locations of each of the tiles.
//
//  Parameters:
//
//      dwMapSize (input)
//          The number of items in the map.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::setMapSize(
    const DWord dwMapSize
)
{
    ASSERT( dwMapSize > 0 );

    //  If there was already a map allocated then free it up.
    if( m_dwMapSize > 0 )
    {
        delete [] m_pnMapX;
        delete [] m_pnMapY;
    }

    //  Set up the new map.
    m_dwMapSize = dwMapSize;
    m_pnMapX    = new int32[ m_dwMapSize ];
    m_pnMapY    = new int32[ m_dwMapSize ];

    //  We consider all items clipped until the client informs us differently.
    for( DWord dwI = 0 ; dwI < m_dwMapSize ; dwI += 1 )
    {
        m_pnMapX[ dwI ] = MAP_CLIPPED;
        m_pnMapY[ dwI ] = MAP_CLIPPED;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: setMapping
//
//  Description:
//
//      This function is called to set the X/Y position of a character
//      in the map for this bitmap.
//
//  Parameters:
//
//      dwItem (input)
//          The item to set the position for.
//
//      nX (input)
//          The X position of the item.
//
//      nY (input)
//          The Y position of the item.
//
//      pClipping (input)
//          An optional clipping region.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::setMapping(
    const DWord dwItem,
    const int32 nX,
    const int32 nY,
    Clipping*   pClipping /* = NULL */
)
{
    ASSERT( dwItem < m_dwMapSize );

    //  If no clipping area is specified then use the whole bitmap.
    if( !pClipping )
    {
        pClipping = m_pFullClipping;
    }

    //  If the item is clipped then note this, otherwise use the positions.
    if( 
        ( nX < pClipping->m_nMinX ) || 
        ( nX > pClipping->m_nMaxX ) ||
        ( nY < pClipping->m_nMinY ) ||
        ( nY > pClipping->m_nMaxY )
    )
    {
        m_pnMapX[ dwItem ] = MAP_CLIPPED;
        m_pnMapY[ dwItem ] = MAP_CLIPPED;
    }
    else
    {
        m_pnMapX[ dwItem ] = nX;
        m_pnMapY[ dwItem ] = nY;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: swapXY
//
//  Description:
//
//      This function is used to swap the X/Y axis in the bitmap.  This is
//      needed for rotated screen orientations.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::swapXY(
)
{
    //  Temporary values used for swapping items.
    DWord  dwTemp;
    Byte** ppbTemp;

    //  Swap the width and height.
    dwTemp     = m_dwHeight;
    m_dwHeight = m_dwWidth;
    m_dwWidth  = dwTemp;

    //  Swap the rows and columns.
    ppbTemp   = m_ppbRows;
    m_ppbRows = m_ppbCols;
    m_ppbCols = ppbTemp;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clear
//
//  Description:
//
//      This member clears the bitmap.
//
//  Parameters:
//
//      dwColour (input)
//          The colour to clear the bitmap with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::clear(
    const DWord dwColour /* = 0x00 */
)
{
    memset( m_pbBuffer, ( Byte )dwColour, m_dwWidth * m_dwHeight );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearArea
//
//  Description:
//
//      This member clears a specified region in the bitmap.
//
//  Parameters:
//
//      pClipping (input)
//          The area to clear.
//
//      dwColour (input)
//          The colour to clear the bitmap with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Bitmap::clearArea(
    Clipping*   pClipping, 
    const DWord dwColour /* = 0x00 */
)
{
    //  The X/Y coordinate of a pixel that is being cleared.
    int32 nX;
    int32 nY;

    ASSERT( pClipping );

    //  Set each pixel to the specified colour.
    for( nY = pClipping->m_nMinY ; nY <= pClipping->m_nMaxY ; nY += 1 )
    {
        for( nX = pClipping->m_nMinX ; nX <= pClipping->m_nMaxX ; nX += 1 )
        {
            setPixel( nX, nY, dwColour, pClipping );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setPixel
//
//  Description:
//
//      This member is used to set the value of a certain pixel.
//
//  Parameters:
//
//      nDestX (input)
//          The X coordinate of the pixel.
//
//      nDestY (input)
//          The Y coordinate of the pixel.
//
//      dwColour (input)
//          The colour to set the pixel to.
//
//      pClipping (input)
//          The clipping region.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
Bitmap::setPixel( 
    int32     nDestX,
    int32     nDestY,
    DWord     dwColour,
    Clipping* pClipping /* = NULL */
)
{
    //  Temporary variables used for swapping items.
    int32 nSwap;

    //  If no clipping region was specified then use the full clipping region.
    if( !pClipping )
    {
        pClipping = m_pFullClipping;
    }

    //  If the coordinates are out of the clipping region then return.
    if( 
        ( nDestX < pClipping->m_nMinX ) ||
        ( nDestX > pClipping->m_nMaxX ) ||
        ( nDestY < pClipping->m_nMinY ) ||
        ( nDestY > pClipping->m_nMaxY )
    )
    {
        return;
    }


    //  If the canvas is transposed then we need to swap the X/Y locations.
    if( m_pCanvas->getTranspose( ) )
    {
        //  Swap the X/Y locations.
        nSwap  = nDestX;
        nDestX = nDestY;
        nDestY = nSwap;
    }

    //  If the canvas is X flipped then we need to adjust the X coordinate.
    if( m_pCanvas->getFlipX( ) )
    {
        //  First the X coordinate should be calculated from the opposite
        //  side of the bitmap.
        nDestX = m_dwWidth - 1 - nDestX;
    }
        
    //  If the canvas is Y flipped then we need to adjust the Y coordinate.
    if( m_pCanvas->getFlipY( ) )
    {
        //  First the Y coordinate should be calculated from the opposite
        //  side of the bitmap.
        nDestY = m_dwHeight - 1 - nDestY;
    }

    //  Set the pixel.
    m_ppbRows[ nDestY ][ nDestX ] = ( Byte )dwColour;
}    




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getPixel
//
//  Description:
//
//      This member is used to get the value of a certain pixel.
//
//  Parameters:
//
//      nDestX (input)
//          The X coordinate of the pixel.
//
//      nDestY (input)
//          The Y coordinate of the pixel.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Bitmap::getPixel( 
    int32     nDestX,
    int32     nDestY
)
{
    //  Temporary variables used for swapping items.
    int32 nSwap;


    //  If the coordinates are out of range then return.  We use the get()
    //  functions since the arguments are not modified by canvas orientation.
    if( 
        ( nDestX < 0 )                    ||
        ( nDestX > ( int32 )getWidth( ) ) ||
        ( nDestY < 0 )                    ||
        ( nDestY > ( int32 )getHeight( ) )
    )
    {
        return( 0x0000 );
    }

    //  If the canvas is transposed then we need to swap the X/Y related
    //  parameters.
    if( m_pCanvas->getTranspose( ) )
    {
        //  Swap the X/Y locations.
        nSwap  = nDestX;
        nDestX = nDestY;
        nDestY = nSwap;
    }

    //  If the canvas is X flipped then we need to adjust the X coordinate.
    if( m_pCanvas->getFlipX( ) )
    {
        //  First the X coordinate should be calculated from the opposite
        //  side of the bitmap.
        nDestX = m_dwWidth - 1 - nDestX;
    }
        
    //  If the canvas is Y flipped then we need to adjust the Y coordinate.
    if( m_pCanvas->getFlipY( ) )
    {
        //  First the Y coordinate should be calculated from the opposite
        //  side of the bitmap.
        nDestY = m_dwHeight - 1 - nDestY;
    }

    //  return the pixel.
    return( m_ppbRows[ nDestY ][ nDestX ] );
}    




///////////////////////////////////////////////////////////////////////////////
//
//  Function: blit
//
//  Description:
//
//      These is a simple blit method that simply places the specified bitmap
//      at the location specified by the mapping for this bitmap.  This is
//      typically used for characters and tiles that have static locations
//      on a bitmap.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      dwItem (input)
//          The item as referenced in the map.
//
//      pdwColourTable (input)
//          The table used to map the values in the bitmap to the pixel
//          values.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
Bitmap::blit( 
    Bitmap*      pSrc,
    DWord        dwItem, 
    DWord*       pdwColourTable
)
{
    //  The following point to the source and destination pixels.
    Byte*  pbS;
    Byte*  pbD;
    Byte*  pbDEnd;


    //  Find the destination of this item.
    int32 nDestX = m_pnMapX[ dwItem ];
    int32 nDestY = m_pnMapY[ dwItem ];


    //  If the item isn't clipped then draw it.
    if( nDestX != MAP_CLIPPED )
    {
        //  Make sure the source fits into the bitmap.
        ASSERT( nDestX + pSrc->m_dwWidth  <= m_dwWidth  );
        ASSERT( nDestY + pSrc->m_dwHeight <= m_dwHeight );
    
        //  If the canvas is transposed then we need to swap the X/Y locations.
        if( m_pCanvas->getTranspose( ) )
        {
            int32 nSwap = nDestX;
            nDestX      = nDestY;
            nDestY      = nSwap;
        }
    
        //  If the canvas is X flipped then we need to adjust the X coordinate
        //  to the opposite side of the bitmap in the horizontal direction.  
        if( m_pCanvas->getFlipX( ) )
        {
            nDestX = m_dwWidth - pSrc->m_dwWidth - nDestX;
        }
            
        //  If the canvas is Y flipped then we need to adjust the Y coordinate 
        //  to the opposite side of the bitmap in the vertical direction.
        if( m_pCanvas->getFlipY( ) )
        {
            nDestY = m_dwHeight - pSrc->m_dwHeight - nDestY;
        }
    
        //  If there was a colour table specified then we use it, otherwise we
        //  write the values directly.
        if( pdwColourTable )
        {
            for( DWord dwY = 0 ; dwY < pSrc->m_dwHeight ; dwY++ )
            {
                pbD    = m_ppbRows[ nDestY + dwY ] + nDestX;
                pbS    = pSrc->m_ppbRows[ dwY ];
                pbDEnd = pbD + pSrc->m_dwWidth - 7;
                for( ; pbD < pbDEnd ; pbD += 8 )
                {
                    pbD[ 0 ] = ( Byte )pdwColourTable[ pbS[ 0 ] ];
                    pbD[ 1 ] = ( Byte )pdwColourTable[ pbS[ 1 ] ];
                    pbD[ 2 ] = ( Byte )pdwColourTable[ pbS[ 2 ] ];
                    pbD[ 3 ] = ( Byte )pdwColourTable[ pbS[ 3 ] ];
                    pbD[ 4 ] = ( Byte )pdwColourTable[ pbS[ 4 ] ];
                    pbD[ 5 ] = ( Byte )pdwColourTable[ pbS[ 5 ] ];
                    pbD[ 6 ] = ( Byte )pdwColourTable[ pbS[ 6 ] ];
                    pbD[ 7 ] = ( Byte )pdwColourTable[ pbS[ 7 ] ];
                    pbS += 8;
                }
                pbDEnd += 7;
                for( ; pbD < pbDEnd ; pbD++ )
                {
                    *pbD = ( Byte )pdwColourTable[ *( pbS++ ) ];
                }
            }
        }
        else
        {
            for( DWord dwY = 0 ; dwY < pSrc->m_dwHeight ; dwY++ )
            {
                pbD    = m_ppbRows[ nDestY + dwY ] + nDestX;
                pbS    = pSrc->m_ppbRows[ dwY ];
                pbDEnd = pbD + pSrc->m_dwWidth - 7;
                for( ; pbD < pbDEnd ; pbD += 8 )
                {
                    pbD[ 0 ] = pbS[ 0 ];
                    pbD[ 1 ] = pbS[ 1 ];
                    pbD[ 2 ] = pbS[ 2 ];
                    pbD[ 3 ] = pbS[ 3 ];
                    pbD[ 4 ] = pbS[ 4 ];
                    pbD[ 5 ] = pbS[ 5 ];
                    pbD[ 6 ] = pbS[ 6 ];
                    pbD[ 7 ] = pbS[ 7 ];
                    pbS += 8;
                }
                pbDEnd += 7;
                for( ; pbD < pbDEnd ; pbD++ )
                {
                    *pbD = *( pbS++ );
                }
            }
        }
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: blit
//
//  Description:
//
//      These is a more complicated blit method that allows flipping and
//      clipping.  This is typically used for sprites or scrolled layers.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      nOX (input)
//          The X coordinate to blit to.
//
//      nOY (input)
//          The Y coordinate to blit to.
//
//      pdwColourTable (input)
//          The table used to map the values in the bitmap to the pixel
//          values.
//
//      bFlipX (input)
//          Indicates whether or not to flip the image horizontally.
//
//      bFlipY (input)
//          Indicates whether or not to flip the image vertically.
//    
//      pClipping (input)
//          The clipping region.
//
//      eTransparency (input)
//          The type of transparency.
//  
//      bTransparentColour (input)
//          The transparent colour.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
Bitmap::blit( 
    Bitmap*      pSrc,
    int32        nOX,
    int32        nOY,
    DWord*       pdwColourTable,
    Byte         bFlipX,
    Byte         bFlipY,
    Clipping*    pClipping,
    Transparency eTransparency,
    Byte         bTransparentColour /* = 0x00 */
)
{
    //  Check the parameters.
    ASSERT( pClipping != NULL );

    //  Indices used for iterations.
    int32 nY;

    //  The following indicate the start, end and original positions of the
    //  bitmap being blitted.
    int32 nSX;
    int32 nSY;
    int32 nEX;
    int32 nEY;

    //  The following point to the source and destination pixels.
    Byte*  pbS;
    Byte*  pbD;
    Byte*  pbDEnd;

    //  The width and height of the source bitmap.
    DWord dwSrcWidth  = pSrc->m_dwWidth;
    DWord dwSrcHeight = pSrc->m_dwHeight;

    //  An offset used for calculating starting blit positions..
    int32 nOffset;

    //  The current colour.
    Byte  bCol;

    //  Temporary variables used for swapping items.
    int32 nSwap;
    Byte  bSwap;

    //  An adjusted clipping region used if the screen is rotated.
    Clipping adjustedClipping;



    //  If the canvas is transposed then we need to swap the X/Y related
    //  parameters.
    if( m_pCanvas->getTranspose( ) )
    {
        //  Swap the X/Y locations.
        nSwap = nOX;
        nOX   = nOY;
        nOY   = nSwap;

        //  Now the flipping variables.
        bSwap  = bFlipX;
        bFlipX = bFlipY;
        bFlipY = bSwap;

        //  Finally, we need to create a new clipping region to 
        //  reflect the rotated screen.
        adjustedClipping.m_nMinX = pClipping->m_nMinY;
        adjustedClipping.m_nMaxX = pClipping->m_nMaxY;
        adjustedClipping.m_nMinY = pClipping->m_nMinX;
        adjustedClipping.m_nMaxY = pClipping->m_nMaxX;
        pClipping = &adjustedClipping;
    }

    //  If the canvas is X flipped then we need to adjust the X coordinate and
    //  the clipping box to the opposite side of the bitmap in the horizontal
    //  direction.  The flipping variables are not affected since the source
    //  bitmap should be pre-flipped.
    if( m_pCanvas->getFlipX( ) )
    {
        //  First the X coordinate should be calculated from the opposite
        //  side of the bitmap.  The width of the source bitmap should be
        //  compensated for.
        nOX = m_dwWidth - pSrc->m_dwWidth - nOX;
       
        //  Now adjust the clipping region.
        adjustedClipping.m_nMinX = m_dwWidth - 1 - pClipping->m_nMaxX;
        adjustedClipping.m_nMaxX = m_dwWidth - 1 - pClipping->m_nMinX;
        adjustedClipping.m_nMinY = pClipping->m_nMinY;
        adjustedClipping.m_nMaxY = pClipping->m_nMaxY;
        pClipping = &adjustedClipping;
    }
        
    //  If the canvas is Y flipped then we need to adjust the Y coordinate and
    //  the clipping box to the opposite side of the bitmap in the vertical
    //  direction.  The flipping variables are not affected since the source
    //  bitmap should be pre-flipped.
    if( m_pCanvas->getFlipY( ) )
    {
        //  First the Y coordinate should be calculated from the opposite
        //  side of the bitmap.  The height of the source bitmap should be
        //  compensated for.
        nOY = m_dwHeight - pSrc->m_dwHeight - nOY;
       
        //  Now adjust the clipping region.
        adjustedClipping.m_nMinX = pClipping->m_nMinX;
        adjustedClipping.m_nMaxX = pClipping->m_nMaxX;
        adjustedClipping.m_nMinY = m_dwHeight - 1 - pClipping->m_nMaxY;
        adjustedClipping.m_nMaxY = m_dwHeight - 1 - pClipping->m_nMinY;
        pClipping = &adjustedClipping;
    }


    //  Calculate starting and ending coordinates for the blitting.
    nSX = ( nOX < pClipping->m_nMinX ) ? pClipping->m_nMinX : nOX;
    nSY = ( nOY < pClipping->m_nMinY ) ? pClipping->m_nMinY : nOY;
    nEX = nOX + dwSrcWidth - 1;
    if( nEX > pClipping->m_nMaxX )
    {
        nEX = pClipping->m_nMaxX;
    }
    if( nSX > nEX )
    {
        return;
    }
    nEY = nOY + dwSrcHeight - 1;
    if( nEY > pClipping->m_nMaxY )
    {
        nEY = pClipping->m_nMaxY;
    }
    if( nSY > nEY )
    {
        return;
    }

    //  Calculate the starting line and the direction we will blit in the
    //  vertical direction.
    int32 nStart;
    int32 nDirY;
    if( bFlipY )
    {
        nStart = dwSrcHeight - 1 - ( nSY - nOY );
        nDirY  = -1;
    }
    else
    {
        nStart = nSY - nOY;
        nDirY  = 1;
    }

    //  If there was a colour table specified then we use it, otherwise we
    //  write the values directly.
    if( pdwColourTable )
    {
        //  Draw based on the transparency.
        switch( eTransparency )
        {
            //  There is no transparency, obliterate anything underneath the
            //  graphic.
            case TRANSPARENCY_NONE:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd - 7 ; pbD += 8 )
                        {
                            pbS -= 8;
                            pbD[ 0 ] = ( Byte )pdwColourTable[ pbS[ 8 ] ];
                            pbD[ 1 ] = ( Byte )pdwColourTable[ pbS[ 7 ] ];
                            pbD[ 2 ] = ( Byte )pdwColourTable[ pbS[ 6 ] ];
                            pbD[ 3 ] = ( Byte )pdwColourTable[ pbS[ 5 ] ];
                            pbD[ 4 ] = ( Byte )pdwColourTable[ pbS[ 4 ] ];
                            pbD[ 5 ] = ( Byte )pdwColourTable[ pbS[ 3 ] ];
                            pbD[ 6 ] = ( Byte )pdwColourTable[ pbS[ 2 ] ];
                            pbD[ 7 ] = ( Byte )pdwColourTable[ pbS[ 1 ] ];
                        }
                        for( ; pbD <= pbDEnd ; pbD++ )
                        {
                            *pbD = ( Byte )pdwColourTable[ *( pbS-- ) ];
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd - 7 ; pbD += 8 )
                        {
                            pbD[ 0 ] = ( Byte )pdwColourTable[ pbS[ 0 ] ];
                            pbD[ 1 ] = ( Byte )pdwColourTable[ pbS[ 1 ] ];
                            pbD[ 2 ] = ( Byte )pdwColourTable[ pbS[ 2 ] ];
                            pbD[ 3 ] = ( Byte )pdwColourTable[ pbS[ 3 ] ];
                            pbD[ 4 ] = ( Byte )pdwColourTable[ pbS[ 4 ] ];
                            pbD[ 5 ] = ( Byte )pdwColourTable[ pbS[ 5 ] ];
                            pbD[ 6 ] = ( Byte )pdwColourTable[ pbS[ 6 ] ];
                            pbD[ 7 ] = ( Byte )pdwColourTable[ pbS[ 7 ] ];
                            pbS += 8;
                        }
                        for( ; pbD <= pbDEnd ; pbD++ )
                        {
                            *pbD = ( Byte )pdwColourTable[ *( pbS++ ) ];
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            //  Map transparency means that if the colour the pixel is mapped
            //  to through the colour table is equivalent to the transparent
            //  colour than the pixel is not drawn.
            case TRANSPARENCY_MAP:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            bCol = ( Byte )pdwColourTable[ *( pbS-- ) ];
                            if( bCol != bTransparentColour )
                            {
                                *pbD = bCol;
                            }
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            bCol = ( Byte )pdwColourTable[ *( pbS++ ) ];
                            if( bCol != bTransparentColour )
                            {
                                *pbD = bCol;
                            }
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            //  Raw transparency means that if the pixel is the transparent
            //  colour than do not draw the pixel.
            case TRANSPARENCY_RAW:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbS != bTransparentColour )
                            {
                                *pbD = ( Byte )pdwColourTable[ *pbS ];
                            }
                            pbS--;
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbS != bTransparentColour )
                            {
                                *pbD = ( Byte )pdwColourTable[ *pbS ];
                            }
                            pbS++;
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
    
            //  Reverse transparency is the opposite of Raw.  If the destination
            //  pixel is not equal to the transparent pixel then we do not draw
            //  the pixel.  This gives the effect of drawing the graphic 
            //  underneath the existing background.
            case TRANSPARENCY_REVERSE:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbD == bTransparentColour )
                            {
                                *pbD = ( Byte )pdwColourTable[ *pbS ];
                            }
                            pbS--;
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbD == bTransparentColour )
                            {
                                *pbD = ( Byte )pdwColourTable[ *pbS ];
                            }
                            pbS++;
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            default:
            {
                fatalError( "Unsupported transpareny type %d.", eTransparency );
            }
        }
    }
    else
    {
        //  Draw based on the transparency.
        switch( eTransparency )
        {
            //  There is no transparency, obliterate anything underneath the
            //  graphic.
            case TRANSPARENCY_NONE:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd - 7 ; pbD += 8 )
                        {
                            pbS -= 8;
                            pbD[ 0 ] = pbS[ 8 ];
                            pbD[ 1 ] = pbS[ 7 ];
                            pbD[ 2 ] = pbS[ 6 ];
                            pbD[ 3 ] = pbS[ 5 ];
                            pbD[ 4 ] = pbS[ 4 ];
                            pbD[ 5 ] = pbS[ 3 ];
                            pbD[ 6 ] = pbS[ 2 ];
                            pbD[ 7 ] = pbS[ 1 ];
                        }
                        for( ; pbD <= pbDEnd ; pbD++ )
                        {
                            *pbD = *( pbS-- );
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd - 7 ; pbD += 8 )
                        {
                            pbD[ 0 ] = pbS[ 0 ];
                            pbD[ 1 ] = pbS[ 1 ];
                            pbD[ 2 ] = pbS[ 2 ];
                            pbD[ 3 ] = pbS[ 3 ];
                            pbD[ 4 ] = pbS[ 4 ];
                            pbD[ 5 ] = pbS[ 5 ];
                            pbD[ 6 ] = pbS[ 6 ];
                            pbD[ 7 ] = pbS[ 7 ];
                            pbS += 8;
                        }
                        for( ; pbD <= pbDEnd ; pbD++ )
                        {
                            *pbD = *( pbS++ );
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            //  Map transparency becomes equivalent to Raw transparency
            //  when we do not have a colour translation table.
            case TRANSPARENCY_MAP:
            case TRANSPARENCY_RAW:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbS != bTransparentColour )
                            {
                                *pbD = *pbS;
                            }
                            pbS--;
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbS != bTransparentColour )
                            {
                                *pbD = *pbS;
                            }
                            pbS++;
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            //  Reverse transparency is the opposite of Raw.  If the destination
            //  pixel is not equal to the transparent pixel then we do not draw
            //  the pixel.  This gives the effect of drawing the graphic 
            //  underneath the existing background.
            case TRANSPARENCY_REVERSE:
            {
                if( bFlipX )
                {
                    nOffset = dwSrcWidth - 1 - ( nSX - nOX );
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbD == bTransparentColour )
                            {
                                *pbD = *pbS;
                            }
                            pbS--;
                        }
                        nStart += nDirY;
                    }
                }
                else
                {
                    nOffset = nSX - nOX;
                    for( nY = nSY ; nY <= nEY ; nY++ )
                    {
                        pbD    = m_ppbRows[ nY ];
                        pbDEnd = pbD + nEX;
                        pbS    = pSrc->m_ppbRows[ nStart ] + nOffset;
                        for( pbD += nSX ; pbD <= pbDEnd ; pbD++ )
                        {
                            if( *pbD == bTransparentColour )
                            {
                                *pbD = *pbS;
                            }
                            pbS++;
                        }
                        nStart += nDirY;
                    }
                }
                break;
            }
    
            default:
            {
                fatalError( "Unsupported transpareny type %d.", eTransparency );
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: blitScrollFull
//
//  Description:
//
//      This member is used to blit a bitmap onto this bitmap scrolling
//      the source bitmap with wraparound in all directions.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      nDestX (input)
//          The X coordinate to blit to.
//
//      nDestY (input)
//          The Y coordinate to blit to.
//
//      pClipping (input)
//          The clipping region.
//
//      eTransparency (input)
//          The type of transparency.
//  
//      bTransparentColour (input)
//          The transparent colour.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
Bitmap::blitScrollFull( 
    Bitmap*      pSrc,
    int32        nScrollX,
    int32        nScrollY,
    Clipping*    pClipping,
    Transparency eTransparency,
    Byte         bTransparentColour /* = 0x00 */
)
{
    //  Check the parameters.
    ASSERT( pClipping != NULL );

    //  The adjusted scroll values.
    int32 nScrollXAdjusted;
    int32 nScrollYAdjusted;


    //  Adjust the X scroll value.
    if( nScrollX < 0 )
    {
        nScrollXAdjusted = 
            pSrc->getWidth( ) - ( -nScrollX ) % pSrc->getWidth( );
    }
    else
    {
        nScrollXAdjusted = nScrollX % pSrc->getWidth( );
    }
    
    //  Adjust the Y scroll value.
    if( nScrollY < 0 )
    {
        nScrollYAdjusted = 
            pSrc->getHeight( ) - ( -nScrollY ) % pSrc->getHeight( );
    }
    else
    {
        nScrollYAdjusted = nScrollY % pSrc->getHeight( );
    }

    //  Since we can be scrolling in two directions, the clipping can
    //  result in 4 different regions to blit.  For example, if the screen
    //  is 256x256 and we are scrolling to 128,128 then the original bitmap
    //  and scrolled bitmap look like:
    //
    //            +---+---+ 128    +---+---+
    //            | a | b | --->+  | d | c |
    //            +---+---+     |  +---+---+
    //            | c | d | 128 v  | b | a |
    //            +---+---+        +---+---+

    //  (a)
    blit( 
        pSrc,
        nScrollXAdjusted,
        nScrollYAdjusted, 
        NULL,
        FALSE,
        FALSE,
        pClipping,
        eTransparency,
        bTransparentColour
    );
    //  (b)
    blit(
        pSrc,
        nScrollXAdjusted - pSrc->getWidth( ),
        nScrollYAdjusted,
        NULL,
        FALSE,
        FALSE,
        pClipping,
        eTransparency,
        bTransparentColour
    );
    //  (c)
    blit(
        pSrc,
        nScrollXAdjusted,
        nScrollYAdjusted - pSrc->getHeight( ),
        NULL,
        FALSE,
        FALSE,
        pClipping,
        eTransparency,
        bTransparentColour
    );
    //  (d)
    blit(
        pSrc,
        nScrollXAdjusted - pSrc->getWidth( ),
        nScrollYAdjusted - pSrc->getHeight( ),
        NULL,
        FALSE,
        FALSE,
        pClipping,
        eTransparency,
        bTransparentColour
    );
}
