///////////////////////////////////////////////////////////////////////////////
//
//  File:       canvas.h
//
//  Class:      Canvas - Abstract
//  
//  Hierarchy:  Canvas - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The Canvas class represents the screen for the Replay application.  
//      All screen access is done through the Canvas object.  This is an
//      abstract base class for platform specific derived classes that
//      implement the interface for the hardware of the platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _canvas_h_
#define _canvas_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"

//  Forward class declarations.
class Page;
class Bitmap;
class GfxSet;
class Colour;
class Clipping;


///////////////////////////////////////////////////////////////////////////////
//  Canvas Class Definition
///////////////////////////////////////////////////////////////////////////////
class Canvas : public RepBase
{
    public:

        //  The pages available.
        typedef enum
        {
            PAGE_ZERO,
            PAGE_ONE,
      
            PAGE_COUNT
        }
        PageNumber;

        //  The fonts available from the canvas.
        typedef enum
        {
            FONT_SMALL,
            FONT_LARGE
        }
        Font;

        //  The possible orientations.  These are based on a bitmap like
        //  _____TXY where TXY represent Transpose, XFlip, YFlip respectively.
        typedef enum
        {
            NOTRANSPOSE_NOFLIPX_NOFLIPY = 0x00,
            NOTRANSPOSE_NOFLIPX_FLIPY   = 0x01,
            NOTRANSPOSE_FLIPX_NOFLIPY   = 0x02,
            NOTRANSPOSE_FLIPX_FLIPY     = 0x03,
            TRANSPOSE_NOFLIPX_NOFLIPY   = 0x04,
            TRANSPOSE_NOFLIPX_FLIPY     = 0x05,
            TRANSPOSE_FLIPX_NOFLIPY     = 0x06,
            TRANSPOSE_FLIPX_FLIPY       = 0x07
        }
        Orientation;

        //  Deletion.
        virtual ~Canvas ( );

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  General Interface.
        void setOrientation ( 
            const Byte bTranspose, const Byte bFlipX, const Byte bFlipY
        );
        void rotate         ( const DWord dwCount = 1 );
        void flip           ( );
        Byte getOrientation ( ) const;
        Byte getTranspose   ( ) const;
        Byte getFlipX       ( ) const;
        Byte getFlipY       ( ) const;

        //  Page Interface.
        virtual void addPage     ( 
            const KString& pageName,
            PageNumber     ePageNumber, 
            DWord          dwWidth, 
            DWord          dwHeight 
        ) = 0;
        void         removePage  ( PageNumber ePageNumber );
        virtual void turnPage    ( PageNumber ePageNumber );
 
        //  Bitmap Interface.
        virtual Bitmap*   createBitmap         ( 
            const KString& iName,
            DWord          dwWidth,
            DWord          dwHeight,
            Byte           bForScreen = FALSE
        ) = 0;
        void              logon                ( Bitmap* pBitmap );
        void              logoff               ( Bitmap* pBitmap );
        virtual void      draw                 ( 
            Bitmap* pBitmap, int32 nX, int32 nY, DWord dwWidth, DWord dwHeight
        ) = 0;
        void              drawText             (
            const KString& drawText,
            Bitmap*        pDestBitmap,
            const Font     eFont,
            Clipping*      pClipping,
            DWord*         pdwColourTable,
            const Byte     bCenter = FALSE
        );  
        

        //  Font Interface.
        GfxSet& getFont    ( Font eFont ); 

        //  Palette Interface.
        virtual DWord getNumColours    ( ) const = 0;
        void          setColour        ( 
            DWord dwIndex, DWord dwRed, DWord dwGreen, DWord dwBlue
        );
        Byte          getColour        ( 
            DWord* pdwIndex, 
            DWord  dwRed, 
            DWord  dwGreen, 
            DWord  dwBlue, 
            DWord  dwStart = 0
        );
        Byte          getFreeColour    ( DWord* pdwIndex );
        Byte          getClosestColour ( 
            DWord* pdwIndex, 
            DWord  dwRed, 
            DWord  dwGreen, 
            DWord  dwBlue,
            DWord  dwStart = 0
        );
        void          lockColour       ( DWord dwIndex );
        void          unlockColour     ( DWord dwIndex );
        void          captureColour    ( 
            DWord dwIndex, DWord dwRed, DWord dwGreen, DWord dwBlue
        );
        void          freeColours      ( );

        //  General Interface.
        virtual Byte update( ) = 0;

    protected:

        //  Used to indicate orientation (Transpose, Xflip, Yflip).
        enum
        {
            T = 0x04,
            X = 0x02,
            Y = 0x01
        };

        //  Creation is protected.  Clients should use a build method.
        Canvas ( const KString& iName );

        //  This function should be called from the build method.
        virtual void init ( );

        //  Utility.
        void  loadSmallFont   ( );
        void  loadLargeFont   ( );

        //  Helper.
        virtual void  changeColour ( 
            DWord dwIndex, DWord dwRed, DWord dwGreen, DWord dwBlue
        ) = 0;

        //  Member Data.
        Byte                    m_bOrientation;
        Byte                    m_bUpdateNeeded;
        GfxSet*                 m_pFontSmall;
        GfxSet*                 m_pFontLarge;
        Page*                   m_apPageList[ PAGE_COUNT ];
        Page*                   m_pPage;
        KPtrList<Bitmap>        m_bitmapList;
        Colour*                 m_pPaletteColours;
        Byte*                   m_pbCaptured;

        //  Static Member Data.
        static Byte             sm_abOrientTable[ 8 ][ 8 ];
        static Byte             sm_abRotateTable[ 8 ];
        static Byte             sm_bLUTInitialized;
        static uint32           sm_anRedLUT[ 256 ];
        static uint32           sm_anGreenLUT[ 256 ];
        static uint32           sm_anBlueLUT[ 256 ];

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  Canvas Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte
Canvas::getOrientation(
    void
) const
{
    return( m_bOrientation );
}

inline
Byte
Canvas::getTranspose(
    void
) const
{
    return( ( m_bOrientation & Canvas::T ) ? TRUE : FALSE );
}

inline
Byte
Canvas::getFlipX(
    void
) const
{
    return( ( m_bOrientation & Canvas::X ) ? TRUE : FALSE );
}

inline
Byte
Canvas::getFlipY(
    void
) const
{
    return( ( m_bOrientation & Canvas::Y ) ? TRUE : FALSE );
}

inline
void
Canvas::logon(
    Bitmap* pBitmap
)
{
    m_bitmapList.add( pBitmap );
}

inline
void
Canvas::logoff(
    Bitmap* pBitmap
)
{
    m_bitmapList.remove( pBitmap );
}

#endif
