///////////////////////////////////////////////////////////////////////////////
//
//  File:    config.cpp
//
//  Class:   Configuration
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used by the Replay application for Configuration
//      duties.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//  Application Headers.
#include "config.h"
#include "registry.h"
#include "param.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Initialization.
///////////////////////////////////////////////////////////////////////////////
//  A pointer to the singleton instance.
Configuration* Configuration::sm_pInstance = NULL;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: Configuration
//
//  Description:
//
//      This is the main constructor for the configuration object.  It
//      is protected because it is a singleton and therefore cannot be 
//      instantiated by anyone but itself or a derived class.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Configuration::Configuration(
    const KString& iName
)
:
    RepBase             ( iName ),
    m_bInitialized      ( FALSE ),
    m_nArgc             ( 0 ),
    m_ppstrArgv         ( NULL ),
    m_pSettingsRegistry ( NULL ),
    m_parameterList     ( 5 )
{
    //  Assign the instance pointer.
    ASSERT( sm_pInstance == NULL );
    sm_pInstance = this;

    //  Add the parameters that are common for all Replay+ versions.
    addParam( "-game",    "<gameid>", "Run the specified game." );
    addParam( "-nojoy",   " ",        "Disable joystick usage." );
    addParam( "-nonet",   " ",        "Disable network support." );
    addParam( "-nosound", " ",        "Disable sound." );
    addParam( "-noopl",   " ",        "Disable OPL chip emulation." );
    addParam( "-srate",   "<rate>",   "Sample Rate (5000-65535 [44100])" );
    addParam( "-sbits",   "<bits>",   "Sample Bits (8,16 [8])" );
    addParam( "+90",      " ",        "Rotate 90 degrees clockwise." );
    addParam( "-90",      " ",        "Rotate 90 degrees counterclockwise." );
    addParam( "+180",     " ",        "Rotate 180 degrees." );
    addParam( "-180",     " ",        "Rotate 180 degrees." );
    addParam( "-flipx",   " ",        "Flip horizontally." );
    addParam( "-flipy",   " ",        "Flip vertically." );
    addParam( "-path",    "<paths>",  "Colon-separated list of search dirs." );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Configuration
//
//  Description:
//
//      This is the destructor for the configuration object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Configuration::~Configuration(
)
{
    //  Free up the parameter list.
    for( DWord dwI = 0 ; dwI < m_parameterList.entries( ) ; dwI += 1 )
    {
        delete m_parameterList[ dwI ];
    }
    m_parameterList.clear( );

    //  Delete the registries.
    delete m_pSettingsRegistry;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_instance
//
//  Description:
//
//      This member returns the configuration object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A reference to the singleton Configuration object.
//
///////////////////////////////////////////////////////////////////////////////
Configuration&
Configuration::s_instance(
)
{
    //  Return the instance.
    ASSERT( sm_pInstance != NULL );
    return( *sm_pInstance );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initialize
//
//  Description:
//
//      This member is called to initialize the Configuration object.
//
//  Parameters:
//
//      nArgc (input)
//          The number of command line parameters available for config.
//
//      ppstrArgv (input)
//          The list of command line parameters.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Configuration::initialize(
    int32  nArgc,
    char** ppstrArgv
) 
{
    //  This should only be initialized once.
    CONFIRM( m_bInitialized == FALSE, "Configuration already initialized" );
    m_bInitialized = TRUE;

    //  Process the command line.
    processCommandLine( nArgc, ppstrArgv );

    //  Create the registries.
    m_pSettingsRegistry = new Registry( "replay", Registry::CONFIG );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: processCommandLine
//
//  Description:
//
//      This member is called to process the command line arguments.
//
//  Parameters:
//
//      nArgc (input)
//          The number of command line parameters available for config.
//
//      ppstrArgv (input)
//          The list of command line parameters.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Configuration::processCommandLine(
    int32  nArgc,
    char** ppstrArgv
) 
{
    //  Make sure the parameters are valid.
    CONFIRM( nArgc > 0, "Configuration: Bad values passed in." );
    CONFIRM( ppstrArgv != NULL, "Configuration: Bad values passed in." );

    //  Hold on to the command line parameters.
    m_nArgc     = nArgc;
    m_ppstrArgv = ppstrArgv;

    //  Search the command line parameter list and ensure that each of them
    //  are valid command line parameters.  If so, then note that the 
    //  parameter has been found.
    for( int32 nI = 1 ; nI < m_nArgc ; )
    {
        Parameter* pParameter = findParam( m_ppstrArgv[ nI ] );
        if( pParameter == NULL )
        {
            usage( );
            exit( 1 );
        }

        //  Increment the index and check if the following item is an
        //  argument to the parameter (i.e. not another parameter).  If
        //  it is then set it in the parameter, otherwise just note that
        //  the parameter has been found.
        nI += 1;
        if( ( nI < m_nArgc ) && ( !isParam( m_ppstrArgv[ nI ] ) ) )
        {
            pParameter->found( m_ppstrArgv[ nI ] );
            nI += 1;
        }
        else
        {
            pParameter->found( KStringNULL );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParam
//
//  Description:
//
//      This member is called to return a parameter that can be either
//      TRUE (parameter found) or FALSE (parameter not found).
//
//  Parameters:
//
//      paramName (input)
//          The name of the parameter to search for.
//
//      bResult (output)
//          The buffer to store the result in.
//
//  Returns:
//
//      TRUE if the parameter was found, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
const 
Byte  
Configuration::getParam( 
    const KString& paramName, 
    Byte*          bResult   /* = NULL */
)
{
    //  Find the parameter.
    Parameter* pParameter = findParam( paramName.data( ) );
    ASSERT( pParameter != NULL );
    
    //  Assign the result if desired.
    if( bResult != NULL )
    {
        *bResult = pParameter->isFound( );
    }
  
    return( pParameter->isFound( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParam
//
//  Description:
//
//      This member is called to return a parameter that is numeric.
//
//  Parameters:
//
//      paramName (input)
//          The name of the parameter to search for.
//
//      nResult (output)
//          The value of the parameter.
//
//  Returns:
//
//      TRUE if the parameter was found, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
const 
Byte  
Configuration::getParam( 
    const KString& paramName, 
    int32*         nResult 
)
{
    //  Find the parameter.
    Parameter* pParameter = findParam( paramName.data( ) );
    ASSERT( pParameter != NULL );

    //  If the parameter was found then assign the result.
    if( ( nResult != NULL ) && ( pParameter->isFound( ) ) )
    {
        //  Read it based on whether it's in hexadecimal notation.
        if( 
            ( strchr( pParameter->getArgument( ).data( ), 'x' ) != NULL ) ||
            ( strchr( pParameter->getArgument( ).data( ), 'X' ) != NULL )
        )
        {
            sscanf( pParameter->getArgument( ).data( ), "%lx", nResult );
        }
        else
        {
            *nResult = atoi( pParameter->getArgument( ).data( ) );
        }
    }

    return( pParameter->isFound( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParam
//
//  Description:
//
//      This member is called to return a parameter that is a string.
//
//  Parameters:
//
//      paramName (input)
//          The name of the parameter to search for.
//
//      result (output)
//          The value of the parameter.
//
//  Returns:
//
//      TRUE if the parameter was found, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
const 
Byte  
Configuration::getParam( 
    const KString& paramName, 
    KString*       pResult 
)
{
    ASSERT( pResult != NULL );

    //  Find the parameter.
    Parameter* pParameter = findParam( paramName.data( ) );
    ASSERT( pParameter != NULL );

    //  If the parameter was found then assign the result.
    if( ( pResult != NULL ) && ( pParameter->isFound( ) ) )
    {
        *pResult = pParameter->getArgument( );
    }

    return( pParameter->isFound( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addParam
//
//  Description:
//
//      This member is called to add a parameter description to the list of
//      valid parameters.
//
//  Parameters:
//
//      pstrParameter (input)
//          The name of the parameter.
//
//      pstrArgumentDesc (input)
//          A description of the argument to the parameter.
//
//      pstrParameterDesc (input)
//          The description of the parameter.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
Configuration::addParam ( 
    const char* pstrParameter,
    const char* pstrArgumentDesc,
    const char* pstrParameterDesc
)
{
    //  Add the parameter to the list.
    m_parameterList.add( 
        new Parameter( pstrParameter, pstrArgumentDesc, pstrParameterDesc ) 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: findParam
//
//  Description:
//
//      This member is called to find a parameter object in the list of
//      valid parameters.
//
//  Parameters:
//
//      pstrParameter (input)
//          The name of the parameter.
//
//  Returns:
//
//      A pointer to the parameter or NULL if the parameter does not exist.
//
///////////////////////////////////////////////////////////////////////////////
Parameter* 
Configuration::findParam ( 
    const char* pstrParameter
) const
{
    //  Attempt to match the specified parameter to one in the list.
    for( DWord dwI = 0 ; dwI < m_parameterList.entries( ) ; dwI += 1 )
    {
        if( 
            strcmp( 
                m_parameterList[ dwI ]->getParameter( ).data( ), pstrParameter
            ) == 0
        )
        {
            return( m_parameterList[ dwI ] );
        }
    }

    //  If the loop completes then the parameter was not found.
    return( NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: usage
//
//  Description:
//
//      This member is called to print the usage of the application and
//      then exit.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Configuration::usage ( 
)
{
    printf( "\nUsage: %s <options>\n\n", getExecutable( ) );
    for( DWord dwI = 0 ; dwI < m_parameterList.entries( ) ; dwI += 1 )
    {
        Parameter* pParameter = m_parameterList[ dwI ];
        printf(  
            "  %-10s  %-10s  %s\n", 
            pParameter->getParameter( ).data( ),
            pParameter->getArgumentDesc( ).data( ),
            pParameter->getParameterDesc( ).data( )
        );
    }
    printf( "\n" );
}
