///////////////////////////////////////////////////////////////////////////////
//
//  File:    disk.cpp
//
//  Class:   Disk
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Disk class is an abstract base class encapsulating disk
//      access.  A derived class for each platform should be created
//      from this base.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <string.h> 
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>

//  Application Headers.
#include "disk.h"
#include "config.h"
#include "regular.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Disk
//
//  Description:
//
//      This is the main constructor for a disk object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Disk::Disk(
    const KString&  iName
)
:
    RepBase           ( iName ),
    m_pathList        ( 2 )
{
    //  First change to the directory where Replay+ was run from.
    setWorkingDirectory( );

    //  Now we need to keep track of the paths.
    getPaths( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Disk
//
//  Description:
//
//      This is the destructor for a disk object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Disk::~Disk(
)
{
    //  Free the path list.
    for( DWord dwI = 0 ; dwI < m_pathList.entries( ) ; dwI += 1 )
    {
        delete m_pathList[ dwI ];
    }
    m_pathList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: gameFileExists
//
//  Description:
//
//      This member is used to determine whether or not the specified game
//      file exists in the possible locations for game files.
//
//  Parameters:
//
//      gameId (input)
//          The ID of the game the file is needed for.
//
//      fileName (input)
//          The name of the file to look for.
//
//  Returns:
//
//      TRUE  if the file was found.
//      FALSE if the file was not found.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
Disk::gameFileExists(
    const KString& gameId,
    const KString& fileName
) const
{
    //  Check for the file.
    return( 
        gameFileLocation( gameId, fileName ) == LOC_NOTFOUND ? FALSE : TRUE 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: gameFileOpen
//
//  Description:
//
//      This member is used to open the specified binary file and return
//      the handle to it.
//
//  Parameters:
//
//      gameId (input)
//          The ID of the game that the file is to be opened for.
//
//      fileName (input)
//          The name of the file to open.
//
//  Returns:
//
//      A pointer to the GameFile object.
//      NULL if the file is not found.
//
///////////////////////////////////////////////////////////////////////////////
GameFile*
Disk::gameFileOpen(
    const KString&     gameId, 
    const KString&     fileName
) const
{
    //  The game file to open.
    GameFile* pGameFile( NULL );

    //  Open a file of the appropriate type.
    switch( gameFileLocation( gameId, fileName ) )
    {
        case LOC_NOTFOUND:
        {
            break;
        }
        case LOC_REGULAR:
        {
            pGameFile = new RegularFile( gameId, fileName );
            break;
        }
        default:
        {
            CONFIRM( FALSE, "Unknown file location to open." );
        }
    }
 
    //  Return the object pointer.
    return( pGameFile );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: gameFileClose
//
//  Description:
//
//      This member is used to close the specified game file and delete it.
//      It assumes the file was opened with gameFileOpen.
//
//  Parameters:
//
//      pGameFile (input)
//          A pointer to the game file object.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Disk::gameFileClose(
    GameFile* pGameFile
) const
{
    ASSERT( pGameFile != NULL );
    delete pGameFile;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: gameFileLoad
//
//  Description:
//
//      This member is used to load the specified binary file into
//      the given buffer.
//
//  Parameters:
//
//      gameId (input)
//          The ID of the game that the file is to be loaded for.
//
//      fileName (input)
//          The name of the file to load.
//
//      pbDest (output)
//          The buffer to contain the file.
//
//      dwMaxLength (input)
//          The maximum number of bytes to copy from the file.
//
//      dwOffset (input)
//          The offset within the specified file that the copy should start
//          at.
//
//      bSkip (input)
//          Indicates whether to perform a skip read (i.e. read into every
//          other byte).
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
Disk::gameFileLoad(
    const KString&     gameId, 
    const KString&     fileName, 
    Byte*              pbDest, 
    const DWord        dwMaxLength, 
    const DWord        dwOffset     /* = 0x0000 */,
    const Byte         bSkip        /* = FALSE */ 
)
{
    //  The game file to open.
    GameFile* pGameFile = gameFileOpen( gameId, fileName );

    //  If the file wasn't opened then return.
    if( pGameFile == NULL )
    {
        return( FALSE );
    }
 
    //  Load the file.
    if( bSkip )
    {
        pGameFile->readSkip( pbDest, dwMaxLength, dwOffset );
    }
    else
    {
        pGameFile->read( pbDest, dwMaxLength, dwOffset );
    }

    //  Done with the file.
    delete pGameFile;
    
    //  The file was read successfully. 
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkForFile
//
//  Description:
//
//      This member attempts to find the specified disk file regardless of
//      the case of the file.
//
//  Parameters:
//
//      path (input/output)
//          The directory to look in.
//
//      fileName (input)
//          The name of the file.
//
//  Returns:
//
//      The size of the file if found.
//      0 otherwise.
//
///////////////////////////////////////////////////////////////////////////////
const 
DWord 
Disk::checkForFile( 
    KString&       path, 
    const KString& fileName 
)
{
    //  The stat buffer used when determining if a file exists.
    struct stat statBuf;

    //  The fill path name.
    KString fullPath;


    //  Build the full path.
    fullPath  = path;
    fullPath += getDirectorySeparator( );
    fullPath += fileName;

    //  Does the file exist.
    if( stat( fullPath.data( ), &statBuf ) == 0 )
    {
        path = fullPath;
        return( statBuf.st_size );
    }
    else
    {
        //  Check for a file that only differs by case.
        DIR*           pDir;
        struct dirent* pDirEnt;

        //  Open the directory.
        pDir = opendir( path.data( ) );
        if( pDir )
        {
            //  Run through each of the entries.
            for( 
                pDirEnt = readdir( pDir ) ; 
                pDirEnt != NULL ; 
                pDirEnt = readdir( pDir )
            )
            {
                KString dName( pDirEnt->d_name );
                fullPath  = path;
                fullPath += getDirectorySeparator( );
                fullPath += dName;
                if( strcasecmp( pDirEnt->d_name, fileName.data( ) ) == 0 )
                {
                    if( stat( fullPath.data( ), &statBuf ) == 0 )
                    {
                        closedir( pDir );
                        path = fullPath;
                        return( statBuf.st_size );
                    }
                }
            }
            closedir( pDir );
        }
    }

    return( 0 );
}
                    
                    
    
///////////////////////////////////////////////////////////////////////////////
//
//  Function: gameFileLocation
//
//  Description:
//
//      This member attempts to find the specified game file and returns the
//      location if found.
//
//  Parameters:
//
//      gameId (input)
//          The ID of the game the file belongs to.
//
//      fileName (input)
//          The name of the file of interest.
//
//  Returns:
//
//      The location of the file.
//
///////////////////////////////////////////////////////////////////////////////
Disk::GameFileLocation 
Disk::gameFileLocation( 
    const KString& gameId, 
    const KString& fileName 
) const
{
    //  OK, try the regular file out.
    if( RegularFile::s_fileExists( gameId, fileName ) )
    {
        return( LOC_REGULAR );
    }

    //  The file was not found in any of the expected locations therefore
    //  indicate this. 
    return( LOC_NOTFOUND );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: makeDirectoryComponents
//
//  Description:
//
//      This member is used to construct the components of the directory
//      that leads to the specified file.
//
//  Parameters:
//
//      fileName (input)
//          The name of the file that we need to create directory components
//          for.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Disk::makeDirectoryComponents(
    const KString& fileName
)
{
    //  The stat buffer used when determining if a directory exists.
    struct stat statBuf;

    //  Make a copy of the file passed in.
    char* pstrFileName = strdup( fileName.data( ) );

    //  What separator are we using?
    char separator = *( getDirectorySeparator( ).data( ) );

    //  A pointer used to traverse the file name.
    char* pTraverse;


    //  Loop until we run out of components to create.
    for( 
        pTraverse = strchr( pstrFileName, separator ) ; 
        pTraverse != NULL ; 
        pTraverse = strchr( pTraverse + 1, separator )
    )
    {
        //  Ignore the root directory.
        if( pTraverse == pstrFileName )
        {
            continue;
        }

        //  First off, replace the trailing separator with the EOS.
        *pTraverse = '\0';

        //  If the directory so far already exists then move on.
        if( stat( pstrFileName, &statBuf ) == 0 )
        {
            //  Now put the separator back.
            *pTraverse = separator;
            continue;
        }

        //  Now create the directory up until the current component.
        int nCheck = mkdir( pstrFileName, 0777 );
        CONFIRM( nCheck != -1, "Could not create directory component" );
      
        //  Now put the separator back.
        *pTraverse = separator;
    }

    //  Dispose of the copy.
    free( pstrFileName );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getDirectorySeparator
//
//  Description:
//
//      This member is used to return the directory separator character
//      as a string.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A pointer to the directory separator character.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Disk::getDirectorySeparator(
) const
{
    //  The directory separator is by default a forward slash.  Brain-dead
    //  OS's can override to be something different.
    static KString separator( "/" );

    //  Send it back.
    return( separator );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setWorkingDirectory
//
//  Description:
//
//      This member is used to set the working directory of the Replay 
//      application.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Disk::setWorkingDirectory(
)
{
    //  The directory where the replay executable is.
    char strReplayDir[256];


    //  Make sure we're in the correct directory.
    strcpy( strReplayDir, Configuration::s_instance( ).getExecutable( ) );
    if( strchr ( strReplayDir, '/' ) != ( char* )NULL )
    {
        *( strrchr ( strReplayDir, '/' ) ) = '\0';
        chdir( strReplayDir );
    }
    else
    if( strchr ( strReplayDir, '\\' ) != ( char* )NULL )
    {
        *( strrchr ( strReplayDir, '\\' ) ) = '\0';
        chdir( strReplayDir );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getPaths
//
//  Description:
//
//      This member is used to retrieve the paths where Replay+ is to look
//      for files.  The first path specified is the default root directory
//      where any newly created files are stored under.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Disk::getPaths(
)
{
    //  If a path or paths were specified then add them to the list of paths, 
    if( Configuration::s_instance( ).getParam( "-path" ) )
    {
        //  Get the list of paths.
        KString paths;
        Configuration::s_instance( ).getParam( "-path", &paths );

        //  Make a copy of the string so that we can chop it up.
        char* pCopy = strdup( paths.data( ) );

        //  Make sure all of the separators are correct.
        char* pTraverse;
        for( pTraverse = pCopy ; *pTraverse != '\0' ; pTraverse += 1 )
        {
            if( ( *pTraverse == '/' ) || ( *pTraverse == '\\' ) )
            {
                *pTraverse = getDirectorySeparator( )[ 0 ];
            }
        }

        //  Parse the paths.
        for(
            pTraverse = strtok( pCopy, ":" ) ;
            pTraverse != NULL ;
            pTraverse = strtok( NULL, ":" )
        )
        {
            m_pathList.add( new KString( pTraverse ) );
        }
    }

    //  Always add the current directory since some files (e.g. fonts)
    //  need to be loaded from there.  This means that files can be
    //  be placed in the Replay+ directory as a last resort.  However, I
    //  wouldn't recommend this.
    m_pathList.add( new KString( "." ) );
}
