///////////////////////////////////////////////////////////////////////////////
//
//  File:       network.h
//
//  Class:      Network - Abstract
//  
//  Hierarchy:  Network - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The Network class is an abstract base class encapsulating a network
//      connection between 2 or more Replay processes.  A derived class 
//      should be created for each network protocol.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _network_h_
#define _network_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repbase.h"
#include "kstring.h"


///////////////////////////////////////////////////////////////////////////////
//  Network Class Definition
///////////////////////////////////////////////////////////////////////////////
class Network : public RepBase
{
    public:

        typedef enum
        {
            UNKNOWN,
            CLIENT,
            SERVER
        }
        ConnectType;

        //  The possible messages.  We arbitrarily assign them ASCII values.
        typedef enum
        {
            MSG_NONE,
            MSG_SERVER,
            MSG_CLIENT,
            MSG_CLOSE,
            MSG_OK,
            MSG_GAME,
            MSG_DIP,
            MSG_PLAYER,
            MSG_CTRL
        }
        MsgType;

        //  The possible message structures.
        typedef struct
        {
            MsgType eType;
            Byte    bFiller[ 9 ];
        }
        Msg;

        typedef struct Msg MsgNone;
        typedef struct Msg MsgServer;
        typedef struct Msg MsgClient;
        typedef struct Msg MsgClose;

        typedef struct 
        {
            MsgType eType;
            char    strGameName[ 9 ];
        }
        MsgGame;

        typedef struct
        {
            MsgType eType;
            Byte    bDipIndex;
            Byte    bValue;
        }
        MsgDip;

        typedef struct
        {
            MsgType eType;
            Byte    bPlayerNum;
        }
        MsgPlayer; 

        typedef struct
        {
            MsgType eType;
            DWord   dwCtrlMask;
        }
        MsgCtrl;
        
        //  Deletion.
                 Network ( const KString& iName );
        virtual ~Network ( );

        //  Connection.
        virtual Byte openNetwork     ( ConnectType eType );
        virtual void closeNetwork    ( );
        virtual Byte acceptClient    ( ) = 0;
        virtual Byte connectToServer ( ) = 0;
        virtual void closeConnection ( DWord dwConnection ) = 0;

        //  Access.
        Byte         isEnabled         ( ) const;
        Byte         isOpen            ( ) const;
        ConnectType  getConnectType    ( ) const;
        DWord        getNumConnections ( ) const;
        DWord        getMaxConnections ( ) const;

        //  Message passing.
        virtual void readMsg  ( const DWord dwConnection, Msg* pMsg ) = 0;
        virtual void writeMsg ( const DWord dwConnection, Msg* pMsg ) = 0;
        virtual void writeMsg ( Msg* pMsg );

    protected:

        //  Member Data.
        Byte         m_bEnabled;
        ConnectType  m_eConnectType;
        DWord        m_dwNumConnections;

        //  Static Member Data.
        static DWord sm_dwMaxConnections;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  Network Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte
Network::isEnabled(
) const
{
    return( m_bEnabled );
}

inline
Byte
Network::isOpen(
) const
{
    return( !( m_eConnectType == Network::UNKNOWN ) );
}

inline
Network::ConnectType
Network::getConnectType(
) const
{
    return( m_eConnectType );
}

inline
DWord
Network::getNumConnections(
) const
{
    return( m_dwNumConnections );
}

inline
DWord
Network::getMaxConnections(
) const
{
    return( Network::sm_dwMaxConnections );
}

inline
void
Network::writeMsg(
    Network::Msg* pMsg
)
{
    for( DWord dwI = 0 ; dwI < m_dwNumConnections ; dwI += 1 )
    {
        writeMsg( dwI, pMsg );
    }
}


#endif
