///////////////////////////////////////////////////////////////////////////////
//
//  File:    registry.cpp
//
//  Class:   Registry
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used by the Replay application for keeping a "registry"
//      of values based on group/name pairs.
//
//      Registries are kept in a regular disk file and are structured as 
//      follows:
//
//          AAAABCC..CDEE..EFFFFGG..GBCC..CDEE..EFFFFGG..G etc.
//
//      where:
//
//          AAAA  - The number of entries in the registry.
//          B     - The length of the group name of this entry.
//          CC..C - The group name.
//          D     - The length of the name of this entry.
//          EE..E - The name.
//          FFFF  - The length of the data for this entry.
//          GG..G - The data.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <string.h>

//  Application Headers.
#include "registry.h"
#include "replay.h"
#include "appfile.h"
#include "disk.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Registry
//
//  Description:
//
//      This is the main constructor for the registry object.  
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      eContents (input)
//          What is to be stored in this registry.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Registry::Registry(
    const KString& iName,
    const Contents eContents
)
:
    RepBase        ( iName ),
    m_eContents    ( eContents ),
    m_fileName     ( ),
    m_pAppFile     ( NULL ),
    m_dwNumEntries ( 0 ),
    m_pEntryHead   ( NULL ),
    m_pEntryTail   ( NULL )
{
    //  Initialize the registry.
    initialize( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Registry
//
//  Description:
//
//      This is the destructor for the registry object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Registry::~Registry(
)
{
    //  Free up the list of entries.
    while( m_pEntryHead != NULL )
    {
        deleteEntry( m_pEntryHead );
    }

    //  Done with the application file.
    delete m_pAppFile;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the configuration object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Registry::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Registry" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initialize
//
//  Description:
//
//      This member is called to initialize the registry.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::initialize(
) 
{
    //  The disk object.
    Disk* pDisk = Replay::s_instance( ).getDisk( );


    //  First we need to build the registry file name.
    m_fileName  = pDisk->getPath( );
    m_fileName += pDisk->getDirectorySeparator( );
    switch( m_eContents )
    {
        case CONFIG:  m_fileName += "config";                      break;
        case SCORES:  m_fileName += "scores";                      break;
        default:      fatalError( "Unknown Registry Contents." );  break;
    }
    m_fileName += pDisk->getDirectorySeparator( );
    m_fileName += getInstanceName( );
    switch( m_eContents )
    {
        case CONFIG:  m_fileName += ".cfg";                        break;
        case SCORES:  m_fileName += ".scr";                        break;
        default:      fatalError( "Unknown Registry Contents." );  break;
    }
    
    //  Make sure the directory exists to store the registry in.
    pDisk->makeDirectoryComponents( m_fileName );

    //  Open the registry file.
    m_pAppFile = new AppFile( m_fileName );
 
    //  Make sure the file was opened.
    CONFIRM( m_pAppFile->isOpen( ), "Could not open %s.", m_fileName.data( ) );

    //  Load the registry entries.
    loadEntries( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadEntries
//
//  Description:
//
//      This member is called to load the registry entries from the registry
//      file into our local list.
//
//  Parameters:
//
//      pRegistry (input)
//          The opened registry file.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::loadEntries(
) 
{
    //  Check to make sure the current entry list is empty.
    ASSERT( m_dwNumEntries == 0    );
    ASSERT( m_pEntryHead   == NULL );
    ASSERT( m_pEntryTail   == NULL );

    //  A pointer to the current entry.
    RegistryEntry* pCurEntry( NULL );

    //  The length of a name.
    Byte  bLength;

    //  The number of entries.
    DWord dwNumEntries = 0;


    //  Read in the number of entries.
    m_pAppFile->seek( 0 );
    m_pAppFile->read( ( Byte* )&dwNumEntries, sizeof( dwNumEntries ) );

    //  If the number of entries could not be read then the file must have
    //  just been created so initialize it as an empty registry.
    if( m_pAppFile->count( ) != sizeof( dwNumEntries ) )
    {
        createEmpty( );
    }
 
    //  Now read each of the entries.
    for( DWord dwI = 0 ; dwI < dwNumEntries ; dwI += 1 )
    {
        //  Allocate a new item on the list.
        pCurEntry = new RegistryEntry( );

        //  Read in the length of the group.
        m_pAppFile->read( &bLength, sizeof( bLength ) );
        ASSERT( m_pAppFile->count( ) == sizeof( bLength ) );
        pCurEntry->pstrGroup = new char[ bLength + 1 ];
        
        //  Read in the group.
        m_pAppFile->read( pCurEntry->pstrGroup, bLength );
        ASSERT( m_pAppFile->count( ) == bLength );
        pCurEntry->pstrGroup[ bLength ] = '\0';

        //  Read in the length of the name.
        m_pAppFile->read( &bLength, sizeof( bLength ) );
        ASSERT( m_pAppFile->count( ) == sizeof( bLength ) );
        pCurEntry->pstrName = new char[ bLength + 1 ];
        
        //  Read in the name.
        m_pAppFile->read( pCurEntry->pstrName, bLength );
        ASSERT( m_pAppFile->count( ) == bLength );
        pCurEntry->pstrName[ bLength ] = '\0';

        //  Read in the size.
        m_pAppFile->read( 
            ( Byte* )&( pCurEntry->dwSize ), sizeof( pCurEntry->dwSize )
        );
        ASSERT( m_pAppFile->count( ) == sizeof( pCurEntry->dwSize ) );

        //  The offset is the current location of the file.
        pCurEntry->dwOffset = m_pAppFile->tell( );

        //  Seek past to the next entry.
        m_pAppFile->seek( pCurEntry->dwSize, AppFile::CUR );

        //  Terminate the list.
        pCurEntry->pNext = NULL;

        //  Add it to the list.
        newEntry( pCurEntry );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createEmpty
//
//  Description:
//
//      This member is called to create an empty registry file on disk.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::createEmpty(
) 
{
    //  Write the number of entries (should be 0 at this time).
    m_dwNumEntries = 0;
    m_pAppFile->write( ( Byte* )&m_dwNumEntries, sizeof( m_dwNumEntries ) );
    ASSERT( m_pAppFile->count( ) == sizeof( m_dwNumEntries ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteValue
//
//  Description:
//
//      This member is called to delete a value from the registry.
//
//  Parameters:
//
//      group (input)
//          The value group.
//
//      name (input)
//          The value name.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::deleteValue( 
    const KString& group,
    const KString& name
)
{
    //  Search for the value.
    RegistryEntry* pCurEntry = findEntry( group, name );

    //  If the value was found then delete it.
    if( pCurEntry != NULL )
    {
        removeEntry( pCurEntry );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getValueSize
//
//  Description:
//
//      This member is called to return the length of the specified registry
//      value.
//
//  Parameters:
//
//      group (input)
//          The value group.
//
//      name (input)
//          The value name.
//
//  Returns:
//
//      The size of the value if found.
//      0 if the value is not found.
//
///////////////////////////////////////////////////////////////////////////////
const 
DWord 
Registry::getValueSize( 
    const KString& group,
    const KString& name
)
{
    //  Search for the value.
    RegistryEntry* pCurEntry = findEntry( group, name );

    //  Return the value's size if found, otherwise 0.
    return( pCurEntry != NULL ? pCurEntry->dwSize : 0 );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getValue
//
//  Description:
//
//      This member is called to return the specified registry value as a 
//      stream of bytes up to the length specified.
//
//  Parameters:
//
//      group (input)
//          The value group.
//
//      name (input)
//          The value name.
//
//      pbResult (output)
//          The buffer to place the value in.
//
//      dwMaxLength (input)
//          The maximum number of bytes that can be placed in the result.
//
//  Returns:
//
//      The number of items (bytes) placed in the result buffer.
//
///////////////////////////////////////////////////////////////////////////////
const 
DWord  
Registry::getValue( 
    const KString& group,
    const KString& name,
    Byte*          pbResult, 
    DWord          dwMaxLength
)
{
    //  Check the parameters.
    ASSERT( pbResult != NULL );
    ASSERT( dwMaxLength > 0  );

    //  Search for the value.  If not found then return.
    RegistryEntry* pCurEntry = findEntry( group, name );
    if( pCurEntry == NULL )
    {
        return( 0 );
    }

    //  Seek to the specified location.
    m_pAppFile->seek( pCurEntry->dwOffset );
    CONFIRM( 
        m_pAppFile->tell( ) == pCurEntry->dwOffset, 
        "%s corrupt.", 
        m_fileName.data( )
    );

    //  Load the value.
    m_pAppFile->read( pbResult, dwMaxLength );

    //  Return the number of bytes read.
    return( m_pAppFile->count( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setValue
//
//  Description:
//
//      This member is called to set the specified value to the registry.
//
//  Parameters:
//
//      group (input)
//          The value group.
//
//      name (input)
//          The value name.
//
//      pbValue (output)
//          The buffer containing the value.
//
//      dwLength (input)
//          The of bytes to write.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::setValue( 
    const KString& group,
    const KString& name,
    Byte*          pbValue, 
    DWord          dwLength
)
{
    //  Check the parameters.
    ASSERT( pbValue != NULL );
    ASSERT( dwLength > 0 );

    //  Does the value exist?
    RegistryEntry* pCurEntry = findEntry( group, name );

    //  If the value exists but is not the correct size then delete it.
    if( ( pCurEntry != NULL ) && ( pCurEntry->dwSize != dwLength ) )
    {
        //  Get rid of the value. 
        removeEntry( pCurEntry );

        //  Note that it's gone.
        pCurEntry = NULL;
    }

    //  If there is no entry then we need to create a new one.
    if( pCurEntry == NULL )
    {
        //  The new entry.
        pCurEntry = new RegistryEntry( );

        //  Allocate some space for the new entry.
        pCurEntry->pstrGroup = new char[ group.length( ) + 1 ];
        pCurEntry->pstrName  = new char[ name.length( )  + 1 ];

        //  Fill out the new entry.
        strcpy( pCurEntry->pstrGroup, group.data( ) );
        strcpy( pCurEntry->pstrName, name.data( ) );
        pCurEntry->dwOffset = 0;
        pCurEntry->dwSize   = dwLength;
        pCurEntry->pNext    = NULL;

        //  Add the new entry.
        addEntry( pCurEntry );
    } 

    //  Seek to the position to write.
    m_pAppFile->seek( pCurEntry->dwOffset );
    CONFIRM( 
        m_pAppFile->tell( ) == pCurEntry->dwOffset, 
        "Could not seek to write value in %s.",
        m_fileName.data( )
    );

    //  Write the value.
    m_pAppFile->write( pbValue, dwLength );
    ASSERT( m_pAppFile->count( ) == dwLength );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: newEntry
//
//  Description:
//
//      This member is called to add an entry to the list of entries.
//
//  Parameters:
//
//      pEntry (input/output)
//          The entry to add.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::newEntry( 
    RegistryEntry* pEntry
)
{
    //  Check the parameters.
    ASSERT( pEntry != NULL );

    //  If there is no tail then the list is empty and this becomes the
    //  first item in the list.
    if( m_pEntryTail == NULL )
    {
        //  Sanity check.
        ASSERT( m_pEntryHead == NULL );

        //  OK, put the entry at the front of the list.
        m_dwNumEntries += 1;
        m_pEntryHead    = pEntry;
        m_pEntryTail    = m_pEntryHead;
    }
    else
    {
        //  Sanity check.
        ASSERT( m_pEntryHead != NULL );

        //  Put the entry at the end of the list.
        m_dwNumEntries      += 1;
        m_pEntryTail->pNext  = pEntry;
        m_pEntryTail         = pEntry;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteEntry
//
//  Description:
//
//      This member is called to remove an entry from the list of entries and
//      free the space it occupied.
//
//  Parameters:
//
//      pEntry (input/output)
//          The entry to remove.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::deleteEntry( 
    RegistryEntry* pEntry
)
{
    //  Check the parameters.
    ASSERT( pEntry != NULL );

    //  The previous entry in the list.
    RegistryEntry* pPrevious;


    //  Handle the special case where the entry to delete is at the head
    //  of the list.
    if( pEntry == m_pEntryHead )
    {
        //  Move the head pointer to the next item.
        m_pEntryHead = pEntry->pNext;

        //  If the head is now NULL then the list is empty so we need to
        //  set the tail of the list.
        if( m_pEntryHead == NULL )
        {
            m_pEntryTail = NULL;
        }
    }
    else
    {
        //  Find the previous entry in the list.
        for(
            pPrevious = m_pEntryHead ;
            ( pPrevious != NULL ) && ( pPrevious->pNext != pEntry ) ;
            pPrevious = pPrevious->pNext
        )
        {
            //  All work is done in the loop.
        }

        //  A previous entry should have been found.
        ASSERT( pPrevious != NULL );

        //  Skip over the entry to be deleted.
        pPrevious->pNext = pEntry->pNext;

        //  If the entry being deleted is the tail then the tail becomes
        //  the previous item.
        if( pEntry == m_pEntryTail )
        {
            m_pEntryTail = pPrevious;
        }
    }

    //  We know have one less entry.
    m_dwNumEntries -= 1;

    //  Now free up the space associated with the entry.
    delete [] pEntry->pstrGroup;
    delete [] pEntry->pstrName;
    delete    pEntry;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addEntry
//
//  Description:
//
//      This member is called to add an entry to the list of entries and to
//      update the registry file accordingly.
//
//  Parameters:
//
//      pEntry (input/output)
//          The entry to add.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::addEntry( 
    RegistryEntry* pEntry
)
{
    //  Check the parameters.
    ASSERT( pEntry != NULL );

    //  The length of a string.
    Byte bLength;

    //  The start of the entry.
    DWord dwEntryStart;

    //  First of all let's find out how much space the entry will take in
    //  the registry file.
    DWord dwEntrySize = 
        sizeof( Byte )              +
        strlen( pEntry->pstrGroup ) +
        sizeof( Byte )              +
        strlen( pEntry->pstrName )  +
        sizeof( pEntry->dwSize );


    //  Assign the start of the new entry and subsequently, the offset for 
    //  the new entry value.  It will be placed at the end of the file after 
    //  the new entry.
    if( m_dwNumEntries == 0 )
    {
        dwEntryStart = sizeof( m_dwNumEntries );
    }
    else
    {
        dwEntryStart = m_pEntryTail->dwOffset + m_pEntryTail->dwSize;
    }
    pEntry->dwOffset = dwEntryStart + dwEntrySize;


    //  Add the new entry to the list.
    newEntry( pEntry );


    //  Seek to the beginning of the registry file.
    m_pAppFile->seek( 0 );

    //  Write the new number of entries.
    m_pAppFile->write( ( Byte* )&m_dwNumEntries, sizeof( m_dwNumEntries ) );
    ASSERT( m_pAppFile->count( ) == sizeof( m_dwNumEntries ) );

    
    //  Seek to just past the end of the registry file.
    m_pAppFile->seek( dwEntryStart );
     
    //  Write the length of the group.
    bLength = strlen( pEntry->pstrGroup ); 
    m_pAppFile->write( &bLength, sizeof( bLength ) );
    ASSERT( m_pAppFile->count( ) == sizeof( bLength ) );

    //  Write the group itself.
    m_pAppFile->write( pEntry->pstrGroup, bLength );
    ASSERT( m_pAppFile->count( ) == bLength );

    //  Write the length of the name.
    bLength = strlen( pEntry->pstrName ); 
    m_pAppFile->write( &bLength, sizeof( bLength ) );
    ASSERT( m_pAppFile->count( ) == sizeof( bLength ) );

    //  Write the name itself.
    m_pAppFile->write( pEntry->pstrName, bLength );
    ASSERT( m_pAppFile->count( ) == bLength );

    //  Write the size.
    m_pAppFile->write( ( Byte* )&( pEntry->dwSize ), sizeof( pEntry->dwSize ) );
    ASSERT( m_pAppFile->count( ) == sizeof( pEntry->dwSize ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: removeEntry
//
//  Description:
//
//      This member is called to remove an entry from the list of entries 
//      and to update the registry file accordingly.
//
//  Parameters:
//
//      pEntry (input/output)
//          The entry to remove.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Registry::removeEntry( 
    RegistryEntry* pEntry
)
{
    //  Check the parameters.
    ASSERT( pEntry != NULL );


    //  First of all let's find out how much space the entry we're deleting
    //  takes up.
    DWord dwEntrySize = 
        sizeof( Byte )              +
        strlen( pEntry->pstrGroup ) +
        sizeof( Byte )              +
        strlen( pEntry->pstrName )  +
        sizeof( pEntry->dwSize );

    //  The start of the block to delete and the size of the block to delete.
    DWord dwStartDelete = pEntry->dwOffset - dwEntrySize;
    DWord dwSizeDelete  = pEntry->dwSize + dwEntrySize;


    //  Remove the item from the list.
    deleteEntry( pEntry );
    

    //  Seek to the beginning of the registry file.
    m_pAppFile->seek( 0 );

    //  Write the new number of entries.
    m_pAppFile->write( ( Byte* )&m_dwNumEntries, sizeof( m_dwNumEntries ) );
    ASSERT( m_pAppFile->count( ) == sizeof( m_dwNumEntries ) );


    //  Delete the block in the file that contains the entry and the data.
    m_pAppFile->deleteBlock( dwStartDelete, dwSizeDelete );

    //  We now must update the offsets of any entries that fell after the
    //  entry deleted.
    for(
        RegistryEntry* pTraverse = m_pEntryHead ;
        pTraverse != NULL ;
        pTraverse = pTraverse->pNext
    )
    {
        //  If the entry fell after the deleted one then we need to update it.
        if( pTraverse->dwOffset > dwStartDelete )
        {
            //  Update the offset.
            pTraverse->dwOffset -= dwSizeDelete;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: findEntry
//
//  Description:
//
//      This member is called to find the specified entry in the list of 
//      entries.
//
//  Parameters:
//
//      group (input)
//          The value group.
//
//      name (input)
//          The value name.
//
//  Returns:
//
//      A pointer to the entry if found.
//      NULL if the entry is not found.
//
///////////////////////////////////////////////////////////////////////////////
Registry::RegistryEntry*
Registry::findEntry( 
    const KString& group,
    const KString& name
) const
{
    //  Look for the entry.
    for( 
        RegistryEntry* pTraverse = m_pEntryHead ; 
        pTraverse != NULL ; 
        pTraverse = pTraverse->pNext
    )
    {
        //  If the entry is found then return it. 
        if( 
            ( strcmp( pTraverse->pstrGroup, group.data( ) ) == 0 ) && 
            ( strcmp( pTraverse->pstrName, name.data( ) ) == 0 )
        )
        {
            return( pTraverse );
        }
    }

    //  Entry not found.
    return( NULL );
}
