///////////////////////////////////////////////////////////////////////////////
//
//  File:    repbase.cpp
//
//  Class:   Replay
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a common base for each object in the Replay 
//      application.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: RepBase
//
//  Description:
//
//      This is the main constructor for the base object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
RepBase::RepBase(
    const KString& iName
)
:
    m_iName( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~RepBase
//
//  Description:
//
//      This is the destructor for the Replay base object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
RepBase::~RepBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: confirm
//
//  Description:
//
//      This function is called to check a condition and possibly report a 
//      fatal error.
//
//  Parameters:
//
//      bTestThis (input)
//          The result of the condition.
//
//      pstrFormat (input)
//          The format of the error message to print.
//
//      ... (input)
//          The remainder of the arguments that modify the format.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
RepBase::confirm(
    const Byte bTestThis,
    char*      pstrFormat,
    ...
) const
{
    //  The variable argument list.
    va_list ap;

    
    ASSERT( pstrFormat != ( char* )NULL );

    //  If the condition is true then just return.
    if( bTestThis )
    {
        return;
    }

    //  The variable argument list begins after the format string.
    va_start( ap, pstrFormat );

    //  Print out the error.
    vfprintf( stdout, pstrFormat, ap );
    fprintf( stdout, "\n" );
    fflush( stdout );

    //  All done with the list.
    va_end( ap );

    //  Exit the application harshly.
    exit( 1 );
}



#ifndef NDEBUG
///////////////////////////////////////////////////////////////////////////////
//
//  Function: check
//
//  Description:
//
//      This function is called when a warning has been raised.  This is
//      similar to assert() without stopping execution.
//
//  Parameters:
//
//      bTestThis (input)
//          The result of the condition.
//
//      pstrFormat (input)
//          The format of the error message to print.
//
//      ... (input)
//          The remainder of the arguments that modify the format.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
RepBase::check(
    const Byte bTestThis,
    char*      pstrFormat,
    ...
) const
{
    //  The variable argument list.
    va_list ap;

    
    ASSERT( pstrFormat != ( char* )NULL );

    //  If the condition is true then just return.
    if( bTestThis )
    {
        return;
    }

    //  The variable argument list begins after the format string.
    va_start( ap, pstrFormat );

    //  Print out the error.
    fprintf( stdout, "Warning: " );
    vfprintf( stdout, pstrFormat, ap );
    fprintf( stdout, "\n" );
    fflush( stdout );

    //  All done with the list.
    va_end( ap );
}
#endif



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fatalError
//
//  Description:
//
//      This function is called to report a fatal error.
//
//  Parameters:
//
//      pstrFormat (input)
//          The format of the error message to print.
//
//      ... (input)
//          The remainder of the arguments that modify the format.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
RepBase::fatalError(
    char* pstrFormat,
    ...
) const
{
    //  The variable argument list.
    va_list ap;

    
    ASSERT( pstrFormat != ( char* )NULL );

    //  The variable argument list begins after the format string.
    va_start( ap, pstrFormat );

    //  Print out the error.
    vfprintf( stdout, pstrFormat, ap );
    fprintf( stdout, "\n" );

    //  All done with the list.
    va_end( ap );

    //  Exit the application harshly.
    exit( 1 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This function is called to ask the object to save itself.  By 
//      default, this function does nothing.  However, sub-classes overriding
//      this function should always call their direct base class before
//      performing their own saving.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the object's state to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
RepBase::save(
    AppFile* pSaveFile
)
{
    //  If the file was not opened then report an error.
    if( !pSaveFile->isOpen( ) )
    {
        return( TRUE );
    }

    //  Nothing to save so no errors.
    return( FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This function is called to ask the object to load itself.  By 
//      default, this function does nothing.  However, sub-classes overriding
//      this function should always call their direct base class before
//      performing their own loading.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the object's state from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
RepBase::load(
    AppFile* /* pLoadFile is not used */
)
{
    //  Nothing to load so no errors.
    return( FALSE );
}
