///////////////////////////////////////////////////////////////////////////////
//
//  File:    replay.cpp
//
//  Class:   Replay
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is the main Replay application class.  It is responsible
//      for driving the replay application.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h> 
#include <stdlib.h> 
#include <string.h> 

//  Application Headers.
#include "replay.h"
#include "clock.h"
#include "repfact.h"
#include "gamfact.h"
#include "canvas.h"
#include "screen.h"
#include "keyb.h"
#include "joy.h"
#include "mouse.h"
#include "sound.h"
#include "disk.h"
#include "network.h"
#include "game.h"
#include "gameinfo.h"
#include "bitmap.h"
#include "select.h"
#include "meddler.h"
#include "clip.h"
#include "ctable.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif




///////////////////////////////////////////////////////////////////////////////
//
//  Function: Replay
//
//  Description:
//
//      This is the main constructor for the Replay application object.  It
//      is protected because it is a singleton and therefore cannot be 
//      instantiated by anyone but itself.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Replay::Replay(
    const KString& iName
)
:
    RepBase          ( iName ),
    m_pReplayFactory ( &( ReplayFactory::s_instance( ) ) ),
    m_pCanvas        ( m_pReplayFactory->createCanvas( "Canvas" ) ),
    m_pKeyboard      ( m_pReplayFactory->createKeyboard( "Keybrd" ) ),
    m_pJoystick      ( m_pReplayFactory->createJoystick( "JoyStk" ) ),
    m_pMouse         ( m_pReplayFactory->createMouse( "Mouse" ) ),
    m_pSound         ( m_pReplayFactory->createSound( "Sound" ) ),
    m_pDisk          ( m_pReplayFactory->createDisk( "Disk" ) ),
    m_pNetwork       ( m_pReplayFactory->createNetwork( "Network" ) ),
    m_pClock         ( m_pReplayFactory->createClock( "Clock" ) ),
    m_pMeddler       ( NULL ),
    m_runGame        ( ),
    m_gameInfoList   ( 32 ),
    m_pGameFactory   ( m_pReplayFactory->createGameFactory( "GameFactory" ) ),
    m_pGame          ( NULL ),
    m_pDecColTable   ( new ColourTable( "Decoration", 128, 2 ) ),
    m_bDecColIndex   ( 1 ),
    m_bThrottle      ( TRUE ),
    m_bShowFPS       ( FALSE )
{
    //  Initialize the decoration colour table.
    for( DWord dwI = 0 ; dwI < 128 ; dwI += 1 )
    {
        ( *m_pDecColTable )[ dwI * 2 ]     = 0;
        ( *m_pDecColTable )[ dwI * 2 + 1 ] = dwI;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Replay
//
//  Description:
//
//      This is the main destructor for the Replay application object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Replay::~Replay(
)
{
    //  Get rid of the current game.
    clearGame( );
 
    //  Dispose of the replay objects.
    delete m_pGameFactory;
    delete m_pMeddler;
    delete m_pClock;
    delete m_pDisk;
    delete m_pNetwork;
    delete m_pSound;
    delete m_pMouse;
    delete m_pJoystick;
    delete m_pKeyboard;
    delete m_pCanvas;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Replay application object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Replay::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Replay" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_instance
//
//  Description:
//
//      This member returns the singleton Replay object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A reference to the singleton Replay object.
//
///////////////////////////////////////////////////////////////////////////////
Replay&
Replay::s_instance(
)
{
    //  The instance.
    static Replay replay( "Replay" );

    return( replay );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member function is called to run Replay.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A status code indicating the successful completion of Replay.
//
///////////////////////////////////////////////////////////////////////////////
int32
Replay::run(
)
{
    //  Check that the application has been compiled for the correct endianess.
    checkBitByteOrder( );

    //  Initialize canvas.
    initializeCanvas( );

    //  Build the game list.
    buildGameList( );

    //  Create the game selection screen.
    SelectScreen* pSelectScreen = 
        new SelectScreen( "Selection", m_gameInfoList );

    //  The state the select screen should be in.
    SelectScreen::State eSSState = SelectScreen::STATE_INITIALIZE;

    //  Loop until the user quits.
    while( TRUE )
    {
        //  Execute the selection screen and if a game hasn't been selected
        //  then break out.
        pSelectScreen->execute( eSSState );
        if( m_pGame == NULL )
        {
            break;
        }


        //  Keep calling the game startup until it indicates it's done.
        while( m_pGame->startUp( ) )
        {
            //  All of the work is done in the loop.
        }

       
        //  Create a page for the game.
        ASSERT( m_pGame->getGameInfo( ) != NULL );
        m_pCanvas->addPage( 
            m_pGame->getGameInfo( )->getGameName( ),
            Canvas::PAGE_ZERO,
            m_pGame->getScreen( )->getBitmap( )->getWidth( ), 
            m_pGame->getScreen( )->getBitmap( )->getHeight( )
        );
        m_pCanvas->turnPage( Canvas::PAGE_ZERO );

        //  Set the throttling for the game based on the global setting and
        //  make sure the FPS is set correctly.
        m_pClock->setThrottle( m_bThrottle );
        m_pClock->setFPS( m_pGame->getScreen( )->getFPS( ) );
 
        //  If there is a meddler, then activate it.
        if( m_pMeddler )
        {
            m_pMeddler->activate( );
        }

        //  Since we're going to use the screen a couple times within the
        //  loop we assign a convenience to get it outside and improve
        //  performance ever-so slightly.
        Bitmap* pScreen = m_pGame->getScreen( )->getBitmap( );

        //  Run the game.
        for( ; ; )
        {
            //  Pause the game?
            if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );
                break;
            } 
            //  Reset the game?
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F4 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F4 );
                m_pGame->reset( );
            }
            //  Change the decoration colour?
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F5 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F5 );
                m_bDecColIndex = ( m_bDecColIndex + 1 ) % 128;
            }
            //  Toggle speed throttling?
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F6 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F6 );
                m_bThrottle = m_bThrottle ? FALSE : TRUE;
                m_pClock->setThrottle( m_bThrottle );
            }
            //  Toggle FPS display.
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F7 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F7 );
                m_bShowFPS = m_bShowFPS ? FALSE : TRUE;
            }
#ifdef DEBUGGER
            //  Enter the debugger.
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F8 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F8 );
                Debugger::s_instance( ).interrupt( );
            }
#endif
            //  Flip Screen.
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F9 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F9 );
                m_pCanvas->flip( );
            }
            //  Rotate Screen.
            else
            if( m_pKeyboard->switchOn( Keyboard::KEY__F10 ) )
            {
                m_pKeyboard->waitUntilOff( Keyboard::KEY__F10 );
                m_pCanvas->rotate( );
            }
                

            //  Allow the system to update and let the meddler add to the
            //  update (if one exists).
            //
            //  Important Note:  The updates *must* be done immediately
            //      before running a frame of the game.  Do not place
            //      anything between the following two updates and the run( ).
            //      The reason is that meddlers depend on the state of the
            //      controls and if any calls are made between the meddling
            //      and the running of the game then it is possible that
            //      the state will change before the frame is run.
            update( );

            //  If something's meddling in our updates then let it.
            if( m_pMeddler )
            {
                //  If the meddler is finished then quit out of the game.
                if( m_pMeddler->complete( ) )
                {
                    delete m_pMeddler;
                    m_pMeddler = NULL;
                    break;
                }

                //  Update the meddler.
                m_pMeddler->update( );
            }
         
            //  Run a frame of the game.
            m_pGame->run( );

            //  Wait for the next frame.
            m_pClock->frameWait( );

            //  Draw the decorations.  This includes items like the current
            //  FPS.
            drawDecorations( pScreen );

            //  Draw a frame.
            m_pCanvas->draw(
                pScreen,
                0,
                0,
                pScreen->getWidth( ),
                pScreen->getHeight( )
            );
        }

        //  Remove the page and ensure all of the colours have been freed.
        m_pCanvas->removePage( Canvas::PAGE_ZERO );

        //  We don't want to shut the game down because it will be used in
        //  the selection screen as we loop back around.
        //  Also, the game should start off paused in the selection screen.
        eSSState = SelectScreen::STATE_PAUSE;

        //  Deactivate the meddler (if any) until we get back to the game.
        if( m_pMeddler )
        {
            m_pMeddler->deactivate( );
        }

        //  Go back to the default 60 FPS.
        m_pClock->setFPS( 60 );

        //  Stop the sound.
        m_pSound->stopAll( );
    }

    //  We're all finished with the selection screen.
    delete pSelectScreen;

    //  The application completed successfully.
    return( 0 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkBitByteOrder
//
//  Description:
//
//      This member function ensures that the application has been compiled
//      with the correct macros according to the endianess of the machine.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::checkBitByteOrder(
) const
{
    //  The following union is used to check the bit order.
    union
    {
        struct
        {
            uint8 bField1 : 1;
            uint8 bDummy  : 6;
            uint8 bField2 : 1;
        }
        tBits;
        uint8 bValue;
    }
    uBitTest;

    //  The following union is used to check the byte order.
    union
    {
        struct
        {
            uint8 bField1;
            uint8 bField2;
        }
        tBytes;
        uint16 wValue;
    }
    uByteTest;


    //  Check that the bit order macro is correct.
    uBitTest.bValue = 0;
#ifdef BIT_LS_FIRST
    uBitTest.tBits.bField1 = 1;
    if( uBitTest.bValue != 1 )
    {
        fatalError(
            "Replay should be compiled with BIT_MS_FIRST defined "
            "instead of BIT_LS_FIRST.\nContact the author."
        );
    }
#else
    uBitTest.tBits.bField2 = 1;
    if( uBitTest.bValue != 1 )
    {
        fatalError(
            "Replay should be compiled with BIT_LS_FIRST defined "
            "instead of BIT_MS_FIRST.\nContact the author."
        );
    }
#endif


    //  Check that the byte order macro is correct.
    uByteTest.wValue = 0;
#ifdef BIT_LS_FIRST
    uByteTest.tBytes.bField1 = 1;
    if( uByteTest.wValue != 1 )
    {
        fatalError(
            "Replay should be compiled with BYTE_MS_FIRST defined "
            "instead of BYTE_LS_FIRST.\nContact the author."
        );
    }
#else
    uByteTest.tBytes.bField2 = 1;
    if( uByteTest.wValue != 1 )
    {
        fatalError(
            "Replay should be compiled with BYTE_LS_FIRST defined "
            "instead of BYTE_MS_FIRST.\nContact the author."
        );
    }
#endif
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initializeCanvas
//
//  Description:
//
//      This member function is called to initialize the canvas so it's
//      ready for use.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::initializeCanvas(
)
{
    //  The number of colours supported in the canvas.
    DWord dwNumColours = m_pCanvas->getNumColours( );

    //  Set an initial palette.
    for( DWord dwI = 0 ; dwI < dwNumColours ; dwI += 1 )
    {
        m_pCanvas->setColour(
            dwI, rand( ) % 256, rand( ) % 256, rand( ) % 256
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: buildGameList
//
//  Description:
//
//      This member function creates the list of games that are available to
//      the user.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::buildGameList(
)
{
    //  Add the information for each game to the list.
    for( DWord dwI = 0 ; dwI < m_pGameFactory->numGames( ) ; dwI += 1 )
    {
        m_gameInfoList.add( m_pGameFactory->getGameInfo( dwI ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: drawDecorations
//
//  Description:
//
//      This member function is called to place any decorations on the
//      screen before drawing.  This includes items such as the current 
//      FPS.
//
//  Parameters:
//
//      pDest (input/output)
//          The destination bitmap to draw the decorations on.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::drawDecorations(
    Bitmap* pDest
)
{
    ASSERT( pDest != NULL );

    //  Should the current FPS be drawn.
    if( m_bShowFPS )
    {
        //  The following buffer will contain the FPS information.
        char strFPSInfo[16];

        //  The measured FPS.
        DWord dwMeasuredFPS = m_pClock->getMeasuredFPS( );

        //  Indicate the current FPS and the percentage of full speed the
        //  game is running at.
        sprintf( 
            strFPSInfo, 
            "%03ld - %03ld%%", 
            dwMeasuredFPS,
            dwMeasuredFPS * 100 / m_pClock->getFPS( )
        );

        //  Draw the information.
        m_pCanvas->drawText( 
            strFPSInfo,
            pDest,  
            Canvas::FONT_SMALL,
            pDest->getFullClipping( ),
            m_pDecColTable->getEntry( m_bDecColIndex )
        );
    }

    //  If there is a meddler then draw the decorations for it.
    if( m_pMeddler )
    {
        //  This should do it.
        Clipping clipMeddler( 0, 192, 8, 15 );

        //  Draw the information.
        m_pCanvas->drawText( 
            m_pMeddler->getDecoration( ),
            pDest,  
            Canvas::FONT_SMALL,
            &clipMeddler,
            m_pDecColTable->getEntry( m_bDecColIndex )
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This function should be called periodically during execution.  It
//      allows the system to perform any platform specific updating that
//      is required to keep things moving.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      TRUE  if something has changed that requires response from clients.
//      FALSE if nothing out of the ordinary has occurred.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Replay::update(
)
{
    //  Update the keyboard.
    m_pKeyboard->update( );

    //  Update the joystick.
    m_pJoystick->update( );

    //  Update the audio.
    m_pSound->update( );

    //  Update the canvas.
    return( m_pCanvas->update( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_compareGame
//
//  Description:
//
//      This member function is used to compare two games by name.
//
//  Parameters:
//
//      dwGame
//          The index of the game to create.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::setGame(
    const DWord dwIdx
)
{
    //  First get rid of any previous game.
    clearGame( );

    //  Now create the new game.
    m_pGame = m_pGameFactory->newGame( dwIdx );
    ASSERT( m_pGame != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearGame
//
//  Description:
//
//      This member function is used to remove the current game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Replay::clearGame(
)
{
    //  If there is a meddler then get rid of it.
    if( m_pMeddler != NULL )
    {
        deleteMeddler( );
    }

    //  Now dispose of the game.
    if( m_pGame != NULL )
    {
        m_pGameFactory->deleteGame( m_pGame );
        m_pGame = NULL;
    }
}

