///////////////////////////////////////////////////////////////////////////////
//
//  File:    sound.cpp
//
//  Class:   Sound
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Sound class is an abstract base class encapsulating sound
//      control.  A derived class for each platform should be created
//      from this base.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "sound.h"
#include "config.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////

//  The following represent the minimum, maximum and default values for
//  the sample rate and bits.
const DWord Sound::sm_dwSampleRateMin = 5000;
const DWord Sound::sm_dwSampleRateMax = 65535;
const DWord Sound::sm_dwSampleRateDef = 22050;
const DWord Sound::sm_dwSampleBitsDef = 8;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Sound
//
//  Description:
//
//      This is the main constructor for a sound object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Sound::Sound(
    const KString&  iName
)
:
    RepBase                  ( iName ),
    m_bEnabled               ( TRUE ),
    m_bAllowsOPL             ( TRUE ),
    m_bSupportsOPL           ( FALSE ),
    m_dwSampleRate           ( sm_dwSampleRateDef ),
    m_dwSampleBits           ( sm_dwSampleBitsDef ),
    m_dwNextAvailableChannel ( 0 )
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    //  An integer command line parameter.
    int32 nParam;


    //  Retrieve the sample rate from the command line.
    if( config.getParam( "-srate", &nParam ) )
    {
        CONFIRM( 
            ( ( DWord )nParam >= sm_dwSampleRateMin ) && 
            ( ( DWord )nParam <= sm_dwSampleRateMax ),
            "Sample rate must be between %d and %d.",
            sm_dwSampleRateMin,
            sm_dwSampleRateMax
        );

        m_dwSampleRate = ( DWord )nParam;
    }

    //  Retrieve the sample bits from the command line.
    if( config.getParam( "-sbits", &nParam ) )
    {
        CONFIRM( 
            ( nParam != 8 ) && ( nParam != 16 ),
            "Sample bits must be either 8 or 16."
        );

        m_dwSampleBits = ( DWord )nParam;
    }

    //  Is sound disabled?
    if( config.getParam( "-nosound" ) )
    {
        m_bEnabled = FALSE;
    }

    //  Is OPL allowed?
    if( config.getParam( "-noopl" ) )
    {
        m_bAllowsOPL = FALSE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Sound
//
//  Description:
//
//      This is the destructor for a sound object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Sound::~Sound(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getChannel
//
//  Description:
//
//      This member is called to reserve a voice channel.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The reserved channel.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Sound::getChannel(
)
{
    if( m_dwNextAvailableChannel < getNumChannels( ) )
    {
        m_dwNextAvailableChannel += 1;
    }
    return( m_dwNextAvailableChannel - 1 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: resetChannels
//
//  Description:
//
//      This member is called to reset the reserved channels.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Sound::resetChannels(
)
{
    m_dwNextAvailableChannel = 0;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: stopAll
//
//  Description:
//
//      This member is called to stop the sample playing in all channels.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Sound::stopAll(
)
{
    for( DWord dwI = 0 ; dwI < getNumChannels( ) ; dwI += 1 )
    {
        stop( dwI );
    }
}
