///////////////////////////////////////////////////////////////////////////////
//
//  File:    select.cpp
//
//  Class:   SelectScreen
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used by the Replay application to select a game from
//      the list of games available.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//  Application Headers.
#include "select.h"
#include "replay.h"
#include "canvas.h"
#include "keyb.h"
#include "clock.h"
#include "bitmap.h"
#include "gfxset.h"
#include "ctable.h"
#include "gameinfo.h"
#include "game.h"
#include "clip.h"
#include "sssbase.h"
#include "sssmenu.h"
#include "sssinit.h"
#include "sssdemo.h"
#include "ssspause.h"
#include "sssdip.h"
#include "sssserv.h"
#include "ssscli.h"
#include "ssskeyc.h"
#include "sssjoyc.h"
#include "sssmiss.h"
#include "sssmsg.h"

///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* SELECT_SCREEN_NAME  = "Select Screen";
static const char* BG_IMAGE            = "replay.pic";

//  The page attributes.
static const Canvas::PageNumber PAGE   = Canvas::PAGE_ZERO;
static const DWord PAGE_WIDTH          = 640;
static const DWord PAGE_HEIGHT         = 480;

//  The size of the overlays.
static const DWord LIST_WIDTH          = 204;
static const DWord LIST_HEIGHT         = 196;
static const DWord HELP_WIDTH          = 188;
static const DWord HELP_HEIGHT         = 50;
static const DWord DEMO_WIDTH          = 256;
static const DWord DEMO_HEIGHT         = 256;
static const DWord CREDITS_WIDTH       = 224;
static const DWord CREDITS_HEIGHT      = 58;

//  The position of the overlays.
static const int32 LIST_X              = 60;
static const int32 LIST_Y              = 144;
static const int32 HELP_X              = 65;
static const int32 HELP_Y              = 381;
static const int32 DEMO_X              = 348;
static const int32 DEMO_Y              = 65;
static const int32 CREDITS_X           = 365;
static const int32 CREDITS_Y           = 386;

//  The colours for the various pieces of the select picture.
static const Byte  COLOUR_START        = 250;
static const Byte  COLOUR_GREEN        = COLOUR_START + 0; 
static const Byte  COLOUR_BLUE         = COLOUR_START + 1;
static const Byte  COLOUR_RED          = COLOUR_START + 2;
static const Byte  COLOUR_WHITE        = COLOUR_START + 3;
static const Byte  COLOUR_GREY         = COLOUR_START + 4;
static const Byte  COLOUR_BLACK        = COLOUR_START + 5;
static const Byte  COLOUR_END          = COLOUR_BLACK;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SelectScreen
//
//  Description:
//
//      This is the main constructor for the game select screen.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      gameInfoList (input)
//          The list of games to select from.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SelectScreen::SelectScreen(
    const KString&            iName,
    const KPtrList<GameInfo>& gameInfoList
)
:
    RepBase              ( iName ),
    m_eState             ( STATE_INITIALIZE ),
    m_pState             ( NULL ),
    m_pStateMainMenu     ( NULL ),
    m_pStateInitialize   ( NULL ),
    m_pStateDemo         ( NULL ),
    m_pStatePause        ( NULL ),
    m_pStateKeyConfig    ( NULL ),
    m_pStateJoyConfig    ( NULL ),
    m_pStateDipSwitch    ( NULL ),
    m_pStateServer       ( NULL ),
    m_pStateClient       ( NULL ),
    m_pStateMissingFiles ( NULL ),
    m_pStateMessage      ( NULL ),
    m_gameInfoList       ( gameInfoList ),
    m_dwCurGame          ( 0 ),
    m_pCanvas            ( NULL ),
    m_pScreenBitmap      ( NULL ),
    m_pHelpBitmap        ( NULL ),
    m_pListBitmap        ( NULL ),
    m_pCreditsBitmap     ( NULL ),
    m_pFontSmall         ( NULL ),
    m_pFontLarge         ( NULL ),
    m_pHelpColTable      ( NULL ),
    m_pListColTable      ( NULL ),
    m_pListHiColTable    ( NULL ),
    m_pCreditsColTable   ( NULL ),
    m_pStateColTable     ( NULL ),
    m_pDefStateBitmap    ( NULL ),
    m_pStateBitmap       ( NULL ),
    m_nStateX            ( DEMO_X ),
    m_nStateY            ( DEMO_Y ),
    m_dwStateWidth       ( DEMO_WIDTH ),
    m_dwStateHeight      ( DEMO_HEIGHT ),
    m_bDrawBG            ( TRUE )
{
    //  Check the parameters.
    CONFIRM( getNumGames( ) > 0, "Game List is empty." );

    //  For convenience we keep a pointer to the canvas.
    m_pCanvas = Replay::s_instance( ).getCanvas( );

    //  Create that bitmaps that are used in rendering the select screen.
    m_pScreenBitmap   = m_pCanvas->createBitmap(
        SELECT_SCREEN_NAME, PAGE_WIDTH, PAGE_HEIGHT, TRUE 
    );
    m_pHelpBitmap     = m_pCanvas->createBitmap(
        "Help", HELP_WIDTH, HELP_HEIGHT 
    );
    m_pListBitmap     = m_pCanvas->createBitmap(
        "List", LIST_WIDTH, LIST_HEIGHT 
    );
    m_pCreditsBitmap  = m_pCanvas->createBitmap(
        "Credits", CREDITS_WIDTH, CREDITS_HEIGHT 
    );
    m_pDefStateBitmap = m_pCanvas->createBitmap(
        "State", DEMO_WIDTH, DEMO_HEIGHT, TRUE
    );

    //  We load up a purty background.
    m_pScreenBitmap->loadPCX( BG_IMAGE );

    //  Create the colour tables.
    m_pHelpColTable    = new ColourTable( "Help CTable", 2 );
    m_pListColTable    = new ColourTable( "List CTable", 2 );
    m_pListHiColTable  = new ColourTable( "List CTable", 2 );
    m_pCreditsColTable = new ColourTable( "Credits CTable", 2 );
    m_pStateColTable   = new ColourTable( "State CTable", 3 );

    //  We follow a state pattern to allow flexible extensibility of the
    //  selection screen.  Create the possible states here.
    m_pStateMainMenu     = new SSStateMainMenu( "Menu", this, m_pCanvas );
    m_pStateInitialize   = new SSStateInitialize( "Init", this, m_pCanvas );
    m_pStateDemo         = new SSStateDemo( "Demo", this, m_pCanvas );
    m_pStatePause        = new SSStatePause( "Pause", this, m_pCanvas );
    m_pStateKeyConfig    = new SSStateKeyConfig( "Keys", this, m_pCanvas );
    m_pStateJoyConfig    = new SSStateJoyConfig( "Joysick", this, m_pCanvas );
    m_pStateDipSwitch    = new SSStateDipSwitch( "Dips", this, m_pCanvas );
    m_pStateServer       = new SSStateServer( "Server", this, m_pCanvas );
    m_pStateClient       = new SSStateClient( "Client", this, m_pCanvas );
    m_pStateMissingFiles = new SSStateMissingFiles( "File", this, m_pCanvas );
    m_pStateMessage      = new SSStateMessage( "Message", this, m_pCanvas );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SelectScreen
//
//  Description:
//
//      This is the destructor for the game select screen.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SelectScreen::~SelectScreen(
)
{
    ASSERT( m_pStateMainMenu     != NULL );
    ASSERT( m_pStateInitialize   != NULL );
    ASSERT( m_pStateDemo         != NULL );
    ASSERT( m_pStatePause        != NULL );
    ASSERT( m_pStateKeyConfig    != NULL );
    ASSERT( m_pStateJoyConfig    != NULL );
    ASSERT( m_pStateDipSwitch    != NULL );
    ASSERT( m_pStateServer       != NULL );
    ASSERT( m_pStateClient       != NULL );
    ASSERT( m_pStateMissingFiles != NULL );
    ASSERT( m_pStateMessage      != NULL );

    //  Free the states.
    delete m_pStateMainMenu;
    delete m_pStateInitialize;
    delete m_pStateDemo;
    delete m_pStatePause;
    delete m_pStateKeyConfig;
    delete m_pStateJoyConfig;
    delete m_pStateDipSwitch;
    delete m_pStateServer;
    delete m_pStateClient;
    delete m_pStateMissingFiles;
    delete m_pStateMessage;

    //  Free the bitmaps.
    delete m_pScreenBitmap;
    delete m_pHelpBitmap;
    delete m_pListBitmap;
    delete m_pCreditsBitmap;
    delete m_pDefStateBitmap;

    //  Free the colour tables.
    delete m_pHelpColTable;
    delete m_pListColTable;
    delete m_pCreditsColTable;
    delete m_pStateColTable;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the game select screen.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SelectScreen::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SelectScreen" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setState
//
//  Description:
//
//      This member is called to set the current state of the selection
//      screen.
//
//  Parameters:
//
//      eState (input)
//          The state to change to.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
SelectScreen::setState ( 
    State eState 
)
{
    //  The old state.
    SSStateBase* pOldState = m_pState;

    //  Change to the desired state.
    switch( eState )
    {
        case STATE_PREVIOUS:      
        {
            //  If there is no previous state then there is nothing to do.
            if( m_pState->getPrevious( ) == NULL )
            {
                return;
            }
            m_pState = m_pState->getPrevious( );     
            break;
        }
        case STATE_MAIN_MENU:     m_pState = m_pStateMainMenu;     break;
        case STATE_INITIALIZE:    m_pState = m_pStateInitialize;   break;
        case STATE_DEMO:          m_pState = m_pStateDemo;         break;
        case STATE_PAUSE:         m_pState = m_pStatePause;        break;
        case STATE_KEY_CONFIG:    m_pState = m_pStateKeyConfig;    break;
        case STATE_JOY_CONFIG:    m_pState = m_pStateJoyConfig;    break;
        case STATE_DIP_SWITCH:    m_pState = m_pStateDipSwitch;    break;
        case STATE_SERVER:        m_pState = m_pStateServer;       break;
        case STATE_CLIENT:        m_pState = m_pStateClient;       break;
        case STATE_MISSING_FILES: m_pState = m_pStateMissingFiles; break;
        case STATE_MESSAGE:       m_pState = m_pStateMessage;      break;
        default:
        {
            fatalError( "Unknown selection screen state %d.", eState );
            break;
        }
    }

    //  Set the previous state (if we're not reverting to a previous state).
    if( eState != STATE_PREVIOUS )
    {
        m_pState->setPrevious( pOldState );
    }

    //  Record the state.
    m_eState = eState;

    //  Tell the state it has been set and initialize the state with the 
    //  current game information.
    m_pState->setGameInfo( m_gameInfoList[ m_dwCurGame ] );
    m_pState->setState( );

    //  Update out screen to reflect the new state.
    update( );

    //  Grab the bitmap associated with the state.
    m_pStateBitmap = m_pState->getScreen( );
    ASSERT( m_pStateBitmap != NULL );

    //  Calculate the positioning of the bitmap.
    if( m_pStateBitmap->getWidth( ) <= DEMO_WIDTH )
    {
        m_dwStateWidth = m_pStateBitmap->getWidth( );
        m_nStateX      = DEMO_X + ( DEMO_WIDTH - m_dwStateWidth ) / 2;
    }
    else
    {
        m_dwStateWidth = DEMO_WIDTH;
        m_nStateX      = DEMO_X;
    }
    if( m_pStateBitmap->getHeight( ) <= DEMO_HEIGHT )
    {
        m_dwStateHeight = m_pStateBitmap->getHeight( );
        m_nStateY       = DEMO_Y + ( DEMO_HEIGHT - m_dwStateHeight ) / 2;
    }
    else
    {
        m_dwStateHeight = DEMO_HEIGHT;
        m_nStateY       = DEMO_Y;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called to execute the game selection screen and allow
//      the user to make a choice.
//
//  Parameters:
//
//      eState (input)
//          The state to start in.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::execute(
    State eState
) 
{
    //  The result of a state execution.
    SSStateBase::Action eResult;

    //  Indicates that the execution is complete.
    Byte bDone;


    //  Disable key mappings while we're in the selection screen so that
    //  key presses don't go to the game.
    Replay::s_instance( ).getKeyboard( )->enableMappings( FALSE );

    //  For the selection screen we want throttling on.
    Replay::s_instance( ).getClock( )->setThrottle( TRUE );

    //  Create the page the selection screen uses.
    startUpPage( );

    //  Set the current state.
    setState( eState );

    //  We always should do an initial draw of the background.
    m_bDrawBG = TRUE;

    //  Loop until a game has been selected.
    for( bDone = FALSE ; bDone == FALSE ; )
    {
        //  Set the colours for the Game selection screen.
        //setColours( );

        //  If the background is to be drawn then draw it.
        if( m_bDrawBG )
        {
            m_pCanvas->draw( 
                m_pScreenBitmap, 
                0, 
                0, 
                m_pScreenBitmap->getWidth( ), 
                m_pScreenBitmap->getHeight( )
            );
            m_bDrawBG = FALSE;
        }

        //  Draw the state's screen.
        ASSERT( m_pStateBitmap != NULL );
        m_pCanvas->draw(
            m_pStateBitmap,
            m_nStateX,
            m_nStateY,
            m_dwStateWidth,
            m_dwStateHeight
        );


        //  Execute the current state.
        eResult = m_pState->execute( m_pStateColTable );
        switch( eResult )
        {
            case SSStateBase::SSS_CONTINUE:
            {
                break;
            }
            case SSStateBase::SSS_RUN:
            {
                if( getGame( ) == NULL )
                {
                    setGame( );
                }
                bDone = TRUE;
                break;
            }
            case SSStateBase::SSS_QUIT:
            {
                clearGame( );
                bDone = TRUE;
                break;
            }
        }
             
        //  Allow the system to update.
        if( Replay::s_instance( ).update( ) )
        {
            m_bDrawBG = TRUE;
        }
    }

    //  Shut down the page for the selection screen.
    shutDownPage( );

    //  Renable the key mappings.
    Replay::s_instance( ).getKeyboard( )->enableMappings( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setCurGame
//
//  Description:
//
//      This member is called to set the current game that is selected
//      in the selection screen.
//
//  Parameters:
//
//      dwGame (input)
//          The index of the game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::setCurGame(
    const DWord dwGame
)
{
    ASSERT( dwGame < getNumGames( ) );

    //  If the game hasn't changed then return.
    if( m_dwCurGame == dwGame )
    {
        return;
    }

    //  Assign the index of the new game.
    m_dwCurGame = dwGame;

    //  Clear the current game.
    if( getGame( ) != NULL )
    {
        getGame( )->shutDown( );
    }
    clearGame( );

    //  Inform the state of the game.
    m_pState->setGameInfo( m_gameInfoList[ m_dwCurGame ] );
}


   
///////////////////////////////////////////////////////////////////////////////
//
//  Function: setCurGame
//
//  Description:
//
//      This member is called to set the current game that is executing
//      in the selection screen.
//
//  Parameters:
//
//      gameId (input)
//          The identifier of the game to set to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::setCurGame(
    const KString& gameId
)
{
    //  Find the index of the game specified.
    for( DWord dwI = 0 ; dwI < getNumGames( ) ; dwI += 1 )
    {
        //  Is this the specified game?
        if( gameId == ( m_gameInfoList[ dwI ]->getGameId( ) ) )
        {
            setCurGame( dwI );
        }
    }
}
   


///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpPage
//
//  Description:
//
//      This member is called to create and initialize the page that the
//      selection screen uses.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::startUpPage(
) 
{
    //  Grab the fonts from the Canvas.
    m_pFontSmall = &( m_pCanvas->getFont( Canvas::FONT_SMALL ) );
    m_pFontLarge = &( m_pCanvas->getFont( Canvas::FONT_LARGE ) );
    ASSERT( m_pFontSmall != NULL );
    ASSERT( m_pFontLarge != NULL );

    //  Add a page for the selection screen and make sure it is active.
    m_pCanvas->addPage( SELECT_SCREEN_NAME, PAGE, PAGE_WIDTH, PAGE_HEIGHT );
    m_pCanvas->turnPage( PAGE );

    //  Set the colours for the Game selection screen.
    setColours( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setColours
//
//  Description:
//
//      This member sets the colours to appropriate values for the game
//      selection screen.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::setColours(
)
{
    //  The .PCX file we load should not have any more than dwNumColours 
    //  colours.  Also the dwNumColours colours must be at the upper end of 
    //  the palette since we are going to capture those and let the game 
    //  preview take over the rest of the palette.  A .PCX not conforming 
    //  is subject to weird colour changes.
    m_pCanvas->captureColour( COLOUR_GREEN,        000, 255, 000 );
    m_pCanvas->captureColour( COLOUR_BLUE,         000, 000, 255 );
    m_pCanvas->captureColour( COLOUR_RED,          255, 000, 000 );
    m_pCanvas->captureColour( COLOUR_WHITE,        255, 255, 255 );
    m_pCanvas->captureColour( COLOUR_GREY,         128, 128, 128 );
    m_pCanvas->captureColour( COLOUR_BLACK,        000, 000, 000 );

    //  Initialize the colour table for drawing text.
    ( *m_pListColTable )[ 0 ]    = COLOUR_WHITE;
    ( *m_pListColTable )[ 1 ]    = COLOUR_GREY;
    ( *m_pListHiColTable )[ 0 ]  = COLOUR_WHITE;
    ( *m_pListHiColTable )[ 1 ]  = COLOUR_RED;
    ( *m_pHelpColTable )[ 0 ]    = COLOUR_WHITE;
    ( *m_pHelpColTable )[ 1 ]    = COLOUR_RED;
    ( *m_pCreditsColTable )[ 0 ] = COLOUR_WHITE;
    ( *m_pCreditsColTable )[ 1 ] = COLOUR_RED;
    ( *m_pStateColTable )[ 0 ]   = COLOUR_BLACK;
    ( *m_pStateColTable )[ 1 ]   = COLOUR_WHITE;
    ( *m_pStateColTable )[ 2 ]   = COLOUR_RED;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: shutDownPage
//
//  Description:
//
//      This member is called to remove the page that the
//      selection screen uses.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::shutDownPage(
) 
{
    //  NULL out the fonts.
    m_pFontSmall = NULL;
    m_pFontLarge = NULL;

    //  Remove the page for the selection screen.
    m_pCanvas->removePage( PAGE );

    //  Free all of the colours so that the palette returns to the control
    //  of the game.
    m_pCanvas->freeColours( ); 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to udpate our screen bitmap to reflect the
//      current state of affairs.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
SelectScreen::update(
) 
{
    //  The height of the small font.
    int32 nSmallFontHeight;

    //  The clipping rectangle for various areas;
    Clipping listClipping;
    Clipping creditsClipping;


 
    //  Get the height of the small font.  (The first character in the font
    //  will do fine).
    nSmallFontHeight = ( int32 )( *m_pFontSmall )[ 0 ]->getHeight( );

    //  Clear the overlays.
    m_pHelpBitmap->clear( COLOUR_WHITE );
    m_pListBitmap->clear( COLOUR_WHITE );
    m_pCreditsBitmap->clear( COLOUR_WHITE );

    //  The current state is responsible for filling the help bitmap.
    m_pState->fillHelp( m_pHelpBitmap, Canvas::FONT_SMALL, m_pHelpColTable );

    //  We use an ever shrinking clipping rectangle for drawing the credits.
    creditsClipping.m_nMinX = 0;
    creditsClipping.m_nMaxX = m_pCreditsBitmap->getWidth( ) - 1;
    creditsClipping.m_nMinY = 0;
    creditsClipping.m_nMaxY = m_pCreditsBitmap->getHeight() - 1;
    
    //  Now draw the credits following the current game in the list.
    for( 
        DWord dwI = 0 ; 
        m_gameInfoList[ m_dwCurGame ]->getContributor( dwI ) != KStringNULL ;
        dwI += 1
    )
    { 
        //  Draw the contributor name into the credits.
        m_pCanvas->drawText( 
            m_gameInfoList[ m_dwCurGame ]->getContributor( dwI ),
            m_pCreditsBitmap,
            Canvas::FONT_SMALL,
            &creditsClipping,
            *m_pCreditsColTable
        );

        //  Update the clipping rectangle.
        creditsClipping.m_nMinY += nSmallFontHeight;
    }

    //  We use an ever shrinking clipping rectangle for drawing the names.
    //  once the rectangle is too small to draw a complete name in we are
    //  done drawing.
    listClipping.m_nMinX = 0;
    listClipping.m_nMaxX = m_pListBitmap->getWidth( ) - 1;
    listClipping.m_nMinY = 0;
    listClipping.m_nMaxY = m_pListBitmap->getHeight( ) - 1;
    
    //  Now draw the game names.
    int32 nNumRows = m_pListBitmap->getHeight( ) / nSmallFontHeight;
    for( ; ; )
    {
        //  Calculate the index of the game to display on this row.  The
        //  current game should approximately in the middle of the screen.
        int32 nRow       = listClipping.m_nMinY / nSmallFontHeight;
        int32 nGameIndex = ( int32 )( 
            m_dwCurGame - ( ( ( nNumRows - 1 ) / 3 ) - nRow ) 
        );

        //  If we are out of space then quit out of the loop.
        if( 
            nSmallFontHeight > 
            ( listClipping.m_nMaxY - listClipping.m_nMinY + 1)
        )
        {
            break;
        }

        //  If the index is valid then draw the item.
        if( nGameIndex >= 0 )
        {
            //  If we've run out of games then exit the loop.
            if( ( DWord )nGameIndex >= getNumGames( ) )
            {
                break;
            }

            //  Draw the game name into the list.
            m_pCanvas->drawText( 
                m_gameInfoList[ nGameIndex ]->getGameName( ),
                m_pListBitmap,
                Canvas::FONT_SMALL, 
                &listClipping,
                ( ( DWord )nGameIndex == m_dwCurGame ) ? 
                    *m_pListHiColTable : *m_pListColTable,
                TRUE
            );
        }

        //  Update the clipping rectangle.
        listClipping.m_nMinY += nSmallFontHeight;
    }

    //  Blast the overlays onto the screen bitmap.
    m_pScreenBitmap->blit(
        m_pHelpBitmap,
        HELP_X,
        HELP_Y,
        NULL,
        FALSE,
        FALSE,
        m_pScreenBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );
    m_pScreenBitmap->blit(
        m_pListBitmap,
        LIST_X,
        LIST_Y,
        NULL,
        FALSE,
        FALSE,
        m_pScreenBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );
    m_pScreenBitmap->blit(
        m_pCreditsBitmap,
        CREDITS_X,
        CREDITS_Y,
        NULL,
        FALSE,
        FALSE,
        m_pScreenBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );

    //  The background should be redrawn.
    m_bDrawBG = TRUE;
}
